import pandas as pd
import requests
import hashlib
import json
from datetime import datetime

class Employee:
    def __init__(self, id, name, basic_salary, allowances=0, mobile_number=None, bank_account=None):
        self.id = id
        self.name = name
        self.basic_salary = basic_salary
        self.allowances = allowances
        self.mobile_number = mobile_number  # For M-Pesa disbursements
        self.bank_account = bank_account    # For bank transfers
        self.deductions = {}  # e.g., {'PAYE': 0, 'NSSF': 0, ...}

    def calculate_gross_pay(self):
        return self.basic_salary + self.allowances

    def calculate_deductions(self):
        gross = self.calculate_gross_pay()
        
        # Kenyan deductions (2025 rates - approximate; verify with KRA/NSSF/NHIF)
        # NSSF: Tier 1 (6% up to 7,000 KES), Tier 2 (6% on 7,001-36,000 KES)
        nssf = min(0.06 * gross, 2160)  # Max 2,160 KES employee contribution
        
        # NHIF: Graduated rates (e.g., 150 KES for <6,000, up to 1,700 for >100,000)
        if gross < 6000:
            nhif = 150
        elif gross < 8000:
            nhif = 300
        # ... (add full brackets; truncated for brevity)
        else:
            nhif = 1700  # Max
        
        # PAYE: After relief (1,408 KES personal + 2,400 housing)
        taxable = gross - nssf - nhif - 2400 - 1408  # Deduct contributions & reliefs
        if taxable <= 0:
            paye = 0
        elif taxable <= 24000:
            paye = 0.1 * taxable
        elif taxable <= 32333:
            paye = 2400 + 0.25 * (taxable - 24000)
        else:
            paye = 2400 + 2083.25 + 0.3 * (taxable - 32333)
        
        self.deductions = {'NSSF': nssf, 'NHIF': nhif, 'PAYE': paye}
        return sum(self.deductions.values())

    def calculate_net_pay(self):
        return self.calculate_gross_pay() - self.calculate_deductions()

class PayrollSystem:
    def __init__(self, pesapal_api_key, pesapal_api_secret, openfloat_endpoint='https://api.pesapal.com/openfloat/disburse'):  # Hypothetical endpoint
        self.employees = []
        self.api_key = pesapal_api_key
        self.api_secret = pesapal_api_secret
        self.endpoint = openfloat_endpoint
        self.logs = []

    def add_employee(self, employee):
        self.employees.append(employee)

    def process_payroll(self):
        payroll_data = []
        for emp in self.employees:
            net_pay = emp.calculate_net_pay()
            payroll_data.append({
                'ID': emp.id,
                'Name': emp.name,
                'Gross Pay': emp.calculate_gross_pay(),
                'Deductions': emp.deductions,
                'Net Pay': net_pay,
                'Mobile': emp.mobile_number,
                'Bank': emp.bank_account
            })
            self.logs.append(f"{datetime.now()}: Processed {emp.name} - Net: {net_pay}")
        return pd.DataFrame(payroll_data)

    def generate_report(self, df):
        print("Payroll Summary:")
        print(df.to_string(index=False))
        df.to_csv('payroll_report.csv', index=False)
        print("\nAudit Log:")
        for log in self.logs:
            print(log)

    def disburse_payments(self, df, currency='KES'):
        # Hypothetical integration with Openfloat API (based on Pesapal's REST style)
        # In reality, replace with actual endpoint if provided by Pesapal.
        for _, row in df.iterrows():
            payload = {
                'amount': row['Net Pay'],
                'currency': currency,
                'recipient': row['Mobile'] or row['Bank'],
                'type': 'mobile' if row['Mobile'] else 'bank',
                'description': f"Payroll for {row['Name']} - ID: {row['ID']}",
                'reference': hashlib.sha256(str(row['ID']).encode()).hexdigest()  # Secure reference
            }
            headers = {
                'Authorization': f'Bearer {self.api_key}',  # Or use OAuth as per Pesapal docs
                'Content-Type': 'application/json'
            }
            try:
                response = requests.post(self.endpoint, json=payload, headers=headers)
                if response.status_code == 200:
                    self.logs.append(f"{datetime.now()}: Disbursed {row['Net Pay']} to {row['Name']}")
                else:
                    self.logs.append(f"{datetime.now()}: Error disbursing to {row['Name']}: {response.text}")
            except Exception as e:
                self.logs.append(f"{datetime.now()}: API Error for {row['Name']}: {str(e)}")

# Example Usage
if __name__ == "__main__":
    # Replace with real credentials (obtain from Pesapal/Openfloat)
    system = PayrollSystem(pesapal_api_key='5ZBt3vlfzRKk8P/s66z4jiZA2jvfHmm4', pesapal_api_secret='ppS55gdzF9P2zOSAHe0/0vlQDG8=')
    
    # Add employees
    system.add_employee(Employee(1, "John Doe", 50000, 10000, mobile_number="254712345678"))
    system.add_employee(Employee(2, "Jane Smith", 80000, 15000, bank_account="1234567890"))
    
    # Process and report
    payroll_df = system.process_payroll()
    system.generate_report(payroll_df)
    
    # Disburse (hypothetical - test in sandbox)
    system.disburse_payments(payroll_df)