<?php
// callback.php - Handle M-Pesa STK Push callback and update order status
include_once 'config.php';

// Log raw input and headers
$raw_input = file_get_contents('php://input');
$headers = getallheaders();
error_log("M-Pesa Callback Received: Raw Input = " . ($raw_input ?: 'EMPTY'));
error_log("M-Pesa Callback Headers: " . json_encode($headers));

// Check if input is empty
if (empty($raw_input)) {
    error_log("M-Pesa Callback: Empty input received");
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Empty callback data']);
    exit();
}

// Attempt to decode JSON
$data = json_decode($raw_input, true);
if (json_last_error() !== JSON_ERROR_NONE) {
    error_log("M-Pesa Callback JSON decode error: " . json_last_error_msg());
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Invalid JSON: ' . json_last_error_msg()]);
    exit();
}
error_log("M-Pesa Callback Parsed: " . json_encode($data));

// Validate callback data
if (!isset($data['Body']['stkCallback']['CheckoutRequestID'])) {
    error_log("Invalid callback data: Missing CheckoutRequestID");
    http_response_code(400);
    echo json_encode(['status' => 'error', 'message' => 'Missing CheckoutRequestID']);
    exit();
}

$checkout_request_id = $data['Body']['stkCallback']['CheckoutRequestID'];
$result_code = $data['Body']['stkCallback']['ResultCode'] ?? null;
$result_desc = $data['Body']['stkCallback']['ResultDesc'] ?? 'No description provided';
error_log("Processing callback: CheckoutRequestID = $checkout_request_id, ResultCode = $result_code, ResultDesc = $result_desc");

try {
    if (!isset($conn) || !$conn instanceof mysqli) {
        throw new Exception("Database connection not established.");
    }

    $status = $result_code === 0 ? 'completed' : 'failed';
    $stmt = $conn->prepare("UPDATE orders SET status = ?, mpesa_result_desc = ? WHERE mpesa_checkout_request_id = ?");
    $stmt->bind_param("sss", $status, $result_desc, $checkout_request_id);
    if (!$stmt->execute()) {
        throw new Exception("Failed to update order status: " . $conn->error);
    }
    error_log("Order updated: CheckoutRequestID = $checkout_request_id, Status = $status, Affected rows = " . $stmt->affected_rows);

    // Log additional callback details if available
    if (isset($data['Body']['stkCallback']['CallbackMetadata'])) {
        error_log("Callback Metadata: " . json_encode($data['Body']['stkCallback']['CallbackMetadata']));
    }

    http_response_code(200);
    echo json_encode(['status' => 'success', 'message' => 'Callback processed']);
} catch (Exception $e) {
    error_log("Callback processing error: " . $e->getMessage());
    http_response_code(500);
    echo json_encode(['status' => 'error', 'message' => 'Server error: ' . $e->getMessage()]);
    exit();
}
?>