<?php
session_start();
require_once 'config.php';
require_once 'database.php';

// Check if admin is logged in
if (!isset($_SESSION['admin'])) {
    header("Location: admin_login.php");
    exit();
}

// Handle transaction updates
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action'])) {
    $conn = getDB();
    if ($_POST['action'] === 'update_status') {
        $order_id = $_POST['order_id'];
        $status = $_POST['status'];
        $stmt = $conn->prepare("UPDATE transactions SET status = ? WHERE order_id = ?");
        $stmt->bind_param("ss", $status, $order_id);
        $stmt->execute();
        $stmt->close();
    } elseif ($_POST['action'] === 'delete') {
        $order_id = $_POST['order_id'];
        $stmt = $conn->prepare("DELETE FROM transactions WHERE order_id = ?");
        $stmt->bind_param("s", $order_id);
        $stmt->execute();
        $stmt->close();
    }
    $conn->close();
}

// Build filter conditions
$where = [];
$params = [];
$types = '';
if (!empty($_GET['method'])) {
    $where[] = "t.payment_method = ?";
    $params[] = $_GET['method'];
    $types .= 's';
}
if (!empty($_GET['status'])) {
    $where[] = "t.status = ?";
    $params[] = $_GET['status'];
    $types .= 's';
}
if (!empty($_GET['date_from'])) {
    $where[] = "t.created_at >= ?";
    $params[] = $_GET['date_from'];
    $types .= 's';
}
if (!empty($_GET['date_to'])) {
    $where[] = "t.created_at <= ?";
    $params[] = $_GET['date_to'];
    $types .= 's';
}
if (!empty($_GET['username'])) {
    $where[] = "u.username = ?";
    $params[] = $_GET['username'];
    $types .= 's';
}

// Fetch analytics with filters
$conn = getDB();
$where_sql = $where ? " WHERE " . implode(" AND ", $where) : "";
$sql_total_transactions = "SELECT COUNT(*) as count FROM transactions t JOIN users u ON t.user_id = u.id" . $where_sql;
$sql_total_amount = "SELECT SUM(amount) as total FROM transactions t JOIN users u ON t.user_id = u.id" . $where_sql;
$sql_method_breakdown = "SELECT t.payment_method, COUNT(*) as count, SUM(t.amount) as total FROM transactions t JOIN users u ON t.user_id = u.id" . $where_sql . " GROUP BY t.payment_method";

$stmt_total_transactions = $conn->prepare($sql_total_transactions);
$stmt_total_amount = $conn->prepare($sql_total_amount);
$stmt_method_breakdown = $conn->prepare($sql_method_breakdown);

if ($params) {
    $stmt_total_transactions->bind_param($types, ...$params);
    $stmt_total_amount->bind_param($types, ...$params);
    $stmt_method_breakdown->bind_param($types, ...$params);
}

$stmt_total_transactions->execute();
$total_transactions = $stmt_total_transactions->get_result()->fetch_assoc()['count'];
$stmt_total_transactions->close();

$stmt_total_amount->execute();
$total_amount = $stmt_total_amount->get_result()->fetch_assoc()['total'];
$stmt_total_amount->close();

$stmt_method_breakdown->execute();
$method_breakdown = $stmt_method_breakdown->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt_method_breakdown->close();

// Fetch transactions with filters
$sql_transactions = "SELECT t.*, u.username FROM transactions t JOIN users u ON t.user_id = u.id" . $where_sql . " ORDER BY t.created_at DESC";
$stmt_transactions = $conn->prepare($sql_transactions);
if ($params) {
    $stmt_transactions->bind_param($types, ...$params);
}
$stmt_transactions->execute();
$transactions = $stmt_transactions->get_result()->fetch_all(MYSQLI_ASSOC);
$stmt_transactions->close();

$conn->close();

// Build export URL with filter parameters
$export_url = "export_analytics.php?" . http_build_query($_GET);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Admin Dashboard - Pesapal Payment System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body { font-family: 'Inter', sans-serif; background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%); min-height: 100vh; }
        .admin-card { border: none; border-radius: 20px; box-shadow: 0 10px 30px rgba(0,0,0,0.1); }
        .admin-header { background: linear-gradient(135deg, #667eea 0%, #764ba2 100%); color: white; padding: 2rem; text-align: center; }
        .admin-body { padding: 2.5rem; }
        .nav-tabs { border-bottom: 2px solid #e9ecef; }
        .nav-link { color: #333; font-weight: 500; }
        .nav-link.active { background: #e3f2fd; border-color: #2196f3; }
        .stat-card { border-radius: 10px; padding: 1.5rem; background: #fff; box-shadow: 0 5px 15px rgba(0,0,0,0.08); }
        .btn-action { border-radius: 8px; }
        .btn-export { background: linear-gradient(135deg, #28a745 0%, #20c997 100%); color: white; }
        .btn-receipt { background: linear-gradient(135deg, #007bff 0%, #00d4ff 100%); color: white; }
        .btn-receipt:hover { transform: translateY(-2px); box-shadow: 0 5px 15px rgba(0, 123, 255, 0.4); }
        .table-responsive { border-radius: 10px; overflow: hidden; }
        .status-pending { color: #ffc107; }
        .status-completed { color: #28a745; }
        .status-failed { color: #dc3545; }
        .sticky-nav { position: sticky; top: 0; z-index: 1000; background: #fff; padding: 1rem; box-shadow: 0 2px 5px rgba(0,0,0,0.1); }
        @media (max-width: 768px) { .admin-body { padding: 1.5rem; } }
    </style>
</head>
<body>
    <div class="container py-4">
        <!-- Navigation -->
        <div class="sticky-nav">
            <div class="d-flex justify-content-between align-items-center">
                <h4><i class="fas fa-user-shield me-2"></i>Admin Dashboard</h4>
                <a href="admin_login.php?logout=1" class="btn btn-outline-danger btn-sm"><i class="fas fa-sign-out-alt me-2"></i>Logout</a>
            </div>
        </div>

        <!-- Admin Card -->
        <div class="row justify-content-center mt-4">
            <div class="col-md-12">
                <div class="admin-card">
                    <div class="admin-header">
                        <h2><i class="fas fa-cog me-2"></i>Manage Payments</h2>
                        <p class="mb-0 opacity-75">Control all aspects of the Pesapal payment system</p>
                    </div>
                    <div class="admin-body">
                        <!-- Tabs -->
                        <ul class="nav nav-tabs mb-4">
                            <li class="nav-item"><a class="nav-link active" href="#transactions" data-bs-toggle="tab">Transactions</a></li>
                            <li class="nav-item"><a class="nav-link" href="#analytics" data-bs-toggle="tab">Analytics</a></li>
                        </ul>
                        <div class="tab-content">
                            <!-- Transactions Tab -->
                            <div class="tab-pane fade show active" id="transactions">
                                <!-- Filters -->
                                <form method="GET" class="row g-3 mb-4">
                                    <div class="col-md-3">
                                        <select class="form-select" name="method">
                                            <option value="">All Methods</option>
                                            <option value="M-Pesa" <?php echo isset($_GET['method']) && $_GET['method'] == 'M-Pesa' ? 'selected' : ''; ?>>M-Pesa</option>
                                            <option value="Airtel Money" <?php echo isset($_GET['method']) && $_GET['method'] == 'Airtel Money' ? 'selected' : ''; ?>>Airtel Money</option>
                                            <option value="MTN MoMo" <?php echo isset($_GET['method']) && $_GET['method'] == 'MTN MoMo' ? 'selected' : ''; ?>>MTN MoMo</option>
                                            <option value="Card" <?php echo isset($_GET['method']) && $_GET['method'] == 'Card' ? 'selected' : ''; ?>>Card</option>
                                            <option value="PesaLink" <?php echo isset($_GET['method']) && $_GET['method'] == 'PesaLink' ? 'selected' : ''; ?>>PesaLink</option>
                                            <option value="Pesapal Wallet" <?php echo isset($_GET['method']) && $_GET['method'] == 'Pesapal Wallet' ? 'selected' : ''; ?>>Pesapal Wallet</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <select class="form-select" name="status">
                                            <option value="">All Statuses</option>
                                            <option value="PENDING" <?php echo isset($_GET['status']) && $_GET['status'] == 'PENDING' ? 'selected' : ''; ?>>Pending</option>
                                            <option value="COMPLETED" <?php echo isset($_GET['status']) && $_GET['status'] == 'COMPLETED' ? 'selected' : ''; ?>>Completed</option>
                                            <option value="FAILED" <?php echo isset($_GET['status']) && $_GET['status'] == 'FAILED' ? 'selected' : ''; ?>>Failed</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <input type="date" class="form-control" name="date_from" value="<?php echo isset($_GET['date_from']) ? htmlspecialchars($_GET['date_from']) : ''; ?>">
                                    </div>
                                    <div class="col-md-3">
                                        <input type="date" class="form-control" name="date_to" value="<?php echo isset($_GET['date_to']) ? htmlspecialchars($_GET['date_to']) : ''; ?>">
                                    </div>
                                    <div class="col-md-3">
                                        <input type="text" class="form-control" name="username" placeholder="Filter by Username" value="<?php echo isset($_GET['username']) ? htmlspecialchars($_GET['username']) : ''; ?>">
                                    </div>
                                    <div class="col-12">
                                        <button type="submit" class="btn btn-primary btn-action"><i class="fas fa-filter me-2"></i>Apply Filters</button>
                                        <a href="admin.php" class="btn btn-outline-secondary btn-action"><i class="fas fa-undo me-2"></i>Clear Filters</a>
                                        <a href="export_transactions.php" class="btn btn-export btn-action float-end"><i class="fas fa-download me-2"></i>Export Transactions to CSV</a>
                                    </div>
                                </form>

                                <!-- Transactions Table -->
                                <div class="table-responsive">
                                    <table class="table table-hover">
                                        <thead>
                                            <tr>
                                                <th>Order ID</th>
                                                <th>Username</th>
                                                <th>Amount (KES)</th>
                                                <th>Method</th>
                                                <th>Bank</th>
                                                <th>Email</th>
                                                <th>Phone</th>
                                                <th>Status</th>
                                                <th>Date</th>
                                                <th>Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($transactions as $txn): ?>
                                                <tr>
                                                    <td><?php echo htmlspecialchars($txn['order_id']); ?></td>
                                                    <td><?php echo htmlspecialchars($txn['username']); ?></td>
                                                    <td><?php echo number_format($txn['amount'], 2); ?></td>
                                                    <td><?php echo htmlspecialchars($txn['payment_method']); ?></td>
                                                    <td><?php echo htmlspecialchars($txn['bank'] ?: '-'); ?></td>
                                                    <td><?php echo htmlspecialchars($txn['email']); ?></td>
                                                    <td><?php echo htmlspecialchars($txn['phone']); ?></td>
                                                    <td>
                                                        <span class="badge bg-light text-dark <?php echo $txn['status'] === 'COMPLETED' ? 'status-completed' : ($txn['status'] === 'PENDING' ? 'status-pending' : 'status-failed'); ?>">
                                                            <?php echo htmlspecialchars($txn['status']); ?>
                                                        </span>
                                                    </td>
                                                    <td><?php echo date('M d, Y H:i', strtotime($txn['created_at'])); ?></td>
                                                    <td>
                                                        <form method="POST" style="display:inline;">
                                                            <input type="hidden" name="action" value="update_status">
                                                            <input type="hidden" name="order_id" value="<?php echo htmlspecialchars($txn['order_id']); ?>">
                                                            <select name="status" onchange="this.form.submit()" class="form-select form-select-sm d-inline w-auto">
                                                                <option value="PENDING" <?php echo $txn['status'] == 'PENDING' ? 'selected' : ''; ?>>Pending</option>
                                                                <option value="COMPLETED" <?php echo $txn['status'] == 'COMPLETED' ? 'selected' : ''; ?>>Completed</option>
                                                                <option value="FAILED" <?php echo $txn['status'] == 'FAILED' ? 'selected' : ''; ?>>Failed</option>
                                                            </select>
                                                        </form>
                                                        <form method="POST" style="display:inline;" onsubmit="return confirm('Are you sure you want to delete this transaction?');">
                                                            <input type="hidden" name="action" value="delete">
                                                            <input type="hidden" name="order_id" value="<?php echo htmlspecialchars($txn['order_id']); ?>">
                                                            <button type="submit" class="btn btn-danger btn-sm btn-action"><i class="fas fa-trash"></i></button>
                                                        </form>
                                                        <a href="generate_receipt.php?order_id=<?php echo urlencode($txn['order_id']); ?>" class="btn btn-receipt btn-sm"><i class="fas fa-file-pdf me-2"></i>Receipt</a>
                                                    </td>
                                                </tr>
                                            <?php endforeach; ?>
                                            <?php if (empty($transactions)): ?>
                                                <tr><td colspan="10" class="text-center">No transactions found.</td></tr>
                                            <?php endif; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>

                            <!-- Analytics Tab -->
                            <div class="tab-pane fade" id="analytics">
                                <!-- Analytics Filters -->
                                <form method="GET" class="row g-3 mb-4">
                                    <div class="col-md-3">
                                        <select class="form-select" name="method">
                                            <option value="">All Methods</option>
                                            <option value="M-Pesa" <?php echo isset($_GET['method']) && $_GET['method'] == 'M-Pesa' ? 'selected' : ''; ?>>M-Pesa</option>
                                            <option value="Airtel Money" <?php echo isset($_GET['method']) && $_GET['method'] == 'Airtel Money' ? 'selected' : ''; ?>>Airtel Money</option>
                                            <option value="MTN MoMo" <?php echo isset($_GET['method']) && $_GET['method'] == 'MTN MoMo' ? 'selected' : ''; ?>>MTN MoMo</option>
                                            <option value="Card" <?php echo isset($_GET['method']) && $_GET['method'] == 'Card' ? 'selected' : ''; ?>>Card</option>
                                            <option value="PesaLink" <?php echo isset($_GET['method']) && $_GET['method'] == 'PesaLink' ? 'selected' : ''; ?>>PesaLink</option>
                                            <option value="Pesapal Wallet" <?php echo isset($_GET['method']) && $_GET['method'] == 'Pesapal Wallet' ? 'selected' : ''; ?>>Pesapal Wallet</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <select class="form-select" name="status">
                                            <option value="">All Statuses</option>
                                            <option value="PENDING" <?php echo isset($_GET['status']) && $_GET['status'] == 'PENDING' ? 'selected' : ''; ?>>Pending</option>
                                            <option value="COMPLETED" <?php echo isset($_GET['status']) && $_GET['status'] == 'COMPLETED' ? 'selected' : ''; ?>>Completed</option>
                                            <option value="FAILED" <?php echo isset($_GET['status']) && $_GET['status'] == 'FAILED' ? 'selected' : ''; ?>>Failed</option>
                                        </select>
                                    </div>
                                    <div class="col-md-3">
                                        <input type="date" class="form-control" name="date_from" value="<?php echo isset($_GET['date_from']) ? htmlspecialchars($_GET['date_from']) : ''; ?>">
                                    </div>
                                    <div class="col-md-3">
                                        <input type="date" class="form-control" name="date_to" value="<?php echo isset($_GET['date_to']) ? htmlspecialchars($_GET['date_to']) : ''; ?>">
                                    </div>
                                    <div class="col-md-3">
                                        <input type="text" class="form-control" name="username" placeholder="Filter by Username" value="<?php echo isset($_GET['username']) ? htmlspecialchars($_GET['username']) : ''; ?>">
                                    </div>
                                    <div class="col-12">
                                        <button type="submit" class="btn btn-primary btn-action"><i class="fas fa-filter me-2"></i>Apply Filters</button>
                                        <a href="admin.php" class="btn btn-outline-secondary btn-action"><i class="fas fa-undo me-2"></i>Clear Filters</a>
                                        <a href="<?php echo htmlspecialchars("export_analytics.php?" . http_build_query($_GET)); ?>" class="btn btn-export btn-action float-end"><i class="fas fa-download me-2"></i>Export Analytics to CSV</a>
                                    </div>
                                </form>

                                <!-- Analytics Data -->
                                <div class="row g-4">
                                    <div class="col-md-4">
                                        <div class="stat-card">
                                            <h5><i class="fas fa-list me-2"></i>Total Transactions</h5>
                                            <p class="display-6"><?php echo number_format($total_transactions); ?></p>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="stat-card">
                                            <h5><i class="fas fa-money-bill-wave me-2"></i>Total Amount (KES)</h5>
                                            <p class="display-6"><?php echo number_format($total_amount, 2); ?></p>
                                        </div>
                                    </div>
                                    <div class="col-md-4">
                                        <div class="stat-card">
                                            <h5><i class="fas fa-chart-pie me-2"></i>Method Breakdown</h5>
                                            <ul class="list-group list-group-flush">
                                                <?php foreach ($method_breakdown as $method): ?>
                                                    <li class="list-group-item"><?php echo htmlspecialchars($method['payment_method']); ?>: <?php echo $method['count']; ?> (KES <?php echo number_format($method['total'], 2); ?>)</li>
                                                <?php endforeach; ?>
                                                <?php if (empty($method_breakdown)): ?>
                                                    <li class="list-group-item">No data available for selected filters.</li>
                                                <?php endif; ?>
                                            </ul>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <!-- Footer -->
        <div class="text-center mt-4">
            <small class="text-muted"><i class="fas fa-lock me-1"></i>Secured by Pesapal | Compliant with CBK Regulations</small>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>