<?php
session_start();
require_once 'vendor/autoload.php';
require_once 'config.php';
require_once 'database.php';

use Dompdf\Dompdf;

if (!isset($_GET['order_id'])) {
    $_SESSION['error'] = "Invalid transaction ID.";
    header("Location: user_dashboard.php");
    exit();
}

$order_id = filter_input(INPUT_GET, 'order_id', FILTER_SANITIZE_STRING);

// Fetch transaction and user details
$conn = getDB();
$stmt = $conn->prepare("SELECT t.*, u.username, u.email AS user_email, u.phone AS user_phone FROM transactions t LEFT JOIN users u ON t.user_id = u.id WHERE t.order_id = ?");
$stmt->bind_param("s", $order_id);
if (!$stmt->execute()) {
    error_log("Database Error: Failed to fetch transaction for order_id $order_id: " . $stmt->error, 3, 'C:\xampp\htdocs\payment_system\error.log');
    $_SESSION['error'] = "Failed to fetch transaction details.";
    header("Location: user_dashboard.php");
    exit();
}
$transaction = $stmt->get_result()->fetch_assoc();
$stmt->close();
$conn->close();

if (!$transaction) {
    error_log("Error: Transaction not found for order_id $order_id", 3, 'C:\xampp\htdocs\payment_system\error.log');
    $_SESSION['error'] = "Transaction not found.";
    header("Location: user_dashboard.php");
    exit();
}

// Handle user details
$username = $transaction['user_id'] == 0 ? 'Guest' : ($transaction['username'] ?? 'Unknown User');
$email = $transaction['user_id'] == 0 ? $transaction['email'] : ($transaction['user_email'] ?? $transaction['email']);
$phone = $transaction['user_id'] == 0 ? $transaction['phone'] : ($transaction['user_phone'] ?? $transaction['phone']);
$payment_method = $transaction['payment_method'] === 'PesaLink' ? 'Bank Transfer via PesaLink' : $transaction['payment_method'];
$transaction_ref = $transaction['id'] . '-' . time(); // Simulated transaction reference

// Generate professional bank-style PDF receipt
$dompdf = new Dompdf();
$html = '
<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Official Payment Receipt - ' . htmlspecialchars($order_id) . '</title>
    <style>
        body { font-family: "Helvetica", sans-serif; font-size: 12px; color: #333; margin: 0; padding: 0; }
        .container { width: 560px; margin: 20px auto; padding: 15px; border: 1px solid #ccc; background: #fff; box-sizing: border-box; }
        .header { text-align: center; padding-bottom: 10px; border-bottom: 2px solid #003087; }
        .header img { max-width: 120px; height: auto; }
        .header h1 { margin: 10px 0; font-size: 18px; color: #003087; }
        .header p { margin: 4px 0; font-size: 10px; color: #555; }
        .details { margin: 15px 0; }
        .details table { width: 100%; border-collapse: collapse; table-layout: fixed; }
        .details th, .details td { padding: 8px; border: 1px solid #ddd; text-align: left; vertical-align: top; font-size: 10px; word-wrap: break-word; overflow-wrap: break-word; }
        .details th { background: #003087; color: white; font-weight: bold; }
        .details .col-user, .details .col-payment { width: 50%; max-width: 50%; }
        .details td { line-height: 1.4; }
        .barcode { text-align: center; margin: 15px 0; font-family: "Courier New", monospace; font-size: 10px; border: 1px solid #ccc; padding: 5px; word-wrap: break-word; }
        .terms { margin: 15px 0; font-size: 9px; color: #666; page-break-inside: avoid; }
        .terms p { margin: 4px 0; }
        .footer { text-align: center; margin-top: 15px; font-size: 9px; color: #777; border-top: 1px solid #ccc; padding-top: 8px; page-break-inside: avoid; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Official Payment Receipt</h1>
            <p>Fine Mark Tracking Limited, P.O. Box 1234-00100, Nairobi, Kenya</p>
            <p>Email: harronmugo@gmail.com | Phone: +254 758 788 248 | Website: www.payment system.com</p>
            <p>Transaction Reference: ' . htmlspecialchars($transaction_ref) . '</p>
        </div>
        <div class="details">
            <table>
                <tr>
                    <th class="col-user">User Details</th>
                    <th class="col-payment">Payment Details</th>
                </tr>
                <tr>
                    <td class="col-user">
                        <strong>Username:</strong> ' . htmlspecialchars($username) . '<br>
                        <strong>Email:</strong> ' . htmlspecialchars($email) . '<br>
                        <strong>Phone:</strong> ' . htmlspecialchars($phone) . '
                    </td>
                    <td class="col-payment">
                        <strong>Order ID:</strong> ' . htmlspecialchars($order_id) . '<br>
                        <strong>Amount:</strong> KES ' . number_format($transaction['amount'], 2) . '<br>
                        <strong>Payment Method:</strong> ' . htmlspecialchars($payment_method) . '<br>
                        <strong>Bank:</strong> ' . htmlspecialchars($transaction['bank'] ?: 'N/A') . '<br>
                        <strong>Status:</strong> ' . htmlspecialchars($transaction['status']) . '<br>
                        <strong>Date:</strong> ' . date('M d, Y H:i', strtotime($transaction['created_at'])) . '
                    </td>
                </tr>
            </table>
        </div>
        <div class="barcode">
            <p>Transaction Barcode: **** ' . htmlspecialchars($transaction_ref) . ' ****</p>
        </div>
        <div class="terms">
            <p><strong>Terms and Conditions:</strong></p>
            <p>1. This receipt is issued by Pesapal Limited and is valid for all transactions processed through our platform.</p>
            <p>2. All payments are subject to verification and approval by the respective financial institutions.</p>
            <p>3. For disputes or inquiries, contact harronmugo@gmail.com within 30 days of the transaction date.</p>
            <p>4. This receipt is not a substitute for official bank statements. Please verify with your bank for final confirmation.</p>
        </div>
        <div class="footer">
            <p>Secured by Pesapal | Compliant with Central Bank of Kenya Regulations</p>
            <p>Generated on ' . date('M d, Y H:i') . '</p>
            <p>Thank you for using Pesapal Payment System</p>
        </div>
    </div>
</body>
</html>';
$dompdf->loadHtml($html);
$dompdf->setPaper('A4', 'portrait');
$dompdf->render();
$pdf_content = $dompdf->output();

// Output PDF for download
header('Content-Type: application/pdf');
header('Content-Disposition: attachment; filename="receipt_' . $order_id . '.pdf"');
header('Content-Length: ' . strlen($pdf_content));
echo $pdf_content;
exit();
?>