<?php
require_once 'vendor/autoload.php';
require_once 'config.php';
require_once 'database.php';

use PHPMailer\PHPMailer\PHPMailer;
use PHPMailer\PHPMailer\Exception;
use Dompdf\Dompdf;

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    error_log("Invalid IPN Request Method at " . date('Y-m-d H:i:s') . ": " . $_SERVER['REQUEST_METHOD'], 3, ERROR_LOG_PATH);
    http_response_code(405);
    exit(json_encode(['status' => 'error', 'message' => 'Method Not Allowed']));
}

// Log raw request
$raw_input = file_get_contents('php://input');
error_log("IPN Received at " . date('Y-m-d H:i:s') . ": $raw_input", 3, ERROR_LOG_PATH);

// Parse IPN data
$callback_data = json_decode($raw_input, true);
if (!$callback_data || !isset($callback_data['order_tracking_id'])) {
    error_log("IPN Error: Invalid or missing order_tracking_id, Raw: $raw_input", 3, ERROR_LOG_PATH);
    http_response_code(400);
    exit(json_encode(['status' => 'error', 'message' => 'Invalid IPN data']));
}

$order_tracking_id = $callback_data['order_tracking_id'];
error_log("Processing IPN for order_tracking_id: $order_tracking_id", 3, ERROR_LOG_PATH);

// Retry status check up to 3 times
$max_attempts = 3;
$attempt = 1;

while ($attempt <= $max_attempts) {
    try {
        // Get transaction status
        $status_url = PESAPAL_ENDPOINT . '/api/Transactions/GetTransactionStatus?orderTrackingId=' . urlencode($order_tracking_id);
        if (!filter_var($status_url, FILTER_VALIDATE_URL)) {
            throw new Exception("Invalid status URL: $status_url");
        }
        $ch = curl_init($status_url);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Authorization: Bearer ' . getPesapalToken(),
            'Content-Type: application/json'
        ]);
        curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, PESAPAL_IS_LIVE ? true : false);
        curl_setopt($ch, CURLOPT_TIMEOUT, 30);
        $response = curl_exec($ch);
        $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        $final_url = curl_getinfo($ch, CURLINFO_EFFECTIVE_URL);
        if (curl_errno($ch)) {
            throw new Exception("Status cURL error: " . curl_error($ch));
        }
        curl_close($ch);

        $status_response = json_decode($response, true);
        error_log("Status Response for order_tracking_id $order_tracking_id (Attempt $attempt): HTTP $http_code (Final URL: $final_url), Response: " . json_encode($status_response), 3, ERROR_LOG_PATH);

        if ($http_code !== 200 || !isset($status_response['status'])) {
            throw new Exception("Status query failed: HTTP $http_code, " . ($status_response['error'] ?? json_encode($status_response)));
        }

        // Map Pesapal status
        $status = $status_response['status'] === 'COMPLETED' ? 'COMPLETED' : ($status_response['status'] === 'FAILED' ? 'FAILED' : 'PENDING');
        $order_id = $status_response['merchant_reference'] ?? $order_tracking_id;

        // Update transactions and payrolls
        $conn = getDB();
        $stmt = $conn->prepare("UPDATE transactions SET status = ? WHERE order_tracking_id = ?");
        $stmt->bind_param("ss", $status, $order_tracking_id);
        $stmt->execute();
        $affected_rows = $stmt->affected_rows;
        error_log("Transaction status updated for order_tracking_id $order_tracking_id to $status, Affected Rows: $affected_rows", 3, ERROR_LOG_PATH);
        $stmt->close();

        // Update payrolls (for payroll disbursements)
        $stmt = $conn->prepare("UPDATE payrolls SET status = ? WHERE order_tracking_id = ?");
        $stmt->bind_param("ss", $status, $order_tracking_id);
        $stmt->execute();
        $affected_rows = $stmt->affected_rows;
        error_log("Payroll status updated for order_tracking_id $order_tracking_id to $status, Affected Rows: $affected_rows", 3, ERROR_LOG_PATH);
        $stmt->close();

        // Send notifications for COMPLETED
        if ($status === 'COMPLETED') {
            $stmt = $conn->prepare("SELECT t.*, u.username, u.email AS user_email, u.phone AS user_phone FROM transactions t LEFT JOIN users u ON t.user_id = u.id WHERE t.order_tracking_id = ?");
            $stmt->bind_param("s", $order_tracking_id);
            $stmt->execute();
            $transaction = $stmt->get_result()->fetch_assoc();
            $stmt->close();
            $conn->close();

            if ($transaction) {
                $username = $transaction['user_id'] == 0 ? 'Guest' : ($transaction['username'] ?? 'Unknown User');
                $email = $transaction['user_id'] == 0 ? $transaction['email'] : ($transaction['user_email'] ?? $transaction['email']);
                $phone = $transaction['user_id'] == 0 ? $transaction['phone'] : ($transaction['user_phone'] ?? $transaction['phone']);
                $payment_method = $transaction['payment_method'] === 'Bank' ? 'Bank Transfer via PesaLink' : $transaction['payment_method'];

                $message = "Transaction {$order_id} completed. Amount: KES " . number_format($transaction['amount'], 2) . ". Method: {$payment_method}. Status: {$status}. Date: " . date('M d, Y H:i', strtotime($transaction['created_at']));

                // SMS via Telkosh
                try {
                    $ch = curl_init('https://api.telkosh.com/v1/sms/send');
                    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
                    curl_setopt($ch, CURLOPT_POST, true);
                    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode([
                        'api_key' => TELKOSH_API_KEY,
                        'sender_id' => TELKOSH_SENDER_ID,
                        'to' => $phone,
                        'message' => $message
                    ]));
                    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
                    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, false);
                    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
                    $sms_response = curl_exec($ch);
                    curl_close($ch);
                    error_log("SMS sent for order_id $order_id to $phone: $sms_response", 3, ERROR_LOG_PATH);
                } catch (Exception $e) {
                    error_log("Telkosh SMS Error for order_id $order_id: " . $e->getMessage(), 3, ERROR_LOG_PATH);
                }

                // Email with PDF receipt
                $dompdf = new Dompdf();
                $html = '<!DOCTYPE html>
<html>
<head>
    <meta charset="UTF-8">
    <title>Official Payment Receipt - ' . htmlspecialchars($order_id) . '</title>
    <style>
        body { font-family: "Helvetica", sans-serif; font-size: 12px; color: #333; margin: 0; padding: 0; }
        .container { width: 560px; margin: 20px auto; padding: 15px; border: 1px solid #ccc; background: #fff; box-sizing: border-box; }
        .header { text-align: center; padding-bottom: 10px; border-bottom: 2px solid #003087; }
        .header h1 { margin: 10px 0; font-size: 18px; color: #003087; }
        .header p { margin: 4px 0; font-size: 10px; color: #555; }
        .details { margin: 15px 0; }
        .details table { width: 100%; border-collapse: collapse; }
        .details th, .details td { padding: 8px; border: 1px solid #ddd; text-align: left; }
        .details th { background: #003087; color: white; }
        .footer { text-align: center; margin-top: 15px; font-size: 9px; color: #777; border-top: 1px solid #ccc; padding-top: 8px; }
    </style>
</head>
<body>
    <div class="container">
        <div class="header">
            <h1>Official Payment Receipt</h1>
            <p>Payroll System, P.O. Box 1234-00100, Nairobi, Kenya</p>
            <p>Email: support@payroll.com | Phone: +254 709 123 456</p>
            <p>Transaction Reference: ' . htmlspecialchars($order_id) . '</p>
        </div>
        <div class="details">
            <table>
                <tr>
                    <th>User Details</th>
                    <th>Payment Details</th>
                </tr>
                <tr>
                    <td>
                        <strong>Username:</strong> ' . htmlspecialchars($username) . '<br>
                        <strong>Email:</strong> ' . htmlspecialchars($email) . '<br>
                        <strong>Phone:</strong> ' . htmlspecialchars($phone) . '
                    </td>
                    <td>
                        <strong>Order ID:</strong> ' . htmlspecialchars($order_id) . '<br>
                        <strong>Amount:</strong> KES ' . number_format($transaction['amount'], 2) . '<br>
                        <strong>Payment Method:</strong> ' . htmlspecialchars($payment_method) . '<br>
                        <strong>Status:</strong> ' . htmlspecialchars($status) . '<br>
                        <strong>Date:</strong> ' . date('M d, Y H:i', strtotime($transaction['created_at'])) . '
                    </td>
                </tr>
            </table>
        </div>
        <div class="footer">
            <p>Secured by Pesapal | Compliant with Central Bank of Kenya Regulations</p>
            <p>Generated on ' . date('M d, Y H:i') . '</p>
        </div>
    </div>
</body>
</html>';
                $dompdf->loadHtml($html);
                $dompdf->setPaper('A4', 'portrait');
                $dompdf->render();
                $pdf_file = sys_get_temp_dir() . "/receipt_{$order_id}.pdf";
                file_put_contents($pdf_file, $dompdf->output());

                try {
                    $mail = new PHPMailer(true);
                    $mail->isSMTP();
                    $mail->Host = SMTP_HOST;
                    $mail->SMTPAuth = true;
                    $mail->Username = SMTP_USERNAME;
                    $mail->Password = SMTP_PASSWORD;
                    $mail->SMTPSecure = SMTP_SECURE;
                    $mail->Port = SMTP_PORT;
                    $mail->setFrom(EMAIL_FROM, EMAIL_FROM_NAME);
                    $mail->addAddress($email);
                    $mail->addAttachment($pdf_file, "receipt_{$order_id}.pdf");
                    $mail->isHTML(true);
                    $mail->Subject = "Official Payment Receipt for {$order_id}";
                    $mail->Body = "Dear " . htmlspecialchars($username) . ",<br>Your payment has been processed. Receipt attached.<br><br>Payroll System";
                    $mail->send();
                    unlink($pdf_file);
                    error_log("Email sent for order_id $order_id to $email", 3, ERROR_LOG_PATH);
                } catch (Exception $e) {
                    error_log("PHPMailer Error for order_id $order_id: " . $e->getMessage(), 3, ERROR_LOG_PATH);
                }
            }
        }

        http_response_code(200);
        echo json_encode(['status' => 'success', 'message' => 'IPN processed']);
        exit();
    } catch (Exception $e) {
        error_log("IPN Processing Error for order_tracking_id $order_tracking_id (Attempt $attempt): " . $e->getMessage(), 3, ERROR_LOG_PATH);
        if ($attempt < $max_attempts) {
            sleep(5);
            $attempt++;
            continue;
        }
        http_response_code(500);
        echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
        exit();
    }
}
?>