<?php
session_start();
require_once 'config.php';
require_once 'database.php';

if (!isset($_GET['order_id'])) {
    error_log("Missing order_id in post_payment.php at " . date('Y-m-d H:i:s'), 3, ERROR_LOG_PATH);
    header("Location: index.php?error=invalid_request");
    exit();
}

$order_id = $_GET['order_id'];
$conn = getDB();
$stmt = $conn->prepare("SELECT * FROM transactions WHERE order_id = ?");
$stmt->bind_param("s", $order_id);
$stmt->execute();
$transaction = $stmt->get_result()->fetch_assoc();
$stmt->close();
$conn->close();

if (!$transaction) {
    error_log("Transaction not found for order_id $order_id at " . date('Y-m-d H:i:s'), 3, ERROR_LOG_PATH);
    header("Location: index.php?error=transaction_not_found");
    exit();
}

// Check transaction status with Pesapal
try {
    $status_url = PESAPAL_ENDPOINT . '/api/Transactions/GetTransactionStatus?orderTrackingId=' . urlencode($transaction['order_tracking_id']);
    $ch = curl_init($status_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . getPesapalToken(),
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, PESAPAL_IS_LIVE ? true : false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if (curl_errno($ch)) {
        error_log("Status check error for order_id $order_id: " . curl_error($ch), 3, ERROR_LOG_PATH);
        curl_close($ch);
    } else {
        curl_close($ch);
        $status_response = json_decode($response, true);
        error_log("Status check for order_id $order_id: HTTP $http_code, " . json_encode($status_response), 3, ERROR_LOG_PATH);

        if ($http_code === 200 && isset($status_response['status'])) {
            $new_status = $status_response['status'] === 'COMPLETED' ? 'COMPLETED' : ($status_response['status'] === 'FAILED' ? 'FAILED' : 'PENDING');
            if ($new_status !== $transaction['status']) {
                $conn = getDB();
                $stmt = $conn->prepare("UPDATE transactions SET status = ? WHERE order_id = ?");
                $stmt->bind_param("ss", $new_status, $order_id);
                $stmt->execute();
                $stmt->close();
                $conn->close();
                $transaction['status'] = $new_status;
            }
        }
    }
} catch (Exception $e) {
    error_log("Status check exception for order_id $order_id: " . $e->getMessage(), 3, ERROR_LOG_PATH);
}

// Redirect if FAILED
if ($transaction['status'] === 'FAILED') {
    header("Location: post_payment.php?order_id=$order_id&error=payment_failed");
    exit();
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Payment Status - Payroll System</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@300;400;500;600;700&display=swap" rel="stylesheet">
    <style>
        body {
            font-family: 'Inter', sans-serif;
            background: linear-gradient(135deg, #f5f7fa 0%, #c3cfe2 100%);
            min-height: 100vh;
            display: flex;
            align-items: center;
            justify-content: center;
        }
        .status-card {
            border: none;
            border-radius: 20px;
            box-shadow: 0 10px 30px rgba(0,0,0,0.1);
            max-width: 500px;
            width: 100%;
            margin: 2rem;
        }
        .status-header {
            background: linear-gradient(135deg, #667eea 0%, #764ba2 100%);
            color: white;
            padding: 1.5rem;
            text-align: center;
            border-radius: 20px 20px 0 0;
        }
        .status-body {
            padding: 2rem;
            text-align: center;
        }
        .btn-action {
            border-radius: 8px;
            padding: 0.75rem 1.5rem;
        }
    </style>
</head>
<body>
    <div class="status-card">
        <div class="status-header">
            <h3><i class="fas fa-receipt me-2"></i>Payment Status</h3>
        </div>
        <div class="status-body">
            <?php if (isset($_GET['error']) && $_GET['error'] === 'payment_failed'): ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle me-2"></i>Payment failed. Please try again.
                </div>
                <a href="index.php" class="btn btn-primary btn-action"><i class="fas fa-redo me-2"></i>Retry Payment</a>
            <?php elseif ($transaction['status'] === 'PENDING'): ?>
                <div class="alert alert-warning">
                    <i class="fas fa-hourglass-half me-2"></i>Your payment is pending. Waiting for confirmation from Pesapal.
                </div>
                <p>Order ID: <?php echo htmlspecialchars($order_id); ?></p>
                <p>Amount: KES <?php echo number_format($transaction['amount'], 2); ?></p>
                <a href="post_payment.php?order_id=<?php echo htmlspecialchars($order_id); ?>" class="btn btn-secondary btn-action"><i class="fas fa-sync me-2"></i>Check Status Again</a>
            <?php elseif ($transaction['status'] === 'COMPLETED'): ?>
                <div class="alert alert-success">
                    <i class="fas fa-check-circle me-2"></i>Payment completed successfully!
                </div>
                <p>Order ID: <?php echo htmlspecialchars($order_id); ?></p>
                <p>Amount: KES <?php echo number_format($transaction['amount'], 2); ?></p>
                <a href="index.php" class="btn btn-primary btn-action"><i class="fas fa-home me-2"></i>Back to Home</a>
            <?php else: ?>
                <div class="alert alert-danger">
                    <i class="fas fa-exclamation-circle me-2"></i>An error occurred. Please contact support.
                </div>
                <p>Order ID: <?php echo htmlspecialchars($order_id); ?></p>
                <a href="index.php" class="btn btn-primary btn-action"><i class="fas fa-home me-2"></i>Back to Home</a>
            <?php endif; ?>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
```