<?php
session_start();
ob_start(); // Start output buffering
require_once 'config.php';
require_once 'database.php';

if ($_SERVER['REQUEST_METHOD'] !== 'POST') {
    $_SESSION['error'] = "Invalid request method.";
    error_log("Invalid request method in process_payment.php", 3, ERROR_LOG_PATH);
    header("Location: index.php", true, 302);
    ob_end_flush();
    exit();
}

// Gather and validate input
$amount = filter_input(INPUT_POST, 'amount', FILTER_VALIDATE_FLOAT);
$payment_method = filter_input(INPUT_POST, 'payment_method', FILTER_SANITIZE_STRING);
$bank = filter_input(INPUT_POST, 'bank', FILTER_SANITIZE_STRING);
$email = filter_input(INPUT_POST, 'email', FILTER_SANITIZE_EMAIL);
$phone = filter_input(INPUT_POST, 'phone', FILTER_SANITIZE_STRING);

// Normalize phone
$phone_clean = preg_replace('/^\+/', '', $phone);
if (!$amount || $amount < 1 || !$email || !$phone || strpos($phone_clean, '254') !== 0 || strlen($phone_clean) !== 12) {
    $_SESSION['error'] = "Invalid input: Amount ≥ 1, valid email, and phone must be +2547xxxxxxxx.";
    error_log("Invalid input: amount=$amount, phone=$phone, email=$email", 3, ERROR_LOG_PATH);
    header("Location: index.php", true, 302);
    ob_end_flush();
    exit();
}

// Set user_id and order_id
$user_id = isset($_SESSION['user_id']) ? $_SESSION['user_id'] : 0;
$order_id = 'ORDER_' . time();

// Save transaction
$conn = getDB();
$stmt = $conn->prepare("INSERT INTO transactions (order_id, amount, payment_method, bank, email, phone, user_id, status) VALUES (?, ?, ?, ?, ?, ?, ?, 'PENDING')");
$stmt->bind_param("sdssssi", $order_id, $amount, $payment_method, $bank, $email, $phone, $user_id);
if (!$stmt->execute()) {
    error_log("Database Error: Failed to save transaction for order_id $order_id: " . $stmt->error, 3, ERROR_LOG_PATH);
    $_SESSION['error'] = "Failed to initiate payment.";
    header("Location: index.php", true, 302);
    ob_end_flush();
    exit();
}
$stmt->close();
$conn->close();

// Pesapal API: Get access token
try {
    $auth_url = PESAPAL_ENDPOINT . '/api/Auth/RequestToken';
    if (!filter_var($auth_url, FILTER_VALIDATE_URL)) {
        throw new Exception("Invalid auth URL: $auth_url");
    }
    error_log("Attempting auth request to: $auth_url", 3, ERROR_LOG_PATH);

    $auth_data = [
        'consumer_key' => PESAPAL_CONSUMER_KEY,
        'consumer_secret' => PESAPAL_CONSUMER_SECRET
    ];
    $ch = curl_init($auth_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($auth_data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, ['Content-Type: application/json']);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, PESAPAL_IS_LIVE ? true : false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if (curl_errno($ch)) {
        throw new Exception("Auth cURL error: " . curl_error($ch));
    }
    curl_close($ch);

    $auth_response = json_decode($response, true);
    error_log("Auth Response for order_id $order_id: HTTP $http_code, " . json_encode($auth_response), 3, ERROR_LOG_PATH);
    if ($http_code !== 200 || !isset($auth_response['token'])) {
        throw new Exception("Auth failed: " . ($auth_response['error'] ?? json_encode($auth_response)));
    }
    $access_token = $auth_response['token'];

    // Submit order (M-Pesa or Bank via Pesapal)
    $order_url = PESAPAL_ENDPOINT . '/api/Transactions/SubmitOrderRequest';
    if (!filter_var($order_url, FILTER_VALIDATE_URL)) {
        throw new Exception("Invalid order URL: $order_url");
    }
    error_log("Attempting order request to: $order_url", 3, ERROR_LOG_PATH);

    $payment_data = [
        'id' => $order_id,
        'currency' => 'KES',
        'amount' => $amount,
        'description' => "Payment for $order_id",
        'callback_url' => PESAPAL_CALLBACK_URL . '?order_id=' . urlencode($order_id),
        'notification_id' => PESAPAL_IPN_ID,
        'billing_address' => [
            'email_address' => $email,
            'phone_number' => $phone_clean,
            'country_code' => 'KE',
            'first_name' => isset($_SESSION['username']) ? $_SESSION['username'] : 'Guest'
        ]
    ];

    $ch = curl_init($order_url);
    curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
    curl_setopt($ch, CURLOPT_POST, true);
    curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payment_data));
    curl_setopt($ch, CURLOPT_HTTPHEADER, [
        'Authorization: Bearer ' . $access_token,
        'Content-Type: application/json'
    ]);
    curl_setopt($ch, CURLOPT_SSL_VERIFYPEER, PESAPAL_IS_LIVE ? true : false);
    curl_setopt($ch, CURLOPT_TIMEOUT, 30);
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    if (curl_errno($ch)) {
        throw new Exception("Order cURL error: " . curl_error($ch));
    }
    curl_close($ch);

    $order_response = json_decode($response, true);
    error_log("Order Response for order_id $order_id: HTTP $http_code, " . json_encode($order_response), 3, ERROR_LOG_PATH);

    if ($http_code === 200 && isset($order_response['redirect_url'])) {
        // Debug headers
        header('X-Redirect-Debug: Attempting redirect to ' . $order_response['redirect_url']);
        // Try PHP redirect
        header("Location: " . $order_response['redirect_url'], true, 302);
        // Fallback JavaScript redirect
        echo "<script>window.location.href='" . htmlspecialchars($order_response['redirect_url'], ENT_QUOTES, 'UTF-8') . "';</script>";
        ob_end_flush();
        exit();
    } else {
        throw new Exception("Order submission failed: " . ($order_response['error'] ?? json_encode($order_response)));
    }
} catch (Exception $e) {
    error_log("Pesapal API Error for order_id $order_id: " . $e->getMessage(), 3, ERROR_LOG_PATH);
    $_SESSION['error'] = "Payment initiation failed: " . $e->getMessage();
    header("Location: index.php", true, 302);
    ob_end_flush();
    exit();
}
?>