<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    redirect('../auth/login.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

$success = $error = '';

// Global garbage fee
$global_garbage = 500;
try {
    $global_garbage = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
} catch (Exception $e) {}

// Process form
if ($_POST) {
    $tenant_id = (int)$_POST['tenant_id'];
    $due_date = $_POST['due_date'];
    $notes = trim($_POST['notes'] ?? '');

    // Verify tenant belongs to this admin
    if (!$IS_SUPER_ADMIN) {
        $check = $pdo->prepare("SELECT t.id FROM tenants t JOIN properties p ON t.property_id = p.id WHERE t.id = ? AND p.admin_id = ?");
        $check->execute([$tenant_id, $admin_id]);
        if (!$check->fetch()) {
            $error = "You can only generate bills for your own tenants!";
            goto end_form;
        }
    }

    // === 1. Get base monthly charges ===
    $stmt = $pdo->prepare("
        SELECT COALESCE(p.rent_amount,0) as rent, COALESCE(p.water_bill,0) as water
        FROM tenants t
        LEFT JOIN properties p ON t.property_id = p.id
        WHERE t.id = ?
    ");
    $stmt->execute([$tenant_id]);
    $charges = $stmt->fetch();
    if (!$charges) {
        $error = "Tenant not found or no property assigned.";
        goto end_form;
    }
    $base_amount = $charges['rent'] + $charges['water'] + $global_garbage;

    // === 2. Get previous arrears ===
    $arrears = $pdo->prepare("
        SELECT COALESCE(SUM(amount - amount_paid), 0)
        FROM payments
        WHERE tenant_id = ? AND amount > amount_paid
    ");
    $arrears->execute([$tenant_id]);
    $arrears_amount = $arrears->fetchColumn();

    // === 3. Get previous credit ===
    $credit = $pdo->prepare("
        SELECT COALESCE(SUM(credit_carried), 0)
        FROM payments
        WHERE tenant_id = ?
    ");
    $credit->execute([$tenant_id]);
    $credit_amount = $credit->fetchColumn();

    // === 4. Calculate net amount due ===
    $net_due = $base_amount + $arrears_amount - $credit_amount;

    if ($net_due <= 0) {
        $final_amount = 0;
        $amount_paid = $credit_amount - $arrears_amount - $base_amount;
        $status = 'paid';
        $note_text = "Rent covered by previous credit/overpayment";
    } else {
        $final_amount = $net_due;
        $amount_paid = 0;
        $status = 'pending';
        $note_text = "Monthly rent";
        if ($arrears_amount > 0) $note_text .= " + arrears";
        if ($credit_amount > 0) $note_text .= " - credit applied";
    }

    // Prevent duplicate bill
    $check = $pdo->prepare("SELECT id FROM payments WHERE tenant_id = ? AND due_date = ?");
    $check->execute([$tenant_id, $due_date]);
    if ($check->fetch()) {
        $error = "Rent bill for this month already exists!";
    } elseif ($final_amount == 0 && $amount_paid <= 0) {
        $error = "No amount due (fully covered by credit)";
    } else {
        $pdo->prepare("
            INSERT INTO payments
            (tenant_id, amount, amount_paid, balance_carried, credit_carried, due_date, status, notes)
            VALUES (?, ?, ?, ?, ?, ?, ?, ?)
        ")->execute([
            $tenant_id,
            $final_amount > 0 ? $final_amount : $base_amount,
            $amount_paid,
            $arrears_amount,
            $net_due <= 0 ? abs($net_due) : 0,
            $due_date,
            $status,
            $notes ?: $note_text
        ]);

        $success = "<div class='alert alert-success'>
            <strong>Rent Bill Generated Successfully!</strong><br><br>
            <strong>Base Charges:</strong> KSh " . number_format($base_amount) . "<br>
            <strong>Previous Arrears:</strong> KSh " . number_format($arrears_amount) . "<br>
            <strong>Previous Credit:</strong> KSh " . number_format($credit_amount) . "<br>
            <strong>NET AMOUNT DUE:</strong> <h4>KSh " . number_format(max(0, $net_due)) . "</h4>
            " . ($net_due <= 0 ? "<span class='text-success'>Tenant has KSh " . number_format(abs($net_due)) . " credit for future!</span>" : "") . "
        </div>";
    }
}
end_form:

// Get tenants — FILTERED BY ADMIN
if ($IS_SUPER_ADMIN) {
    $tenants = $pdo->query("
        SELECT t.id, t.name, p.name as property_name,
               COALESCE(p.rent_amount,0) as rent_amount,
               COALESCE(p.water_bill,0) as water_bill
        FROM tenants t
        LEFT JOIN properties p ON t.property_id = p.id
        WHERE t.status = 'active'
        ORDER BY t.name
    ")->fetchAll();
} else {
    $tenants = $pdo->prepare("
        SELECT t.id, t.name, p.name as property_name,
               COALESCE(p.rent_amount,0) as rent_amount,
               COALESCE(p.water_bill,0) as water_bill
        FROM tenants t
        LEFT JOIN properties p ON t.property_id = p.id
        WHERE t.status = 'active' AND p.admin_id = ?
        ORDER BY t.name
    ");
    $tenants->execute([$admin_id]);
    $tenants = $tenants->fetchAll();
}

include '../includes/header.php';
?>

<div class="mb-4">
    <a href="dashboard.php" class="btn btn-dark btn-lg">Dashboard</a>
</div>

<h3 class="text-success mb-4">Generate Monthly Rent Bill (With Arrears & Credit)</h3>

<?php if ($success) echo $success; ?>
<?php if ($error): ?>
    <div class="alert alert-danger"><?= $error ?></div>
<?php endif; ?>

<div class="row">
    <div class="col-lg-8">
        <div class="card shadow">
            <div class="card-body">
                <form method="POST">
                    <div class="row g-3">
                        <div class="col-md-8">
                            <label class="form-label">Select Tenant *</label>
                            <select name="tenant_id" class="form-select form-select-lg" required onchange="showBreakdown(this)">
                                <option value="">-- Choose Tenant --</option>
                                <?php foreach ($tenants as $t):
                                    $base = $t['rent_amount'] + $t['water_bill'] + $global_garbage;

                                    $bal = $pdo->prepare("SELECT COALESCE(SUM(amount - amount_paid),0) FROM payments WHERE tenant_id=? AND amount > amount_paid");
                                    $bal->execute([$t['id']]); 
                                    $arrears = $bal->fetchColumn();

                                    $cred = $pdo->prepare("SELECT COALESCE(SUM(credit_carried),0) FROM payments WHERE tenant_id=?");
                                    $cred->execute([$t['id']]); 
                                    $credit = $cred->fetchColumn();

                                    $net = $base + $arrears - $credit;
                                ?>
                                <option value="<?= $t['id'] ?>"
                                    data-base="<?= $base ?>"
                                    data-arrears="<?= $arrears ?>"
                                    data-credit="<?= $credit ?>"
                                    data-net="<?= max(0, $net) ?>">
                                    <?= htmlspecialchars($t['name']) ?> (<?= $t['property_name'] ?? 'No Unit' ?>)
                                    → <strong>KSh <?= number_format(max(0, $net)) ?></strong>
                                    <?= $arrears > 0 ? " <span class='text-danger'>(+KSh " . number_format($arrears) . " arrears)</span>" : "" ?>
                                    <?= $credit > 0 ? " <span class='text-success'>(-KSh " . number_format($credit) . " credit)</span>" : "" ?>
                                </option>
                                <?php endforeach; ?>
                            </select>
                        </div>
                        <div class="col-md-4">
                            <label>Due Date *</label>
                            <input type="date" name="due_date" class="form-control form-control-lg"
                                   value="<?= date('Y-m-01', strtotime('+1 month')) ?>" required>
                        </div>
                        <div class="col-12">
                            <div id="breakdown" class="card border-primary">
                                <div class="card-body">
                                    <p class="text-muted">Select a tenant to see bill breakdown...</p>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-8">
                            <label>Notes (optional)</label>
                            <textarea name="notes" class="form-control" rows="2"></textarea>
                        </div>
                        <div class="col-md-4 align-self-end">
                            <button type="submit" class="btn btn-danger btn-lg w-100">
                                GENERATE RENT BILL
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </div>
    </div>
    <div class="col-lg-4">
        <div class="alert alert-info">
            <h5>Smart Billing System</h5>
            <ul class="mb-0">
                <li>Automatically adds arrears</li>
                <li>Automatically applies credit</li>
                <li>Shows net amount due</li>
                <li>Supports overpayments</li>
                <li>Zero due if credit covers rent</li>
            </ul>
        </div>
    </div>
</div>

<script>
function showBreakdown(sel) {
    if (!sel.value) {
        document.getElementById('breakdown').innerHTML = `<div class="card-body"><p class="text-muted">Select a tenant to see breakdown...</p></div>`;
        return;
    }
    const base = parseFloat(sel.selectedOptions[0].dataset.base);
    const arrears = parseFloat(sel.selectedOptions[0].dataset.arrears);
    const credit = parseFloat(sel.selectedOptions[0].dataset.credit);
    const net = parseFloat(sel.selectedOptions[0].dataset.net);
    let html = `<div class="card-body"><table class="table table-sm">`;
    html += `<tr><td>Base Rent + Water + Garbage</td><td class="text-end"><strong>KSh ${base.toLocaleString()}</strong></td></tr>`;
    if (arrears > 0) html += `<tr class="table-danger"><td>Previous Arrears</td><td class="text-end">+ KSh ${arrears.toLocaleString()}</td></tr>`;
    if (credit > 0) html += `<tr class="table-success"><td>Previous Credit Applied</td><td class="text-end">- KSh ${credit.toLocaleString()}</td></tr>`;
    html += `<tr class="table-primary"><td><h5>NET AMOUNT DUE THIS MONTH</h5></td><td class="text-end"><h4>KSh ${net.toLocaleString()}</h4></td></tr>`;
    html += `</table></div>`;
    document.getElementById('breakdown').innerHTML = html;
}
</script>

<?php include '../includes/footer.php'; ?>