<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    redirect('../auth/login.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

$success = $error = '';
$pre_property_id = (int)($_GET['property_id'] ?? 0);
$pre_property_name = htmlspecialchars($_GET['property_name'] ?? '');

// Keep form data on error
$form_data = $_POST;

if ($_POST) {
    $name = trim($_POST['name'] ?? '');
    $id_number = trim($_POST['id_number'] ?? '');
    $email = filter_var($_POST['email'] ?? '', FILTER_VALIDATE_EMAIL) ?: '';
    $phone = trim($_POST['phone'] ?? '');
    $next_of_kin = trim($_POST['next_of_kin'] ?? '');
    $kin_phone = trim($_POST['kin_phone'] ?? '');
    $property_id = (int)($_POST['property_id'] ?? 0);
    $move_in_date = $_POST['move_in_date'] ?? '';
    $generate_rent = isset($_POST['generate_first_rent']);

    if (!$name || !$id_number || !$phone || !$property_id || !$move_in_date) {
        $error = "Please fill all required fields!";
    } elseif ($email === false && !empty($_POST['email'])) {
        $error = "Invalid email address!";
    } else {
        try {
            $pdo->beginTransaction();

            // Verify property belongs to this admin (unless super admin)
            if (!$IS_SUPER_ADMIN) {
                $check_admin = $pdo->prepare("SELECT 1 FROM properties WHERE id = ? AND admin_id = ?");
                $check_admin->execute([$property_id, $admin_id]);
                if (!$check_admin->fetch()) {
                    throw new Exception("You can only assign tenants to your own properties!");
                }
            }

            // Check if property already has a tenant
            $check = $pdo->prepare("SELECT 1 FROM tenants WHERE property_id = ? AND status = 'active'");
            $check->execute([$property_id]);
            if ($check->fetch()) {
                throw new Exception("This property already has an active tenant!");
            }

            // Check duplicate phone or ID
            $dup = $pdo->prepare("SELECT 1 FROM tenants WHERE phone = ? OR id_number = ?");
            $dup->execute([$phone, $id_number]);
            if ($dup->fetch()) {
                throw new Exception("Phone number or ID already registered!");
            }

            // INSERT TENANT
            $stmt = $pdo->prepare("
                INSERT INTO tenants
                (name, id_number, email, phone, next_of_kin, kin_phone, property_id, move_in_date, status)
                VALUES (?, ?, ?, ?, ?, ?, ?, ?, 'active')
            ");
            $stmt->execute([$name, $id_number, $email ?: null, $phone, $next_of_kin, $kin_phone, $property_id, $move_in_date]);
            $tenant_id = $pdo->lastInsertId();

            // Update property status
            $pdo->prepare("UPDATE properties SET status = 'occupied' WHERE id = ?")->execute([$property_id]);

            // Generate first rent bill
            $total = 0;
            if ($generate_rent) {
                $prop = $pdo->prepare("SELECT rent_amount, water_bill FROM properties WHERE id = ?");
                $prop->execute([$property_id]);
                $p = $prop->fetch();

                $garbage = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;

                $total = ($p['rent_amount'] ?? 0) + ($p['water_bill'] ?? 0) + $garbage;
                $due_date = date('Y-m-01', strtotime("$move_in_date +1 month"));

                $pdo->prepare("INSERT INTO payments (tenant_id, amount, due_date, status, notes) VALUES (?, ?, ?, 'pending', 'First month rent')")
                    ->execute([$tenant_id, $total, $due_date]);
            }

            $pdo->commit();

            $success = "<strong>" . htmlspecialchars($name) . "</strong> added successfully!"
                     . ($generate_rent ? " <br><span class='text-success'>First rent bill: KSh " . number_format($total) . "</span>" : "");

            // Clear form after success
            $form_data = [];
            $_POST = [];
            $pre_property_id = 0;
            $pre_property_name = '';

        } catch (Exception $e) {
            $pdo->rollBack();
            $error = $e->getMessage();
        }
    }
}

// Load AVAILABLE properties — FILTERED BY ADMIN
if ($IS_SUPER_ADMIN) {
    $avail = $pdo->query("
        SELECT p.id, p.name 
        FROM properties p
        LEFT JOIN tenants t ON p.id = t.property_id AND t.status = 'active'
        WHERE t.id IS NULL
        ORDER BY p.name
    ")->fetchAll();
} else {
    $avail = $pdo->prepare("
        SELECT p.id, p.name 
        FROM properties p
        LEFT JOIN tenants t ON p.id = t.property_id AND t.status = 'active'
        WHERE t.id IS NULL AND p.admin_id = ?
        ORDER BY p.name
    ");
    $avail->execute([$admin_id]);
    $avail = $avail->fetchAll();
}

include '../includes/header.php';
?>

<div class="container py-5">
    <div class="d-flex gap-3 mb-5 flex-wrap">
        <a href="tenants.php" class="btn btn-outline-secondary btn-lg rounded-pill px-4">All Tenants</a>
        <a href="properties.php" class="btn btn-outline-primary btn-lg rounded-pill px-4">Properties</a>
        <a href="dashboard.php" class="btn btn-dark btn-lg rounded-pill px-4">Dashboard</a>
    </div>

    <h1 class="display-5 text-center fw-bold text-success mb-5">Add New Tenant</h1>

    <!-- SUCCESS TOAST -->
    <?php if ($success): ?>
    <div class="position-fixed top-0 end-0 p-4" style="z-index:9999">
        <div class="toast show text-bg-success border-0 shadow-lg">
            <div class="d-flex">
                <div class="toast-body fw-bold text-white fs-5">Success! <?= $success ?></div>
                <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
            </div>
        </div>
    </div>
    <script>
        setTimeout(() => document.querySelector('.toast')?.remove(), 5000);
    </script>
    <?php endif; ?>

    <!-- ERROR -->
    <?php if ($error): ?>
    <div class="alert alert-danger rounded-4 text-center py-4 mb-4 shadow">
        <strong><?= htmlspecialchars($error) ?></strong>
    </div>
    <?php endif; ?>

    <div class="card border-0 shadow-lg rounded-4 mx-auto" style="max-width:900px">
        <div class="card-header bg-success text-white text-center py-4">
            <h3 class="mb-0 fw-bold">
                <?= $pre_property_id ? "Adding to: <u>$pre_property_name</u>" : "New Tenant Registration" ?>
            </h3>
        </div>
        <div class="card-body p-5 bg-light">
            <form method="POST">
                <div class="row g-4">
                    <div class="col-lg-6">
                        <label class="fw-bold">Full Name <span class="text-danger">*</span></label>
                        <input type="text" name="name" class="form-control form-control-lg rounded-pill" required
                               value="<?= htmlspecialchars($form_data['name'] ?? '') ?>">
                    </div>
                    <div class="col-lg-6">
                        <label class="fw-bold">ID Number <span class="text-danger">*</span></label>
                        <input type="text" name="id_number" class="form-control form-control-lg rounded-pill" required
                               value="<?= htmlspecialchars($form_data['id_number'] ?? '') ?>">
                    </div>
                    <div class="col-lg-6">
                        <label class="fw-bold">Email (Optional)</label>
                        <input type="email" name="email" class="form-control form-control-lg rounded-pill"
                               value="<?= htmlspecialchars($form_data['email'] ?? '') ?>">
                    </div>
                    <div class="col-lg-6">
                        <label class="fw-bold">Phone Number <span class="text-danger">*</span></label>
                        <input type="text" name="phone" class="form-control form-control-lg rounded-pill" required
                               value="<?= htmlspecialchars($form_data['phone'] ?? '') ?>">
                    </div>
                    <div class="col-lg-6">
                        <label class="fw-bold">Next of Kin</label>
                        <input type="text" name="next_of_kin" class="form-control form-control-lg rounded-pill"
                               value="<?= htmlspecialchars($form_data['next_of_kin'] ?? '') ?>">
                    </div>
                    <div class="col-lg-6">
                        <label class="fw-bold">Kin Phone</label>
                        <input type="text" name="kin_phone" class="form-control form-control-lg rounded-pill"
                               value="<?= htmlspecialchars($form_data['kin_phone'] ?? '') ?>">
                    </div>
                    <div class="col-lg-8">
                        <label class="fw-bold text-primary">Property <span class="text-danger">*</span></label>
                        <select name="property_id" class="form-select form-select-lg rounded-pill" required>
                            <option value="">-- Select Available Property --</option>
                            <?php foreach ($avail as $p): ?>
                                <option value="<?= $p['id'] ?>"
                                    <?= ($p['id'] == $pre_property_id || $p['id'] == ($form_data['property_id'] ?? 0)) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($p['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>
                    <div class="col-lg-4">
                        <label class="fw-bold">Move-in Date <span class="text-danger">*</span></label>
                        <input type="date" name="move_in_date" class="form-control form-control-lg rounded-pill" required
                               value="<?= $form_data['move_in_date'] ?? date('Y-m-d') ?>">
                    </div>
                    <div class="col-12">
                        <div class="form-check form-switch">
                            <input class="form-check-input" type="checkbox" name="generate_first_rent" id="gen" checked>
                            <label class="form-check-label fw-bold text-success fs-5" for="gen">
                                Generate First Month Rent
                            </label>
                        </div>
                    </div>
                    <div class="col-12 text-center mt-5">
                        <button type="submit" class="btn btn-success btn-lg px-5 py-3 rounded-pill shadow-lg fw-bold">
                            ADD TENANT NOW
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <div class="text-center mt-5 text-muted">
        <p class="fw-bold mb-0">Rene.C.Agency • Made in Kenya</p>
    </div>
</div>

<?php include '../includes/footer.php'; ?>