<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    redirect('../auth/login.php');
}

$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1);

// GLOBAL GARBAGE FEE
$garbage_fee = 500;
try {
    $garbage_fee = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
} catch (Exception $e) {}

// AUTO GENERATE RENT
$today = date('Y-m-01');
if (!isset($_SESSION['rent_generated']) || $_SESSION['rent_generated'] !== $today) {
    $generated = 0;

    if ($IS_SUPER_ADMIN) {
        $stmt = $pdo->query("SELECT t.id, p.rent_amount, p.water_bill FROM tenants t JOIN properties p ON t.property_id = p.id WHERE t.status = 'active'");
    } else {
        $stmt = $pdo->prepare("SELECT t.id, p.rent_amount, p.water_bill FROM tenants t JOIN properties p ON t.property_id = p.id WHERE t.status = 'active' AND p.admin_id = ?");
        $stmt->execute([$admin_id]);
    }

    foreach ($stmt->fetchAll() as $tenant) {
        $check = $pdo->prepare("SELECT 1 FROM payments WHERE tenant_id = ? AND due_date = ? LIMIT 1");
        $check->execute([$tenant['id'], $today]);
        if (!$check->fetch()) {
            $amount = ($tenant['rent_amount'] ?? 0) + ($tenant['water_bill'] ?? 0) + $garbage_fee;
            $pdo->prepare("INSERT INTO payments (tenant_id, amount, due_date, status) VALUES (?, ?, ?, 'pending')")
                ->execute([$tenant['id'], $amount, $today]);
            $generated++;
        }
    }

    if ($generated > 0) {
        $_SESSION['success'] = "Rent auto-generated for $generated tenants – " . date('F Y');
    }
    $_SESSION['rent_generated'] = $today;
}

// STATS - CURRENT MONTH ONLY
if ($IS_SUPER_ADMIN) {
    $total_tenants = $pdo->query("SELECT COUNT(*) FROM tenants WHERE status='active'")->fetchColumn();
    $occupied_units = $pdo->query("SELECT COUNT(*) FROM properties WHERE status='occupied'")->fetchColumn();
} else {
    $stmt_tenants = $pdo->prepare("SELECT COUNT(*) FROM tenants t JOIN properties p ON t.property_id = p.id WHERE t.status='active' AND p.admin_id = ?");
    $stmt_tenants->execute([$admin_id]);
    $total_tenants = $stmt_tenants->fetchColumn();

    $stmt_units = $pdo->prepare("SELECT COUNT(*) FROM properties WHERE status='occupied' AND admin_id = ?");
    $stmt_units->execute([$admin_id]);
    $occupied_units = $stmt_units->fetchColumn();
}

$current_month = date('Y-m-01');
$this_month_str = date('Y-m');

// Expected This Month
$expected_this_month = 0;
if ($IS_SUPER_ADMIN) {
    $stmt = $pdo->query("SELECT amount FROM payments WHERE due_date = '$current_month'");
} else {
    $stmt = $pdo->prepare("SELECT p.amount FROM payments p JOIN tenants t ON p.tenant_id = t.id JOIN properties pr ON t.property_id = pr.id WHERE p.due_date = ? AND pr.admin_id = ?");
    $stmt->execute([$current_month, $admin_id]);
}
foreach ($stmt->fetchAll() as $row) {
    $expected_this_month += $row['amount'];
}

// Collected This Month
if ($IS_SUPER_ADMIN) {
    $collected_this_month = $pdo->query("SELECT COALESCE(SUM(amount_paid), 0) FROM payments WHERE DATE_FORMAT(paid_date, '%Y-%m') = '$this_month_str'")->fetchColumn();
} else {
    $stmt = $pdo->prepare("SELECT COALESCE(SUM(p.amount_paid), 0) FROM payments p JOIN tenants t ON p.tenant_id = t.id JOIN properties pr ON t.property_id = pr.id WHERE DATE_FORMAT(p.paid_date, '%Y-%m') = ? AND pr.admin_id = ?");
    $stmt->execute([$this_month_str, $admin_id]);
    $collected_this_month = $stmt->fetchColumn();
}

// RECENT PAYMENTS
if (isset($_GET['export_activity']) && $_GET['export_activity'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="ReneCAgency_Recent_Activity_' . date('Y-m-d') . '.csv"');
    $output = fopen('php://output', 'w');
    fputcsv($output, ['RENE.C.AGENCY - RECENT ACTIVITY EXPORT']);
    fputcsv($output, ['Generated On:', date('d F Y, H:i')]);
    fputcsv($output, []);
    fputcsv($output, ['Type', 'Tenant', 'Amount', 'Date', 'Time']);

    if ($IS_SUPER_ADMIN) {
        $activity = $pdo->query("SELECT 'Payment Received' as type, t.name, p.amount_paid as amount, p.paid_date as dt FROM payments p JOIN tenants t ON p.tenant_id = t.id WHERE p.amount_paid > 0 AND p.paid_date IS NOT NULL ORDER BY p.paid_date DESC LIMIT 500");
    } else {
        $activity = $pdo->prepare("SELECT 'Payment Received' as type, t.name, p.amount_paid as amount, p.paid_date as dt FROM payments p JOIN tenants t ON p.tenant_id = t.id JOIN properties pr ON t.property_id = pr.id WHERE p.amount_paid > 0 AND p.paid_date IS NOT NULL AND pr.admin_id = ? ORDER BY p.paid_date DESC LIMIT 500");
        $activity->execute([$admin_id]);
    }

    foreach ($activity as $row) {
        if ($row['dt']) {
            fputcsv($output, [
                $row['type'],
                $row['name'],
                'KSh ' . number_format($row['amount']),
                date('d M Y', strtotime($row['dt'])),
                date('h:i A', strtotime($row['dt']))
            ]);
        }
    }
    exit;
}

if ($IS_SUPER_ADMIN) {
    $recent = $pdo->query("SELECT t.name, p.amount_paid as amount, p.paid_date as dt FROM payments p JOIN tenants t ON p.tenant_id = t.id WHERE p.amount_paid > 0 AND p.paid_date IS NOT NULL ORDER BY p.paid_date DESC LIMIT 12")->fetchAll();
} else {
    $stmt = $pdo->prepare("SELECT t.name, p.amount_paid as amount, p.paid_date as dt FROM payments p JOIN tenants t ON p.tenant_id = t.id JOIN properties pr ON t.property_id = pr.id WHERE p.amount_paid > 0 AND p.paid_date IS NOT NULL AND pr.admin_id = ? ORDER BY p.paid_date DESC LIMIT 12");
    $stmt->execute([$admin_id]);
    $recent = $stmt->fetchAll();
}

function ksh($num) { return 'KSh ' . number_format($num ?? 0); }

// TENANT DETAILS — FULL HISTORY LIKE REPORTS.PHP
$tenant_details = [];

if ($IS_SUPER_ADMIN) {
    $tenants = $pdo->query("
        SELECT t.id, t.name, t.move_in_date, p.rent_amount, p.water_bill, COALESCE(pr.name, '—') AS unit
        FROM tenants t
        JOIN properties p ON t.property_id = p.id
        LEFT JOIN properties pr ON t.property_id = pr.id
        WHERE t.status = 'active'
        ORDER BY t.name
    ");
} else {
    $tenants = $pdo->prepare("
        SELECT t.id, t.name, t.move_in_date, p.rent_amount, p.water_bill, COALESCE(pr.name, '—') AS unit
        FROM tenants t
        JOIN properties p ON t.property_id = p.id
        LEFT JOIN properties pr ON t.property_id = pr.id
        WHERE t.status = 'active' AND p.admin_id = ?
        ORDER BY t.name
    ");
    $tenants->execute([$admin_id]);
}

foreach ($tenants as $t) {
    $monthly_fixed = ($t['rent_amount'] ?? 0) + ($t['water_bill'] ?? 0) + $garbage_fee;
    $move_in = new DateTime($t['move_in_date']);
    $first_bill_month = ($move_in->format('d') == '01') ? $move_in->format('Y-m-01') : (clone $move_in)->modify('first day of next month')->format('Y-m-01');

    $period = new DatePeriod(new DateTime($first_bill_month), new DateInterval('P1M'), new DateTime('first day of next month'));

    $total_billed = $total_paid = $total_credit = 0;
    $history = [];

    foreach ($period as $dt) {
        $due = $dt->format('Y-m-01');
        $month_name = $dt->format('M Y');
        $days = $dt->format('t');
        $prorated = ($due === $first_bill_month) ? round($monthly_fixed * ($days - ($move_in->format('j') - 1)) / $days) : $monthly_fixed;

        $stmt_p = $pdo->prepare("SELECT amount, amount_paid, COALESCE(credit_carried,0) as credit_carried FROM payments WHERE tenant_id = ? AND due_date = ?");
        $stmt_p->execute([$t['id'], $due]);
        $p = $stmt_p->fetch();

        $billed = $p ? $p['amount'] : $prorated;
        $paid = $p ? $p['amount_paid'] : 0;
        $credit_used = $p ? $p['credit_carried'] : 0;
        $balance = $billed - $paid - $credit_used;

        $total_billed += $billed;
        $total_paid += $paid;
        $total_credit += $credit_used;

        $history[] = [
            'month' => $month_name,
            'billed' => $billed,
            'paid' => $paid,
            'credit_used' => $credit_used,
            'balance' => $balance,
            'status' => $balance > 0 ? 'danger' : 'success',
            'status_text' => $balance > 0 ? 'OWES KSh ' . number_format($balance) : 'PAID'
        ];
    }

    $arrears = $total_billed - $total_paid;

    $tenant_details[] = [
        'id' => $t['id'],
        'name' => $t['name'],
        'unit' => $t['unit'],
        'total_billed' => $total_billed,
        'total_paid' => $total_paid,
        'arrears' => $arrears,
        'credit' => $total_credit,
        'history' => $history
    ];
}

include '../includes/header.php';
?>

<?php if (isset($_SESSION['success'])): ?>
<div id="successToast" class="position-fixed top-0 start-50 translate-middle-x shadow-lg rounded-4 border-0" style="z-index: 9999; margin-top: 20px;">
    <div class="alert alert-success m-0 py-4 px-5 text-center fw-bold fs-5 text-white" style="background: linear-gradient(135deg, #16a34a, #15803d);">
        <?= $_SESSION['success'] ?>
    </div>
</div>
<script>setTimeout(() => document.getElementById('successToast')?.remove(), 6000);</script>
<?php unset($_SESSION['success']); ?>
<?php endif; ?>

<div class="container py-5">
    <div class="text-center mb-5 p-5 rounded-4 hero-box card-shadow">
        <h1 class="display-5 fw-bold text-success mb-3">Welcome back, <?= htmlspecialchars($_SESSION['username']) ?>!</h1>
        <p class="lead">Today is <?= date('l, jS F Y') ?> • Everything running smoothly</p>
    </div>

    <div class="row g-4 mb-5 text-center">
        <div class="col-md-3 col-6">
            <div class="card border-0 card-shadow h-100" style="background: #1e40af;">
                <div class="card-body text-white p-4"><h6>Active Tenants</h6><h2 class="fw-bold"><?= $total_tenants ?></h2></div>
            </div>
        </div>
        <div class="col-md-3 col-6">
            <div class="card border-0 card-shadow h-100" style="background: #059669;">
                <div class="card-body text-white p-4"><h6>Occupied Units</h6><h2 class="fw-bold"><?= $occupied_units ?></h2></div>
            </div>
        </div>
        <div class="col-md-3 col-6">
            <div class="card border-0 card-shadow h-100" style="background: #ca8a04;">
                <div class="card-body text-white p-4"><h6>Expected This Month</h6><h3 class="fw-bold"><?= ksh($expected_this_month) ?></h3></div>
            </div>
        </div>
        <div class="col-md-3 col-6">
            <div class="card border-0 card-shadow h-100" style="background: #dc2626;">
                <div class="card-body text-white p-4"><h6>Collected This Month</h6><h3 class="fw-bold"><?= ksh($collected_this_month) ?></h3></div>
            </div>
        </div>
    </div>

    <ul class="nav nav-pills mb-4 justify-content-center card-shadow rounded-pill overflow-hidden" style="background: white;">
        <li class="nav-item"><a class="nav-link active px-5" data-bs-toggle="tab" href="#summary">Summary</a></li>
        <li class="nav-item"><a class="nav-link px-5" data-bs-toggle="tab" href="#activity">Activity</a></li>
        <li class="nav-item"><a class="nav-link px-5" data-bs-toggle="tab" href="#balances">Balances</a></li>
    </ul>

    <div class="tab-content">
        <div class="tab-pane fade show active" id="summary">
            <div class="row g-4">
                <div class="col-lg-8">
                    <div class="card card-shadow border-0">
                        <div class="card-body p-5 text-center">
                            <h4 class="text-success mb-4"><?= date('F Y') ?> Performance</h4>
                            <div class="row g-5">
                                <div class="col"><small class="text-muted">Expected</small><h2><?= ksh($expected_this_month) ?></h2></div>
                                <div class="col"><small class="text-muted">Collected</small><h2 class="text-success"><?= ksh($collected_this_month) ?></h2></div>
                            </div>
                            <div class="mt-5">
                                <div class="progress rounded-pill">
                                    <div class="progress-bar bg-success fw-bold fs-5" style="width: <?= $expected_this_month > 0 ? round(($collected_this_month / $expected_this_month) * 100) : 0 ?>%">
                                        <?= $expected_this_month > 0 ? round(($collected_this_month / $expected_this_month) * 100) : 0 ?>% Collected
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                <div class="col-lg-4">
                    <div class="card card-shadow border-0" style="background: #f8f9fa;">
                        <div class="card-header bg-success text-white text-center py-3"><h5 class="mb-0">Quick Actions</h5></div>
                        <div class="list-group list-group-flush">
                            <a href="add_payment.php" class="list-group-item list-group-item-action py-3 text-danger fw-bold">Generate Monthly Rent</a>
                            <a href="payments.php" class="list-group-item list-group-item-action py-3">Record Payment</a>
                            <a href="add_tenant.php" class="list-group-item list-group-item-action py-3">Add New Tenant</a>
                            <a href="add_property.php" class="list-group-item list-group-item-action py-3">Add New Property</a>
                            <a href="reports.php" class="list-group-item list-group-item-action py-3 bg-warning text-dark fw-bold">View Full Reports</a>
                        </div>
                    </div>
                </div>
            </div>
        </div>

        <div class="tab-pane fade" id="activity">
            <div class="card card-shadow border-0">
                <div class="card-header bg-primary text-white d-flex justify-content-between align-items-center">
                    <h5 class="mb-0">Recent Payments</h5>
                    <a href="?export_activity=csv" class="btn btn-light btn-sm fw-bold">Export to CSV</a>
                </div>
                <div class="card-body p-0">
                    <div class="list-group list-group-flush">
                        <?php if (empty($recent)): ?>
                            <div class="list-group-item text-center py-5 text-muted">No recent payments</div>
                        <?php else: foreach ($recent as $r): ?>
                            <div class="list-group-item px-4 py-3">
                                <div class="d-flex justify-content-between align-items-start">
                                    <div class="me-3">
                                        <i class="bi bi-check-circle-fill text-success fs-4"></i>
                                    </div>
                                    <div class="flex-grow-1">
                                        <strong class="text-success">Payment Received</strong><br>
                                        <span class="text-muted">from</span> <strong><?= htmlspecialchars($r['name']) ?></strong>
                                        <div class="mt-1">
                                            <span class="badge bg-success fw-bold">
                                                <?= ksh($r['amount']) ?>
                                            </span>
                                        </div>
                                    </div>
                                    <small class="text-muted text-nowrap">
                                        <?= date('d M Y', strtotime($r['dt'])) ?><br>
                                        <?= date('h:i A', strtotime($r['dt'])) ?>
                                    </small>
                                </div>
                            </div>
                        <?php endforeach; endif; ?>
                    </div>
                </div>
            </div>
        </div>

        <div class="tab-pane fade" id="balances">
            <div class="card card-shadow border-0">
                <div class="card-header bg-info text-white text-center py-4">
                    <h5 class="mb-0">Tenant Balance Summary • Click Name for Monthly History</h5>
                </div>
                <div class="accordion" id="balancesAccordion">
                    <?php foreach ($tenant_details as $i => $td): ?>
                    <div class="accordion-item border-0 shadow-sm mb-3 rounded overflow-hidden">
                        <h2 class="accordion-header">
                            <button class="accordion-button <?= $i===0 ? '' : 'collapsed' ?> fw-bold py-3 text-start" type="button" data-bs-toggle="collapse"
                                    data-bs-target="#tenant<?= $td['id'] ?>">
                                <div class="w-100 d-flex justify-content-between align-items-center pe-3">
                                    <div>
                                        <strong><?= htmlspecialchars($td['name']) ?></strong> 
                                        <span class="text-muted small">— <?= $td['unit'] ?></span>
                                    </div>
                                    <div class="text-end">
                                        <small>Billed: <?= ksh($td['total_billed']) ?></small> |
                                        <small>Paid: <?= ksh($td['total_paid']) ?></small> |
                                        <?php if ($td['arrears'] > 0): ?>
                                            <span class="badge bg-danger fs-6 px-3">OWES <?= ksh($td['arrears']) ?></span>
                                        <?php elseif ($td['credit'] > 0): ?>
                                            <span class="badge bg-success fs-6 px-3">CREDIT <?= ksh($td['credit']) ?></span>
                                        <?php else: ?>
                                            <span class="badge bg-secondary fs-6 px-3">CLEARED</span>
                                        <?php endif; ?>
                                    </div>
                                </div>
                            </button>
                        </h2>
                        <div id="tenant<?= $td['id'] ?>" class="accordion-collapse collapse <?= $i===0?'show':'' ?>">
                            <div class="accordion-body p-0">
                                <div class="table-responsive">
                                    <table class="table table-sm table-hover mb-0">
                                        <thead class="table-light">
                                            <tr>
                                                <th>Month</th>
                                                <th class="text-end">Billed</th>
                                                <th class="text-end">Paid</th>
                                                <th class="text-end">Credit Used</th>
                                                <th class="text-center">Balance</th>
                                                <th>Status</th>
                                            </tr>
                                        </thead>
                                        <tbody>
                                            <?php foreach ($td['history'] as $h): ?>
                                            <tr>
                                                <td><?= $h['month'] ?></td>
                                                <td class="text-end"><?= ksh($h['billed']) ?></td>
                                                <td class="text-end text-success fw-bold"><?= $h['paid'] > 0 ? ksh($h['paid']) : '—' ?></td>
                                                <td class="text-end text-info"><?= $h['credit_used'] > 0 ? ksh($h['credit_used']) : '—' ?></td>
                                                <td class="text-center fw-bold <?= $h['balance'] > 0 ? 'text-danger' : 'text-success' ?>">
                                                    <?= $h['balance'] > 0 ? '+' . ksh($h['balance']) : ($h['balance'] < 0 ? '-' . ksh(abs($h['balance'])) : 'Cleared') ?>
                                                </td>
                                                <td>
                                                    <span class="badge bg-<?= $h['status'] ?>">
                                                        <?= $h['balance'] > 0 ? 'OWES' : 'PAID' ?>
                                                    </span>
                                                </td>
                                            </tr>
                                            <?php endforeach; ?>
                                        </tbody>
                                    </table>
                                </div>
                            </div>
                        </div>
                    </div>
                    <?php endforeach; ?>
                </div>
            </div>
        </div>
    </div>

    <div class="text-center mt-5 py-4 text-muted">
        <p class="mb-0">© <?= date('Y') ?> Rene.C.Agency • Made in Kenya with Excellence</p>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>