<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    redirect('../auth/login.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

$id = (int)($_GET['id'] ?? 0);
if ($id < 1) {
    redirect('payments.php');
}

// Fetch payment + tenant + property details — WITH ADMIN CHECK
if ($IS_SUPER_ADMIN) {
    $stmt = $pdo->prepare("
        SELECT p.*, t.name as tenant_name, t.property_id,
               pr.rent_amount, pr.water_bill, pr.admin_id
        FROM payments p
        JOIN tenants t ON p.tenant_id = t.id
        LEFT JOIN properties pr ON t.property_id = pr.id
        WHERE p.id = ?
    ");
    $stmt->execute([$id]);
} else {
    $stmt = $pdo->prepare("
        SELECT p.*, t.name as tenant_name, t.property_id,
               pr.rent_amount, pr.water_bill, pr.admin_id
        FROM payments p
        JOIN tenants t ON p.tenant_id = t.id
        LEFT JOIN properties pr ON t.property_id = pr.id
        WHERE p.id = ? AND pr.admin_id = ?
    ");
    $stmt->execute([$id, $admin_id]);
}

$pay = $stmt->fetch();

if (!$pay) {
    $_SESSION['error'] = "Payment not found or access denied.";
    redirect('payments.php');
}

// Get global garbage fee
$global_garbage = 500;
try {
    $global_garbage = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
} catch (Exception $e) {}

if ($_POST) {
    $rent_amount = (float)($_POST['rent_amount'] ?? 0);
    $water_bill = (float)($_POST['water_bill'] ?? 0);
    $due_date = $_POST['due_date'] ?? '';
    $notes = trim($_POST['notes'] ?? '');

    if ($rent_amount < 0 || $water_bill < 0) {
        $error = "Amounts cannot be negative!";
    } elseif (empty($due_date)) {
        $error = "Due date is required!";
    } else {
        $total = $rent_amount + $water_bill + $global_garbage;

        try {
            $pdo->beginTransaction();

            // Update payment
            $update = $pdo->prepare("UPDATE payments SET amount = ?, due_date = ?, notes = ? WHERE id = ?");
            $update->execute([$total, $due_date, $notes, $id]);

            // Update property base charges if property exists
            if ($pay['property_id']) {
                $pdo->prepare("UPDATE properties SET rent_amount = ?, water_bill = ? WHERE id = ? AND admin_id = ?")
                    ->execute([$rent_amount, $water_bill, $pay['property_id'], $admin_id]);
            }

            $pdo->commit();

            $_SESSION['success'] = "Payment updated! New total: KSh " . number_format($total);
            redirect('payments.php');
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "Update failed. Please try again.";
        }
    }
}

$total_current = $pay['amount'];
include '../includes/header.php';
?>

<div class="mb-4">
    <a href="payments.php" class="btn btn-secondary btn-lg">Payments List</a>
    <a href="dashboard.php" class="btn btn-dark btn-lg ms-2">Dashboard</a>
</div>

<h3 class="text-success mb-4">Edit Payment for: <strong><?= htmlspecialchars($pay['tenant_name']) ?></strong></h3>

<div class="alert alert-info">
    <strong>Current Total:</strong> KSh <?= number_format($total_current) ?><br>
    <strong>Global Garbage Fee:</strong> KSh <?= number_format($global_garbage) ?> (not editable here)
</div>

<?php if (isset($error)): ?>
    <div class="alert alert-danger rounded-4 shadow-sm">
        <?= htmlspecialchars($error) ?>
    </div>
<?php endif; ?>

<div class="card border-0 shadow-lg">
    <div class="card-body p-4">
        <form method="POST">
            <div class="row g-3">
                <div class="col-md-4">
                    <label class="form-label fw-bold">Rent Amount (KSh)</label>
                    <input type="number" step="0.01" name="rent_amount" class="form-control form-control-lg"
                           value="<?= htmlspecialchars($pay['rent_amount'] ?? 0) ?>" required>
                </div>
                <div class="col-md-4">
                    <label class="form-label fw-bold">Water Bill (KSh)</label>
                    <input type="number" step="0.01" name="water_bill" class="form-control form-control-lg"
                           value="<?= htmlspecialchars($pay['water_bill'] ?? 0) ?>" required>
                </div>
                <div class="col-md-4">
                    <label class="form-label fw-bold">Total (Auto-calculated)</label>
                    <input type="text" class="form-control form-control-lg bg-light fw-bold text-success"
                           value="KSh <?= number_format(($pay['rent_amount']??0) + ($pay['water_bill']??0) + $global_garbage) ?>" disabled>
                </div>
                <div class="col-md-6">
                    <label class="form-label fw-bold">Due Date</label>
                    <input type="date" name="due_date" class="form-control form-control-lg" value="<?= $pay['due_date'] ?>" required>
                </div>
                <div class="col-md-6">
                    <label class="form-label fw-bold">Notes (optional)</label>
                    <textarea name="notes" class="form-control" rows="3"><?= htmlspecialchars($pay['notes'] ?? '') ?></textarea>
                </div>
                <div class="col-12 mt-4">
                    <button type="submit" class="btn btn-success btn-lg px-5 fw-bold shadow">
                        UPDATE PAYMENT
                    </button>
                    <?php if ($pay['status'] != 'paid'): ?>
                    <a href="payments.php?paid=<?= $pay['id'] ?>" class="btn btn-warning btn-lg ms-3 fw-bold"
                       onclick="return confirm('Mark this bill as PAID today?')">
                        Mark as Paid
                    </a>
                    <?php endif; ?>
                </div>
            </div>
        </form>
    </div>
</div>

<?php include '../includes/footer.php'; ?>