<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    redirect('../auth/login.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

$id = (int)($_GET['id'] ?? 0);
if ($id < 1) {
    redirect('properties.php');
}

// Fetch property — WITH ADMIN CHECK
if ($IS_SUPER_ADMIN) {
    $stmt = $pdo->prepare("SELECT * FROM properties WHERE id = ?");
    $stmt->execute([$id]);
} else {
    $stmt = $pdo->prepare("SELECT * FROM properties WHERE id = ? AND admin_id = ?");
    $stmt->execute([$id, $admin_id]);
}

$prop = $stmt->fetch();

if (!$prop) {
    $_SESSION['error'] = "Property not found or access denied!";
    redirect('properties.php');
}

$success = $error = '';

// Global garbage fee
$garbage_fee = 500;
try {
    $garbage_fee = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
} catch (Exception $e) {}

// Create monthly_bills table safely
try {
    $pdo->exec("CREATE TABLE IF NOT EXISTS monthly_bills (
        id INT AUTO_INCREMENT PRIMARY KEY,
        property_id INT NOT NULL,
        bill_month VARCHAR(7) NOT NULL,
        water_units DECIMAL(8,2) NOT NULL DEFAULT 0,
        water_bill INT NOT NULL DEFAULT 0,
        UNIQUE KEY uniq_month (property_id, bill_month)
    )");
} catch (Exception $e) {}

// SAVE WATER BILL — ONLY ONE MONTH
if (isset($_POST['save_water_bill'])) {
    $month = $_POST['bill_month'] ?? '';
    $units = (float)($_POST['water_units'] ?? 0);
    $water_bill = (int)round($units * 120);

    if (strlen($month) === 7 && $units >= 0) {
        try {
            $pdo->beginTransaction();

            // Save water bill
            $check = $pdo->prepare("SELECT id FROM monthly_bills WHERE property_id = ? AND bill_month = ?");
            $check->execute([$id, $month]);
            if ($check->fetchColumn()) {
                $pdo->prepare("UPDATE monthly_bills SET water_units = ?, water_bill = ? WHERE property_id = ? AND bill_month = ?")
                    ->execute([$units, $water_bill, $id, $month]);
            } else {
                $pdo->prepare("INSERT INTO monthly_bills (property_id, bill_month, water_units, water_bill) VALUES (?, ?, ?, ?)")
                    ->execute([$id, $month, $units, $water_bill]);
            }

            // Update ONLY this month in payments
            $tenant_id = $pdo->prepare("SELECT id FROM tenants WHERE property_id = ? AND status='active'")->execute([$id]) ? $pdo->fetchColumn() : false;

            if ($tenant_id) {
                $due = "$month-01";
                $total = $prop['rent_amount'] + $water_bill + $garbage_fee;

                $exists = $pdo->prepare("SELECT id FROM payments WHERE tenant_id = ? AND due_date = ?");
                $exists->execute([$tenant_id, $due]);

                if ($exists->fetchColumn()) {
                    $pdo->prepare("UPDATE payments SET amount = ? WHERE tenant_id = ? AND due_date = ?")
                        ->execute([$total, $tenant_id, $due]);
                } else {
                    $pdo->prepare("INSERT INTO payments (tenant_id, amount, amount_paid, credit_carried, due_date, status) VALUES (?, ?, 0, 0, ?, 'pending')")
                        ->execute([$tenant_id, $total, $due]);
                }
            }

            $pdo->commit();
            $success = "Water bill for " . date('F Y', strtotime("$month-01")) . " saved: KSh " . number_format($water_bill);
        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "Save failed. Please try again.";
        }
    } else {
        $error = "Invalid month or units.";
    }
}

// UPDATE RENT & PROPERTY DETAILS
if (isset($_POST['update_property'])) {
    $name = trim($_POST['name'] ?? '');
    $rent = (float)($_POST['rent_amount'] ?? 0);
    $address = trim($_POST['address'] ?? '');
    $status = $_POST['status'] ?? 'available';

    if ($name && $rent >= 0) {
        try {
            $pdo->prepare("UPDATE properties SET name = ?, address = ?, rent_amount = ?, status = ? WHERE id = ? AND admin_id = ?")
                ->execute([$name, $address, $rent, $status, $id, $admin_id]);

            $success = "Property updated successfully!";
            $prop['name'] = $name;
            $prop['rent_amount'] = $rent;
            $prop['address'] = $address;
            $prop['status'] = $status;
        } catch (Exception $e) {
            $error = "Update failed.";
        }
    } else {
        $error = "Please fill all required fields.";
    }
}

// Current month water bill
$current_month = date('Y-m');
$current_water = 0;
try {
    $stmt = $pdo->prepare("SELECT water_bill FROM monthly_bills WHERE property_id = ? AND bill_month = ?");
    $stmt->execute([$id, $current_month]);
    $current_water = (int)$stmt->fetchColumn();
} catch (Exception $e) {}

$total_bill = $prop['rent_amount'] + $current_water + $garbage_fee;

include '../includes/header.php';
?>

<?php if ($success): ?>
<div class="position-fixed top-0 end-0 p-4" style="z-index:9999">
    <div class="toast show align-items-center text-bg-success border-0 shadow-lg">
        <div class="d-flex">
            <div class="toast-body fw-bold"><?= $success ?></div>
            <button type="button" class="btn-close btn-close-white" data-bs-dismiss="toast"></button>
        </div>
    </div>
</div>
<script>setTimeout(() => document.querySelector('.toast')?.remove(), 4000);</script>
<?php endif; ?>

<?php if ($error): ?>
<div class="alert alert-danger rounded-4 text-center py-4 mb-4 shadow">
    <strong><?= htmlspecialchars($error) ?></strong>
</div>
<?php endif; ?>

<div class="container py-4">
    <div class="mb-4">
        <a href="properties.php" class="btn btn-secondary btn-lg rounded-pill">Properties List</a>
        <a href="dashboard.php" class="btn btn-dark btn-lg rounded-pill ms-2">Dashboard</a>
    </div>

    <h2 class="text-success fw-bold mb-4">Edit Property: <?= htmlspecialchars($prop['name']) ?></h2>

    <!-- Total Bill -->
    <div class="card border-success shadow-sm mb-4">
        <div class="card-body bg-light text-center py-4">
            <h4 class="text-success mb-3">Tenant's Total Monthly Bill</h4>
            <h1 class="display-5 fw-bold text-primary">KSh <?= number_format($total_bill) ?></h1>
            <p class="mb-0">
                = Rent KSh <?= number_format($prop['rent_amount']) ?>
                + Water KSh <?= number_format($current_water) ?>
                + Garbage KSh <?= number_format($garbage_fee) ?>
            </p>
        </div>
    </div>

    <!-- Update Rent -->
    <div class="card shadow-lg rounded-4 mb-4">
        <div class="card-body p-5">
            <form method="POST">
                <input type="hidden" name="update_property" value="1">
                <div class="row g-4">
                    <div class="col-md-6">
                        <label class="form-label fw-bold">Property Name *</label>
                        <input type="text" name="name" class="form-control form-control-lg" value="<?= htmlspecialchars($prop['name']) ?>" required>
                    </div>
                    <div class="col-md-6">
                        <label class="form-label fw-bold">Status</label>
                        <select name="status" class="form-select form-control-lg">
                            <option value="available" <?= $prop['status']=='available'?'selected':'' ?>>Available</option>
                            <option value="occupied" <?= $prop['status']=='occupied'?'selected':'' ?>>Occupied</option>
                            <option value="maintenance" <?= $prop['status']=='maintenance'?'selected':'' ?>>Under Maintenance</option>
                        </select>
                    </div>
                    <div class="col-12">
                        <label class="form-label fw-bold">Address (Optional)</label>
                        <input type="text" name="address" class="form-control" value="<?= htmlspecialchars($prop['address'] ?? '') ?>">
                    </div>
                    <div class="col-md-6">
                        <label class="form-label fw-bold">Monthly Rent (KSh) *</label>
                        <input type="number" name="rent_amount" class="form-control form-control-lg" value="<?= $prop['rent_amount'] ?>" required>
                    </div>
                    <div class="col-md-6 d-flex align-items-end">
                        <button type="submit" class="btn btn-success btn-lg w-100 rounded-pill shadow-lg">
                            UPDATE PROPERTY
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <!-- Water Bill Per Month -->
    <div class="card shadow-lg rounded-4">
        <div class="card-body p-5">
            <h4 class="text-primary fw-bold mb-4">Water Bill – Per Month (Affects Only That Month)</h4>
            <form method="POST" class="row g-4 mb-4">
                <input type="hidden" name="save_water_bill" value="1">
                <div class="col-md-5">
                    <label class="form-label fw-bold">Month</label>
                    <input type="month" name="bill_month" class="form-control form-control-lg" value="<?= date('Y-m') ?>" required>
                </div>
                <div class="col-md-4">
                    <label class="form-label fw-bold">Water Units</label>
                    <input type="number" step="0.01" name="water_units" class="form-control form-control-lg" placeholder="e.g. 18.5" min="0" required>
                </div>
                <div class="col-md-3 d-flex align-items-end">
                    <button type="submit" class="btn btn-primary btn-lg w-100 rounded-pill">Save</button>
                </div>
            </form>

            <div class="table-responsive">
                <table class="table table-hover">
                    <thead class="table-light">
                        <tr>
                            <th>Month</th>
                            <th class="text-end">Units</th>
                            <th class="text-end">Water Bill</th>
                            <th class="text-end">Total Bill</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php
                        try {
                            $stmt = $pdo->prepare("SELECT bill_month, water_units, water_bill FROM monthly_bills WHERE property_id = ? ORDER BY bill_month DESC LIMIT 12");
                            $stmt->execute([$id]);
                            foreach ($stmt->fetchAll() as $r):
                                $tot = $prop['rent_amount'] + $r['water_bill'] + $garbage_fee;
                        ?>
                        <tr>
                            <td><strong><?= date('M Y', strtotime($r['bill_month'].'-01')) ?></strong></td>
                            <td class="text-end"><?= number_format($r['water_units'], 2) ?></td>
                            <td class="text-end">KSh <?= number_format($r['water_bill']) ?></td>
                            <td class="text-end text-success fw-bold">KSh <?= number_format($tot) ?></td>
                        </tr>
                        <?php endforeach; ?>
                        <?php if ($stmt->rowCount() == 0): ?>
                        <tr><td colspan="4" class="text-center text-muted">No water bills recorded yet</td></tr>
                        <?php endif; ?>
                        <?php } catch (Exception $e) { ?>
                        <tr><td colspan="4" class="text-center text-muted">No water bills recorded yet</td></tr>
                        <?php } ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
</div>

<?php include '../includes/footer.php'; ?>