<?php
require_once '../includes/functions.php';
if (!isLoggedIn()) redirect('../auth/login.php');

$id = (int)($_GET['id'] ?? 0);
if (!$id) {
    $_SESSION['error'] = "Invalid tenant ID.";
    redirect('tenants.php');
}

// Fetch tenant
$stmt = $pdo->prepare("SELECT * FROM tenants WHERE id = ?");
$stmt->execute([$id]);
$tenant = $stmt->fetch();
if (!$tenant) {
    $_SESSION['error'] = "Tenant not found!";
    redirect('tenants.php');
}

// Fetch all properties
$properties = $pdo->query("SELECT id, name FROM properties ORDER BY name")->fetchAll();

$success = $error = '';

if ($_POST) {
    $name         = trim($_POST['name'] ?? '');
    $id_number    = trim($_POST['id_number'] ?? '');
    $email        = filter_var($_POST['email'] ?? '', FILTER_VALIDATE_EMAIL) ?: '';
    $phone        = trim($_POST['phone'] ?? '');
    $next_of_kin  = trim($_POST['next_of_kin'] ?? '');
    $kin_phone    = trim($_POST['kin_phone'] ?? '');
    $property_id  = !empty($_POST['property_id']) ? (int)$_POST['property_id'] : null;
    $move_in_date = $_POST['move_in_date'] ?? '';
    $status       = in_array($_POST['status'], ['active', 'inactive']) ? $_POST['status'] : 'active';

    if (!$name || !$id_number || !$phone || !$move_in_date) {
        $error = "Please fill all required fields!";
    } else {
        try {
            $pdo->beginTransaction();

            // === 1. UPDATE TENANT ===
            $stmt = $pdo->prepare("
                UPDATE tenants SET
                name = ?, id_number = ?, email = ?, phone = ?,
                next_of_kin = ?, kin_phone = ?, property_id = ?,
                move_in_date = ?, status = ?
                WHERE id = ?
            ");
            $stmt->execute([$name, $id_number, $email, $phone, $next_of_kin, $kin_phone, $property_id, $move_in_date, $status, $id]);

            // === 2. HANDLE PROPERTY CHANGE ===
            if ($tenant['property_id'] != $property_id) {
                // Free old property
                if ($tenant['property_id']) {
                    $pdo->prepare("UPDATE properties SET status = 'available' WHERE id = ? AND id NOT IN (SELECT property_id FROM tenants WHERE property_id IS NOT NULL AND id != ?)")
                        ->execute([$tenant['property_id'], $id]);
                }
                // Occupy new property
                if ($property_id) {
                    $pdo->prepare("UPDATE properties SET status = 'occupied' WHERE id = ?")->execute([$property_id]);
                }
            }

            // === 3. IF move_in_date OR status CHANGED → RECALCULATE BILLS & ARREARS ===
            $recalculate_needed = (
                $tenant['move_in_date'] != $move_in_date || 
                $tenant['status'] != $status
            );

            if ($recalculate_needed) {
                // Delete all future/unpaid bills so they can be regenerated correctly
                $pdo->prepare("
                    DELETE FROM payments 
                    WHERE tenant_id = ? 
                      AND status = 'pending' 
                      AND due_date >= ?
                ")->execute([$id, date('Y-m-01')]);

                // Optional: Mark old unpaid bills as 'recalculate needed' or just let generate_rent.php fix them
                // For now, we let the next generate_rent.php run fix everything perfectly
            }

            $pdo->commit();
            $_SESSION['success'] = "Tenant <strong>" . htmlspecialchars($name) . "</strong> updated successfully!<br>
                                   <small>Billing & arrears will be recalculated automatically on next bill generation.</small>";
            redirect('tenants.php');

        } catch (Exception $e) {
            $pdo->rollBack();
            $error = "Update failed: " . $e->getMessage();
        }
    }
}

include '../includes/header.php';
?>

<div class="container py-5">

    <div class="d-flex flex-wrap gap-3 mb-5">
        <a href="tenants.php" class="btn btn-secondary btn-lg rounded-pill px-4 shadow">All Tenants</a>
        <a href="dashboard.php" class="btn btn-dark btn-lg rounded-pill px-4 shadow">Dashboard</a>
    </div>

    <div class="text-center mb-5">
        <h1 class="display-5 fw-bold text-success">Edit Tenant</h1>
        <p class="lead text-muted">Any changes to <strong>Move-in Date</strong> or <strong>Status</strong> will automatically update billing & arrears</p>
    </div>

    <?php if ($error): ?>
        <div class="alert alert-danger rounded-4 shadow-lg text-center py-4">
            <strong><?= htmlspecialchars($error) ?></strong>
        </div>
    <?php endif; ?>

    <div class="card border-0 shadow-lg rounded-4 overflow-hidden">
        <div class="card-header bg-gradient bg-success text-white text-center py-4">
            <h3 class="mb-0 fw-bold">
                Editing: <span class="text-warning"><?= htmlspecialchars($tenant['name']) ?></span>
            </h3>
        </div>
        <div class="card-body p-5 bg-light">

            <!-- LIVE ARREARS PREVIEW -->
            <?php
            $arrears = $pdo->prepare("
                SELECT COALESCE(SUM(amount - amount_paid), 0) as total 
                FROM payments 
                WHERE tenant_id = ? AND status IN ('pending', 'partial')
            ");
            $arrears->execute([$id]);
            $total_arrears = $arrears->fetchColumn();
            ?>
            <?php if ($total_arrears > 0): ?>
            <div class="alert alert-warning text-center py-4 rounded-4 mb-4 shadow">
                <h4>Current Arrears: <strong class="text-danger">KSh <?= number_format($total_arrears) ?></strong></h4>
                <small>Will be recalculated after save if move-in date changes</small>
            </div>
            <?php endif; ?>

            <form method="POST" novalidate>
                <div class="row g-4">

                    <div class="col-lg-6">
                        <label class="form-label fw-bold">Full Name <span class="text-danger">*</span></label>
                        <input type="text" name="name" class="form-control form-control-lg rounded-pill" 
                               value="<?= htmlspecialchars($tenant['name']) ?>" required>
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold">National ID <span class="text-danger">*</span></label>
                        <input type="text" name="id_number" class="form-control form-control-lg rounded-pill" 
                               value="<?= htmlspecialchars($tenant['id_number'] ?? '') ?>" required>
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold">Email (Optional)</label>
                        <input type="email" name="email" class="form-control form-control-lg rounded-pill" 
                               value="<?= htmlspecialchars($tenant['email'] ?? '') ?>">
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold">Phone Number <span class="text-danger">*</span></label>
                        <input type="text" name="phone" class="form-control form-control-lg rounded-pill" 
                               value="<?= htmlspecialchars($tenant['phone'] ?? '') ?>" required>
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold">Next of Kin</label>
                        <input type="text" name="next_of_kin" class="form-control form-control-lg rounded-pill" 
                               value="<?= htmlspecialchars($tenant['next_of_kin'] ?? '') ?>">
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold">Kin Phone</label>
                        <input type="text" name="kin_phone" class="form-control form-control-lg rounded-pill" 
                               value="<?= htmlspecialchars($tenant['kin_phone'] ?? '') ?>">
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold text-primary">Property / Unit</label>
                        <select name="property_id" class="form-select form-select-lg rounded-pill">
                            <option value="">-- No Property (Vacant) --</option>
                            <?php foreach ($properties as $p): ?>
                                <option value="<?= $p['id'] ?>" <?= ($p['id'] == $tenant['property_id']) ? 'selected' : '' ?>>
                                    <?= htmlspecialchars($p['name']) ?>
                                </option>
                            <?php endforeach; ?>
                        </select>
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold text-primary">Move-in Date <span class="text-danger">*</span></label>
                        <input type="date" name="move_in_date" class="form-control form-control-lg rounded-pill" 
                               value="<?= $tenant['move_in_date'] ?>" required>
                        <small class="text-success">Changing this recalculates all bills & arrears</small>
                    </div>

                    <div class="col-lg-6">
                        <label class="form-label fw-bold">Status</label>
                        <select name="status" class="form-select form-select-lg rounded-pill">
                            <option value="active" <?= $tenant['status'] == 'active' ? 'selected' : '' ?>>Active (Billed Monthly)</option>
                            <option value="inactive" <?= $tenant['status'] == 'inactive' ? 'selected' : '' ?>>Inactive (No Billing)</option>
                        </select>
                    </div>

                    <div class="col-12 text-center mt-5">
                        <button type="submit" class="btn btn-success btn-lg px-5 py-3 rounded-pill shadow-lg fw-bold">
                            UPDATE TENANT & RECALCULATE BILLS
                        </button>
                    </div>
                </div>
            </form>
        </div>
    </div>

    <div class="text-center mt-5 py-5 text-muted">
        <p class="mb-0 fw-bold">Rene.C.Agency • Smart • Accurate • Fully Automated</p>
        <small>Made with pride in Kenya • <?= date('Y') ?></small>
    </div>
</div>

<?php include '../includes/footer.php'; ?>