<?php
// admin/generate_rent.php
// FINAL VERSION — 100% ACCURATE BILLING + FULL MULTI-ADMIN SECURITY

require_once '../includes/functions.php';

// Security
if (!isLoggedIn() || ($_SESSION['role'] ?? '') !== 'admin') {
    $_SESSION['error'] = "Access denied.";
    redirect('../auth/login.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

$current_month = date('Y-m-01'); // e.g. 2026-01-01
$generated = 0;
$message = '';

try {
    $pdo->beginTransaction();

    // Global garbage fee
    $garbage_fee = 500;
    try {
        $garbage_fee = (int)$pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn();
    } catch (Exception $e) {}

    // Get active tenants with property details — FILTERED BY ADMIN
    if ($IS_SUPER_ADMIN) {
        $tenants = $pdo->query("
            SELECT
                t.id AS tenant_id,
                t.name,
                t.move_in_date,
                p.rent_amount,
                p.water_bill
            FROM tenants t
            JOIN properties p ON t.property_id = p.id
            WHERE t.status = 'active'
            ORDER BY t.name
        ")->fetchAll();
    } else {
        $tenants = $pdo->prepare("
            SELECT
                t.id AS tenant_id,
                t.name,
                t.move_in_date,
                p.rent_amount,
                p.water_bill
            FROM tenants t
            JOIN properties p ON t.property_id = p.id
            WHERE t.status = 'active' AND p.admin_id = ?
            ORDER BY t.name
        ");
        $tenants->execute([$admin_id]);
        $tenants = $tenants->fetchAll();
    }

    foreach ($tenants as $t) {
        $tenant_id = $t['tenant_id'];
        $move_in_date = $t['move_in_date'];

        if (!$move_in_date || $move_in_date === '0000-00-00') {
            continue;
        }

        // Determine first billable month
        $move_in = new DateTime($move_in_date);
        $first_bill_month = ($move_in->format('d') === '01')
            ? $move_in->format('Y-m-01')
            : (clone $move_in)->modify('first day of next month')->format('Y-m-01');

        // Skip if current month is before first billable month
        if ($current_month < $first_bill_month) {
            continue;
        }

        // Skip if already billed this month
        $check = $pdo->prepare("SELECT id FROM payments WHERE tenant_id = ? AND due_date = ?");
        $check->execute([$tenant_id, $current_month]);
        if ($check->fetch()) {
            continue;
        }

        // Monthly base rent
        $monthly_rent = ($t['rent_amount'] ?? 0) + ($t['water_bill'] ?? 0) + $garbage_fee;

        // Total arrears from billable months only
        $arrears_query = $pdo->prepare("
            SELECT COALESCE(SUM(amount - amount_paid), 0)
            FROM payments
            WHERE tenant_id = ?
              AND status IN ('pending', 'partial')
              AND due_date >= ?
        ");
        $arrears_query->execute([$tenant_id, $first_bill_month]);
        $total_arrears = (float)$arrears_query->fetchColumn();

        // Total available credit
        $credit_query = $pdo->prepare("SELECT COALESCE(SUM(credit_carried), 0) FROM payments WHERE tenant_id = ?");
        $credit_query->execute([$tenant_id]);
        $total_credit = (float)$credit_query->fetchColumn();

        // Final amount due this month
        $final_amount = $monthly_rent + $total_arrears - $total_credit;

        $notes = "Monthly rent for " . date('F Y', strtotime($current_month));
        if ($total_arrears > 0) $notes .= " | +Arrears KSh " . number_format($total_arrears);
        if ($total_credit > 0) $notes .= " | -Credit KSh " . number_format($total_credit);

        if ($final_amount <= 0) {
            // Credit covers this month
            $insert = $pdo->prepare("
                INSERT INTO payments
                (tenant_id, amount, due_date, status, notes, amount_paid, credit_carried)
                VALUES (?, ?, ?, 'paid', ?, ?, ?)
            ");
            $remaining_credit = max(0, $total_credit - $monthly_rent - $total_arrears);
            $insert->execute([
                $tenant_id,
                $monthly_rent,
                $current_month,
                "Auto-paid using credit | " . $notes,
                $monthly_rent, // amount_paid = full monthly
                $remaining_credit
            ]);
        } else {
            // Normal pending bill
            $insert = $pdo->prepare("
                INSERT INTO payments
                (tenant_id, amount, due_date, status, notes)
                VALUES (?, ?, ?, 'pending', ?)
            ");
            $insert->execute([$tenant_id, $final_amount, $current_month, $notes]);
        }

        $generated++;
    }

    $pdo->commit();

    $message = "<strong>$generated tenants</strong> billed successfully for <strong>" . date('F Y') . "</strong><br>
                <small class='text-success'>• Accurate from move-in date<br>• Arrears & credit applied automatically</small>";

} catch (Exception $e) {
    $pdo->rollBack();
    $message = "Error generating bills: " . htmlspecialchars($e->getMessage());
}

$_SESSION['success'] = $message;
redirect('dashboard.php');
?>