<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    redirect('../auth/login.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

// GLOBAL GARBAGE FEE
$garbage_fee = 500;
try {
    $garbage_fee = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
} catch (Exception $e) {}

// ==================== RECORD PAYMENT (FULLY WORKING + SMART CREDIT) ====================
if (isset($_GET['paid']) && $_SERVER['REQUEST_METHOD'] === 'POST') {
    $payment_id = (int)$_GET['paid'];
    $pay_amount = (float)($_POST['pay_amount'] ?? 0);
    if ($pay_amount <= 0) {
        $_SESSION['error'] = "Please enter a valid amount!";
        redirect('payments.php');
    }

    $pdo->beginTransaction();
    try {
        // Get tenant and verify belongs to this admin
        if ($IS_SUPER_ADMIN) {
            $stmt = $pdo->prepare("SELECT p.tenant_id FROM payments p WHERE p.id = ?");
        } else {
            $stmt = $pdo->prepare("SELECT p.tenant_id FROM payments p JOIN tenants t ON p.tenant_id = t.id JOIN properties pr ON t.property_id = pr.id WHERE p.id = ? AND pr.admin_id = ?");
        }
        $params = $IS_SUPER_ADMIN ? [$payment_id] : [$payment_id, $admin_id];
        $stmt->execute($params);
        $tenant_id = $stmt->fetchColumn();

        if (!$tenant_id) {
            throw new Exception("Invalid payment record or access denied");
        }

        // Get ALL bills for this tenant, oldest first
        $bills = $pdo->prepare("
            SELECT id, amount, amount_paid, COALESCE(credit_carried, 0) AS credit_carried, due_date
            FROM payments
            WHERE tenant_id = ?
            ORDER BY due_date ASC
        ");
        $bills->execute([$tenant_id]);
        $bills = $bills->fetchAll();

        $remaining = $pay_amount;
        $applied_messages = [];

        foreach ($bills as $bill) {
            if ($remaining <= 0) break;
            $owing = $bill['amount'] - $bill['amount_paid'];
            if ($owing <= 0) continue;
            $apply_here = min($remaining, $owing);

            $new_paid = $bill['amount_paid'] + $apply_here;
            $new_status = ($new_paid >= $bill['amount']) ? 'paid' : 'partial';

            $update = $pdo->prepare("
                UPDATE payments
                SET amount_paid = ?,
                    paid_date = NOW(),
                    status = ?
                WHERE id = ?
            ");
            $update->execute([$new_paid, $new_status, $bill['id']]);

            $applied_messages[] = "KSh " . number_format($apply_here) . " → " . date('M Y', strtotime($bill['due_date']));
            $remaining -= $apply_here;
        }

        // Leftover → credit on latest bill
        if ($remaining > 0) {
            $latest = $pdo->prepare("SELECT id FROM payments WHERE tenant_id = ? ORDER BY due_date DESC LIMIT 1");
            $latest->execute([$tenant_id]);
            $latest_id = $latest->fetchColumn();

            $pdo->prepare("UPDATE payments SET credit_carried = COALESCE(credit_carried, 0) + ? WHERE id = ?")
                ->execute([$remaining, $latest_id]);

            $applied_messages[] = "KSh " . number_format($remaining) . " saved as credit";
        }

        $pdo->commit();
        $_SESSION['success'] = "Payment of KSh " . number_format($pay_amount) . " recorded! " . implode(" • ", $applied_messages);
    } catch (Exception $e) {
        $pdo->rollBack();
        $_SESSION['error'] = "Payment failed. Please try again.";
    }
    redirect('payments.php');
}

include '../includes/header.php';
?>

<div class="container py-4">
    <div class="d-flex justify-content-between align-items-center mb-4">
        <a href="dashboard.php" class="btn btn-dark btn-lg rounded-pill px-5">Dashboard</a>
        <a href="add_payment.php" class="btn btn-success btn-lg rounded-pill px-5">Generate Bills</a>
    </div>

    <h2 class="text-center text-success mb-5 fw-bold display-5">
        Record Payments & Auto-Apply Credit
    </h2>

    <!-- ALERTS -->
    <?php if (isset($_SESSION['success'])): ?>
        <div class="alert alert-success alert-dismissible fade show text-center fs-5 fw-bold">
            <?= $_SESSION['success'] ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['success']); ?>
    <?php endif; ?>
    <?php if (isset($_SESSION['error'])): ?>
        <div class="alert alert-danger alert-dismissible fade show text-center fs-5">
            <?= $_SESSION['error'] ?>
            <button type="button" class="btn-close" data-bs-dismiss="alert"></button>
        </div>
        <?php unset($_SESSION['error']); ?>
    <?php endif; ?>

    <!-- FILTERS -->
    <div class="card shadow-sm mb-4 border-0">
        <div class="card-body">
            <div class="row g-3">
                <div class="col-lg-4">
                    <input type="text" id="searchInput" class="form-control form-control-lg rounded-pill" placeholder="Search tenant or unit...">
                </div>
                <div class="col-lg-3">
                    <select id="monthFilter" class="form-select form-select-lg rounded-pill">
                        <option value="">All Months</option>
                        <?php
                        if ($IS_SUPER_ADMIN) {
                            $months = $pdo->query("SELECT DISTINCT DATE_FORMAT(due_date, '%Y-%m') AS ym, DATE_FORMAT(due_date, '%M %Y') AS name FROM payments ORDER BY due_date DESC")->fetchAll();
                        } else {
                            $months = $pdo->prepare("SELECT DISTINCT DATE_FORMAT(p.due_date, '%Y-%m') AS ym, DATE_FORMAT(p.due_date, '%M %Y') AS name FROM payments p JOIN tenants t ON p.tenant_id = t.id JOIN properties pr ON t.property_id = pr.id WHERE pr.admin_id = ? ORDER BY p.due_date DESC");
                            $months->execute([$admin_id]);
                            $months = $months->fetchAll();
                        }
                        foreach ($months as $m): ?>
                            <option value="<?= $m['ym'] ?>"><?= $m['name'] ?></option>
                        <?php endforeach; ?>
                    </select>
                </div>
                <div class="col-lg-3">
                    <select id="statusFilter" class="form-select form-select-lg rounded-pill">
                        <option value="">All Status</option>
                        <option value="owes">Owes</option>
                        <option value="paid">Paid</option>
                        <option value="credit">Has Credit</option>
                        <option value="partial">Partial</option>
                    </select>
                </div>
                <div class="col-lg-2">
                    <button onclick="resetFilters()" class="btn btn-outline-danger btn-lg w-100 rounded-pill">Reset</button>
                </div>
            </div>
        </div>
    </div>

    <!-- TENANTS ACCORDION — FILTERED BY ADMIN -->
    <div class="accordion" id="tenantAccordion">
        <?php
        // Get tenants — FILTERED BY ADMIN
        if ($IS_SUPER_ADMIN) {
            $tenants = $pdo->query("
                SELECT t.id, t.name, t.move_in_date,
                       COALESCE(p.rent_amount, 0) + COALESCE(p.water_bill, 0) + $garbage_fee AS monthly_total,
                       COALESCE(pr.name, 'No Unit') AS unit
                FROM tenants t
                LEFT JOIN properties p ON t.property_id = p.id
                LEFT JOIN properties pr ON t.property_id = pr.id
                WHERE t.status = 'active'
                ORDER BY t.name
            ");
        } else {
            $tenants = $pdo->prepare("
                SELECT t.id, t.name, t.move_in_date,
                       COALESCE(p.rent_amount, 0) + COALESCE(p.water_bill, 0) + $garbage_fee AS monthly_total,
                       COALESCE(pr.name, 'No Unit') AS unit
                FROM tenants t
                LEFT JOIN properties p ON t.property_id = p.id
                LEFT JOIN properties pr ON t.property_id = pr.id
                WHERE t.status = 'active' AND p.admin_id = ?
                ORDER BY t.name
            ");
            $tenants->execute([$admin_id]);
        }

        $counter = 0;
        while ($tenant = $tenants->fetch()):
            $counter++;
            $tenant_id = $tenant['id'];
            $name = htmlspecialchars($tenant['name']);
            $unit = htmlspecialchars($tenant['unit']);
            $monthly = $tenant['monthly_total'];
            $move_in = new DateTime($tenant['move_in_date']);
            $first_bill_month = $move_in->format('Y-m-01');
            $start = new DateTime($first_bill_month);
            $end = new DateTime('first day of next month');
            $period = new DatePeriod($start, new DateInterval('P1M'), $end);
        ?>
        <div class="accordion-item tenant-item shadow-sm mb-3 rounded"
             data-tenant="<?= strtolower($name . ' ' . $unit) ?>">
            <h2 class="accordion-header">
                <button class="accordion-button <?= $counter === 1 ? '' : 'collapsed' ?> fw-bold fs-5"
                        type="button" data-bs-toggle="collapse" data-bs-target="#tenant<?= $tenant_id ?>">
                    <?= $name ?> — <?= $unit ?>
                    <span class="ms-auto badge bg-primary">Click for bills</span>
                </button>
            </h2>
            <div id="tenant<?= $tenant_id ?>" class="accordion-collapse collapse <?= $counter === 1 ? 'show' : '' ?>">
                <div class="accordion-body p-0">
                    <div class="table-responsive">
                        <table class="table table-hover mb-0">
                            <thead class="table-success">
                                <tr>
                                    <th>Month</th>
                                    <th class="text-end">Billed</th>
                                    <th class="text-end">Paid</th>
                                    <th class="text-end">Credit Used</th>
                                    <th class="text-center fw-bold">Balance</th>
                                    <th>Status</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($period as $dt):
                                    $due_date = $dt->format('Y-m-01');
                                    $month_ym = $dt->format('Y-m');
                                    $display = $dt->format('M Y');

                                    $stmt = $pdo->prepare("SELECT * FROM payments WHERE tenant_id = ? AND due_date = ?");
                                    $stmt->execute([$tenant_id, $due_date]);
                                    $p = $stmt->fetch();

                                    if (!$p) {
                                        $days_in_month = $dt->format('t');
                                        $days_lived = $days_in_month - ($move_in->format('j') - 1);
                                        $prorated = ($dt->format('Y-m') === $first_bill_month)
                                            ? round($monthly * max(1, $days_lived) / $days_in_month)
                                            : $monthly;
                                        $ins = $pdo->prepare("INSERT INTO payments (tenant_id, amount, due_date, status) VALUES (?, ?, ?, 'pending')");
                                        $ins->execute([$tenant_id, $prorated, $due_date]);
                                        $p = [
                                            'id' => $pdo->lastInsertId(),
                                            'amount' => $prorated,
                                            'amount_paid' => 0,
                                            'credit_carried' => 0,
                                            'status' => 'pending'
                                        ];
                                    }

                                    $credit_used = min($p['credit_carried'] ?? 0, max(0, $p['amount'] - $p['amount_paid']));
                                    $balance = $p['amount'] - $p['amount_paid'] - $credit_used;
                                    $remaining_credit = ($p['credit_carried'] ?? 0) - $credit_used;

                                    $status = $balance > 0 ? 'owes' : ($remaining_credit > 0 ? 'credit' : 'paid');
                                    if ($p['amount_paid'] > 0 && $p['amount_paid'] < $p['amount']) $status = 'partial';
                                ?>
                                <tr class="payment-row" data-month="<?= $month_ym ?>" data-status="<?= $status ?>">
                                    <td><strong><?= $display ?></strong></td>
                                    <td class="text-end">KSh <?= number_format($p['amount']) ?></td>
                                    <td class="text-end text-success fw-bold">KSh <?= number_format($p['amount_paid']) ?></td>
                                    <td class="text-end text-info"><?= $credit_used > 0 ? 'KSh '.number_format($credit_used) : '—' ?></td>
                                    <td class="text-center fw-bold <?= $balance > 0 ? 'text-danger' : 'text-success' ?>">
                                        <?= $balance > 0 ? 'OWES KSh '.number_format($balance) : 'PAID' ?>
                                        <?php if ($remaining_credit > 0): ?>
                                            <br><small class="text-success">(Credit: KSh <?= number_format($remaining_credit) ?>)</small>
                                        <?php endif; ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?= $balance > 0 ? 'danger' : ($remaining_credit > 0 ? 'success' : 'secondary') ?> rounded-pill px-3">
                                            <?= $balance > 0 ? 'OWES' : ($remaining_credit > 0 ? 'CREDIT' : 'PAID') ?>
                                        </span>
                                    </td>
                                    <td>
                                        <button class="btn btn-primary btn-sm w-100" data-bs-toggle="modal" data-bs-target="#payModal<?= $p['id'] ?>">
                                            Pay Now
                                        </button>
                                    </td>
                                </tr>

                                <!-- PAYMENT MODAL -->
                                <div class="modal fade" id="payModal<?= $p['id'] ?>" tabindex="-1">
                                    <div class="modal-dialog modal-md">
                                        <form method="POST" action="payments.php?paid=<?= $p['id'] ?>">
                                            <div class="modal-content rounded-4 shadow-lg">
                                                <div class="modal-header bg-primary text-white">
                                                    <h5 class="modal-title fw-bold">Record Payment - <?= $name ?></h5>
                                                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="modal"></button>
                                                </div>
                                                <div class="modal-body text-center py-5">
                                                    <h4 class="text-primary"><?= $display ?></h4>
                                                    <hr>
                                                    <p><strong>Billed:</strong> KSh <?= number_format($p['amount']) ?></p>
                                                    <p><strong>Already Paid:</strong> KSh <?= number_format($p['amount_paid']) ?></p>
                                                    <p><strong>Credit Available:</strong> KSh <?= number_format($p['credit_carried'] ?? 0) ?></p>
                                                    <p class="fs-3 text-danger fw-bold mt-3">
                                                        OWING: KSh <?= number_format(max(0, $balance)) ?>
                                                    </p>
                                                    <hr>
                                                    <label class="form-label fs-5 fw-bold">Amount Paying Today</label>
                                                    <input type="number" name="pay_amount" class="form-control form-control-lg text-center fw-bold"
                                                           placeholder="Enter amount" min="1" step="1" required autofocus>
                                                    <small class="text-success mt-2 d-block">
                                                        Pay more than owed → becomes credit for future months!
                                                    </small>
                                                </div>
                                                <div class="modal-footer justify-content-center">
                                                    <button type="submit" class="btn btn-success btn-lg px-5 fw-bold">Record Payment</button>
                                                </div>
                                            </div>
                                        </form>
                                    </div>
                                </div>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        <?php endwhile; ?>
    </div>
</div>

<script>
function applyFilters() {
    const search = document.getElementById('searchInput').value.toLowerCase().trim();
    const month = document.getElementById('monthFilter').value;
    const status = document.getElementById('statusFilter').value;

    document.querySelectorAll('.tenant-item').forEach(item => {
        const text = item.dataset.tenant;
        const matchesSearch = !search || text.includes(search);
        let hasMatch = false;

        item.querySelectorAll('.payment-row').forEach(row => {
            const rowMonth = row.dataset.month;
            const rowStatus = row.dataset.status;
            const matchesMonth = !month || rowMonth.startsWith(month);
            const matchesStatus = !status || rowStatus === status;

            if (matchesSearch && matchesMonth && matchesStatus) {
                row.style.display = '';
                hasMatch = true;
            } else {
                row.style.display = 'none';
            }
        });

        item.style.display = (matchesSearch && hasMatch) ? '' : 'none';
    });
}

function resetFilters() {
    document.getElementById('searchInput').value = '';
    document.getElementById('monthFilter').value = '';
    document.getElementById('statusFilter').value = '';
    applyFilters();
}

document.getElementById('searchInput').addEventListener('input', applyFilters);
document.getElementById('monthFilter').addEventListener('change', applyFilters);
document.getElementById('statusFilter').addEventListener('change', applyFilters);
applyFilters();
</script>

<?php include '../includes/footer.php'; ?>