<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn()) {
    redirect('../auth/login.php');
}
if (!isset($_SESSION['role']) || $_SESSION['role'] !== 'admin') {
    redirect('../auth/login.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

// GLOBAL GARBAGE FEE
$garbage_fee = 500;
try {
    $garbage_fee = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
} catch (Exception $e) {}

// AUTO UPDATE PROPERTY STATUS — FILTERED BY ADMIN
$sql_update = $IS_SUPER_ADMIN 
    ? "UPDATE properties p LEFT JOIN tenants t ON p.id = t.property_id AND t.status='active' SET p.status = IF(t.id IS NULL, 'available', 'occupied')"
    : "UPDATE properties p LEFT JOIN tenants t ON p.id = t.property_id AND t.status='active' SET p.status = IF(t.id IS NULL, 'available', 'occupied') WHERE p.admin_id = ?";

$stmt_update = $IS_SUPER_ADMIN ? $pdo->query($sql_update) : $pdo->prepare($sql_update);
if (!$IS_SUPER_ADMIN) $stmt_update->execute([$admin_id]);

include '../includes/header.php';
?>

<div class="container py-4 py-lg-5">
    <!-- Header + Controls + ADD BUTTON -->
    <div class="row align-items-center mb-4">
        <div class="col-md-4">
            <a href="dashboard.php" class="btn btn-outline-dark rounded-pill px-4">Back</a>
        </div>
        <div class="col-md-4 text-center">
            <h2 class="text-success fw-bold mb-0">Properties Management</h2>
        </div>
        <div class="col-md-4 text-end">
            <!-- ADD NEW PROPERTY BUTTON -->
            <a href="add_property.php" class="btn btn-success rounded-pill px-5 shadow">
                <i class="bi bi-plus-circle me-2"></i>Add New Property
            </a>
        </div>
    </div>

    <div class="row align-items-center mb-4">
        <div class="col-md-8 offset-md-2">
            <div class="row g-3">
                <div class="col">
                    <input type="text" id="searchInput" class="form-control rounded-pill" placeholder="Search unit or tenant...">
                </div>
                <div class="col-auto">
                    <select id="filterStatus" class="form-select rounded-pill">
                        <option value="all">All Units</option>
                        <option value="occupied">Occupied</option>
                        <option value="vacant">Vacant</option>
                        <option value="arrears">In Arrears</option>
                        <option value="credit">In Credit</option>
                    </select>
                </div>
            </div>
        </div>
    </div>

    <h4 class="text-center text-success mb-5">Live Status & Auto-Billing</h4>

    <!-- Responsive Card Grid -->
    <div class="row row-cols-2 row-cols-sm-3 row-cols-md-4 row-cols-lg-5 row-cols-xl-6 g-4" id="propertiesGrid">
        <?php
        // Get properties with tenant data — FILTERED BY ADMIN
        $sql = $IS_SUPER_ADMIN 
            ? "SELECT
                p.id AS property_id,
                p.name AS property_name,
                p.rent_amount,
                p.water_bill,
                t.id AS tenant_id,
                t.name AS tenant_name,
                t.move_in_date
               FROM properties p
               LEFT JOIN tenants t ON p.id = t.property_id AND t.status = 'active'
               ORDER BY p.name ASC"
            : "SELECT
                p.id AS property_id,
                p.name AS property_name,
                p.rent_amount,
                p.water_bill,
                t.id AS tenant_id,
                t.name AS tenant_name,
                t.move_in_date
               FROM properties p
               LEFT JOIN tenants t ON p.id = t.property_id AND t.status = 'active'
               WHERE p.admin_id = ?
               ORDER BY p.name ASC";

        $stmt = $IS_SUPER_ADMIN ? $pdo->query($sql) : $pdo->prepare($sql);
        if (!$IS_SUPER_ADMIN) $stmt->execute([$admin_id]);

        while ($unit = $stmt->fetch()):
            $occupied = !empty($unit['tenant_id']);
            $property_id = $unit['property_id'];
            $property_name = htmlspecialchars($unit['property_name']);

            if (!$occupied) {
                $border = "border-success";
                $header_bg = "bg-success";
                $header_text = "text-white";
                $status = "VACANT";
                $status_class = "vacant";
                $search_text = strtolower($property_name);
                $monthly = $total_billed = $total_paid = $arrears = $credit = 0;
                $net_arrears = 0;
            } else {
                $monthly = ($unit['rent_amount'] ?? 0) + ($unit['water_bill'] ?? 0) + $garbage_fee;
                $move_in = new DateTime($unit['move_in_date']);
                $first_bill = ($move_in->format('d') === '01')
                    ? $move_in->format('Y-m-01')
                    : (clone $move_in)->modify('first day of next month')->format('Y-m-01');
                $start = new DateTime($first_bill);
                $end = new DateTime('first day of next month');
                $period = new DatePeriod($start, new DateInterval('P1M'), $end);

                $total_billed = 0;
                $total_paid = 0;
                $total_credit = 0;

                foreach ($period as $dt) {
                    $due_date = $dt->format('Y-m-01');
                    $check = $pdo->prepare("SELECT * FROM payments WHERE tenant_id = ? AND due_date = ?");
                    $check->execute([$unit['tenant_id'], $due_date]);
                    $p = $check->fetch();

                    if (!$p) {
                        $days_in_month = $dt->format('t');
                        $prorated = ($due_date === $first_bill)
                            ? round($monthly * ($days_in_month - ($move_in->format('j') - 1)) / $days_in_month)
                            : $monthly;
                        $ins = $pdo->prepare("INSERT INTO payments (tenant_id, amount, due_date, status) VALUES (?, ?, ?, 'pending')");
                        $ins->execute([$unit['tenant_id'], $prorated, $due_date]);
                        $p = ['amount' => $prorated, 'amount_paid' => 0, 'credit_carried' => 0];
                    }

                    $credit_used = min($p['credit_carried'] ?? 0, max(0, $p['amount'] - $p['amount_paid']));
                    $net_balance = $p['amount'] - $p['amount_paid'] - $credit_used;

                    $total_billed += $p['amount'];
                    $total_paid += $p['amount_paid'];
                    $total_credit += ($p['credit_carried'] ?? 0) - $credit_used;
                }

                $arrears = $total_billed - $total_paid;
                $net_arrears = $arrears - $total_credit;

                $search_text = strtolower($property_name . ' ' . $unit['tenant_name']);

                if ($net_arrears > 0) {
                    $border = "border-danger shadow";
                    $header_bg = "bg-danger";
                    $header_text = "text-white";
                    $status = "OWES<br>KSh " . number_format($net_arrears);
                    $status_class = "arrears";
                } elseif ($total_credit > 0) {
                    $border = "border-success shadow";
                    $header_bg = "bg-success";
                    $header_text = "text-white";
                    $status = "CREDIT<br>KSh " . number_format($total_credit);
                    $status_class = "credit";
                } else {
                    $border = "border-primary";
                    $header_bg = "bg-primary";
                    $header_text = "text-white";
                    $status = "PAID UP";
                    $status_class = "paid";
                }
            }
        ?>
        <div class="col property-card"
             data-search="<?= $search_text ?>"
             data-status="<?= $occupied ? 'occupied' : 'vacant' ?>"
             data-financial="<?= $occupied ? ($net_arrears > 0 ? 'arrears' : ($total_credit > 0 ? 'credit' : 'paid')) : 'vacant' ?>">
            <div class="card h-100 <?= $border ?> rounded-4 overflow-hidden hover-shadow position-relative">
                <div class="card-header <?= $header_bg ?> <?= $header_text ?> py-3 text-center">
                    <h6 class="mb-0 fw-bold"><?= $property_name ?></h6>
                </div>
                <div class="card-body p-3 text-center">
                    <?php if ($occupied): ?>
                        <div class="fw-bold text-primary mb-2" title="<?= htmlspecialchars($unit['tenant_name']) ?>">
                            <?= strlen($unit['tenant_name']) > 18 ? substr($unit['tenant_name'],0,16).'...' : htmlspecialchars($unit['tenant_name']) ?>
                        </div>
                        <div class="text-success small fw-bold">KSh <?= number_format($monthly) ?>/mo</div>
                    <?php else: ?>
                        <div class="text-success fw-bold fs-5 mt-3">VACANT</div>
                    <?php endif; ?>
                </div>
                <div class="card-footer <?= $header_bg ?> <?= $header_text ?> py-3 text-center">
                    <div class="fw-bold small"><?= $status ?></div>
                </div>
                <div class="position-absolute top-0 end-0 p-2">
                    <a href="edit_property.php?id=<?= $property_id ?>" class="btn btn-light btn-sm rounded-circle shadow">
                        <i class="bi bi-pencil"></i>
                    </a>
                </div>
                <div class="position-absolute bottom-0 end-0 p-3">
                    <?php if ($occupied): ?>
                        <a href="payments.php?tenant_id=<?= $unit['tenant_id'] ?>" class="btn btn-light btn-sm rounded-circle shadow">
                            Pay
                        </a>
                    <?php else: ?>
                        <a href="add_tenant.php?property_id=<?= $property_id ?>&property_name=<?= urlencode($property_name) ?>"
                           class="btn btn-light btn-sm rounded-circle shadow">
                            Add
                        </a>
                    <?php endif; ?>
                </div>
            </div>
        </div>
        <?php endwhile; ?>
    </div>

    <div class="text-center mt-4">
        <small class="text-muted" id="countDisplay">Loading...</small>
    </div>
</div>

<script>
function applyFilters() {
    const search = document.getElementById('searchInput').value.toLowerCase().trim();
    const filter = document.getElementById('filterStatus').value;
    const cards = document.querySelectorAll('.property-card');
    let visible = 0;
    cards.forEach(card => {
        const text = card.dataset.search;
        const status = card.dataset.status;
        const financial = card.dataset.financial;
        const matchesSearch = !search || text.includes(search);
        let matchesFilter = true;
        if (filter === 'all') matchesFilter = true;
        else if (filter === 'occupied') matchesFilter = status === 'occupied';
        else if (filter === 'vacant') matchesFilter = status === 'vacant';
        else if (filter === 'arrears') matchesFilter = financial === 'arrears';
        else if (filter === 'credit') matchesFilter = financial === 'credit';
        if (matchesSearch && matchesFilter) {
            card.style.display = '';
            visible++;
        } else {
            card.style.display = 'none';
        }
    });
    const total = cards.length;
    document.getElementById('countDisplay').textContent = `Showing ${visible} of ${total} unit${total !== 1 ? 's' : ''}`;
}
document.getElementById('searchInput').addEventListener('input', applyFilters);
document.getElementById('filterStatus').addEventListener('change', applyFilters);
applyFilters();
</script>

<style>
.hover-shadow {
    transition: all 0.3s ease;
    box-shadow: 0 4px 15px rgba(0,0,0,0.1);
}
.hover-shadow:hover {
    transform: translateY(-8px);
    box-shadow: 0 20px 30px rgba(0,0,0,0.18) !important;
}
.card-header, .card-footer {
    background: linear-gradient(135deg, var(--bs-primary), var(--bs-success)) !important;
}
</style>

<?php include '../includes/footer.php'; ?>