<?php
require_once '../includes/functions.php';

// Security
if (!isLoggedIn() || ($_SESSION['role'] ?? '') !== 'admin') {
    redirect('dashboard.php');
}

// Current admin
$admin_id = (int)$_SESSION['user_id'];
$IS_SUPER_ADMIN = ($admin_id === 1); // Change 1 if your main admin has different ID

// Global garbage fee
$garbage_fee = 500;
try {
    $garbage_fee = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
} catch (Exception $e) {}

// ====================== CSV EXPORT ======================
if (isset($_GET['export']) && $_GET['export'] === 'csv') {
    header('Content-Type: text/csv; charset=utf-8');
    header('Content-Disposition: attachment; filename="ReneCAgency_Report_' . date('Y-m-d_His') . '.csv"');
    $output = fopen('php://output', 'w');

    fputcsv($output, ['RENE.C.AGENCY - DETAILED TENANT REPORT']);
    fputcsv($output, ['Generated:', date('d M Y, H:i')]);
    fputcsv($output, []);

    $filter_data = json_decode($_GET['filters'] ?? '[]', true);
    $textFilter = strtolower($filter_data['text'] ?? '');
    $statusFilter = $filter_data['status'] ?? 'all';
    $arrearsFilter = $filter_data['arrears'] ?? 'any';
    $limits = ['gt5k' => 5000, 'gt10k' => 10000, 'gt20k' => 20000, 'gt50k' => 50000];

    // Filtered tenants query
    if ($IS_SUPER_ADMIN) {
        $tenants = $pdo->query("
            SELECT t.id, t.name, t.move_in_date, p.rent_amount, p.water_bill, p.name AS unit
            FROM tenants t
            JOIN properties p ON t.property_id = p.id
            WHERE t.status = 'active'
            ORDER BY t.name
        ");
    } else {
        $tenants = $pdo->prepare("
            SELECT t.id, t.name, t.move_in_date, p.rent_amount, p.water_bill, p.name AS unit
            FROM tenants t
            JOIN properties p ON t.property_id = p.id
            WHERE t.status = 'active' AND p.admin_id = ?
            ORDER BY t.name
        ");
        $tenants->execute([$admin_id]);
    }

    foreach ($tenants as $t) {
        $monthly_fixed = ($t['rent_amount'] ?? 0) + ($t['water_bill'] ?? 0) + $garbage_fee;
        $move_in = new DateTime($t['move_in_date']);
        $first_bill_month = $move_in->format('Y-m-01');
        $start = new DateTime($first_bill_month);
        $end = new DateTime('first day of next month');
        $period = new DatePeriod($start, new DateInterval('P1M'), $end);

        $total_billed = $total_paid = $total_credit = 0;
        $months = [];

        foreach ($period as $dt) {
            $due_date = $dt->format('Y-m-01');
            $month_name = $dt->format('M Y');

            $check = $pdo->prepare("SELECT * FROM payments WHERE tenant_id = ? AND due_date = ?");
            $check->execute([$t['id'], $due_date]);
            $p = $check->fetch();

            if (!$p) {
                $amount = ($dt->format('Y-m') === $first_bill_month)
                    ? round($monthly_fixed * (date('t', strtotime($due_date)) - ($move_in->format('j') - 1)) / date('t', strtotime($due_date)))
                    : $monthly_fixed;
                $ins = $pdo->prepare("INSERT INTO payments (tenant_id, amount, due_date, status) VALUES (?, ?, ?, 'pending')");
                $ins->execute([$t['id'], $amount, $due_date]);
                $p = ['amount' => $amount, 'amount_paid' => 0, 'credit_carried' => 0];
            }

            $credit_used = min($p['credit_carried'] ?? 0, max(0, $p['amount'] - $p['amount_paid']));
            $arrears_this_month = $p['amount'] - $p['amount_paid'];

            $total_billed += $p['amount'];
            $total_paid += $p['amount_paid'];
            $total_credit += $p['credit_carried'] ?? 0;

            $months[] = [
                'month' => $month_name,
                'billed' => $p['amount'],
                'paid' => $p['amount_paid'],
                'credit_used' => $credit_used,
                'arrears' => $arrears_this_month,
                'status' => $arrears_this_month > 0 ? 'OWES' : 'PAID'
            ];
        }

        $arrears = $total_billed - $total_paid;

        // Apply filters
        $matchesText = !$textFilter || stripos($t['name'], $textFilter) !== false || stripos($t['unit'], $textFilter) !== false;
        $matchesStatus = $statusFilter === 'all' ||
            ($statusFilter === 'owes' && $arrears > 0) ||
            ($statusFilter === 'credit' && $total_credit > 0) ||
            ($statusFilter === 'cleared' && $arrears <= 0);
        $matchesArrears = $arrearsFilter === 'any' || $arrears > ($limits[$arrearsFilter] ?? 0);

        if (!$matchesText || !$matchesStatus || !$matchesArrears) continue;

        fputcsv($output, ["TENANT: {$t['name']} | UNIT: {$t['unit']} | FIXED MONTHLY: KSh " . number_format($monthly_fixed)]);
        fputcsv($output, ['TOTAL BILLED: KSh ' . number_format($total_billed), 'TOTAL PAID: KSh ' . number_format($total_paid), 'ARREARS: KSh ' . number_format($arrears)]);
        fputcsv($output, ['Month', 'Billed', 'Paid', 'Credit Used', 'Arrears', 'Status']);
        foreach ($months as $m) {
            fputcsv($output, [
                $m['month'],
                number_format($m['billed']),
                $m['paid'] > 0 ? number_format($m['paid']) : '',
                $m['credit_used'] > 0 ? number_format($m['credit_used']) : '',
                $m['arrears'] > 0 ? number_format($m['arrears']) : '',
                $m['status']
            ]);
        }
        fputcsv($output, []);
    }
    exit;
}

// ====================== MAIN REPORT ======================
$total_billed_ever = $total_paid_ever = $total_arrears_ever = $total_credit_ever = 0;
$tenant_details = [];

// Filtered tenants query
if ($IS_SUPER_ADMIN) {
    $tenants = $pdo->query("
        SELECT t.id, t.name, t.move_in_date, p.rent_amount, p.water_bill, p.name AS unit
        FROM tenants t
        JOIN properties p ON t.property_id = p.id
        WHERE t.status = 'active'
        ORDER BY t.name
    ");
} else {
    $tenants = $pdo->prepare("
        SELECT t.id, t.name, t.move_in_date, p.rent_amount, p.water_bill, p.name AS unit
        FROM tenants t
        JOIN properties p ON t.property_id = p.id
        WHERE t.status = 'active' AND p.admin_id = ?
        ORDER BY t.name
    ");
    $tenants->execute([$admin_id]);
}

foreach ($tenants as $t) {
    $monthly_fixed = ($t['rent_amount'] ?? 0) + ($t['water_bill'] ?? 0) + $garbage_fee;
    $move_in = new DateTime($t['move_in_date']);
    $first_bill_month = $move_in->format('Y-m-01');
    $start = new DateTime($first_bill_month);
    $end = new DateTime('first day of next month');
    $period = new DatePeriod($start, new DateInterval('P1M'), $end);

    $history = [];
    $tenant_billed = $tenant_paid = $tenant_credit = 0;

    foreach ($period as $dt) {
        $due_date = $dt->format('Y-m-01');
        $month_name = $dt->format('M Y');

        $check = $pdo->prepare("SELECT * FROM payments WHERE tenant_id = ? AND due_date = ?");
        $check->execute([$t['id'], $due_date]);
        $p = $check->fetch();

        if (!$p) {
            $amount = ($dt->format('Y-m') === $first_bill_month)
                ? round($monthly_fixed * (date('t', strtotime($due_date)) - ($move_in->format('j') - 1)) / date('t', strtotime($due_date)))
                : $monthly_fixed;
            $ins = $pdo->prepare("INSERT INTO payments (tenant_id, amount, due_date, status) VALUES (?, ?, ?, 'pending')");
            $ins->execute([$t['id'], $amount, $due_date]);
            $p = ['amount' => $amount, 'amount_paid' => 0, 'credit_carried' => 0];
        }

        $credit_used = min($p['credit_carried'] ?? 0, max(0, $p['amount'] - $p['amount_paid']));
        $arrears_this_month = $p['amount'] - $p['amount_paid'];

        $tenant_billed += $p['amount'];
        $tenant_paid += $p['amount_paid'];
        $tenant_credit += $p['credit_carried'] ?? 0;

        $history[] = [
            'month' => $month_name,
            'billed' => $p['amount'],
            'paid' => $p['amount_paid'],
            'credit_used' => $credit_used,
            'arrears' => $arrears_this_month,
            'status' => $arrears_this_month > 0 ? 'danger' : 'success',
            'status_text' => $arrears_this_month > 0 ? 'OWES KSh ' . number_format($arrears_this_month) : 'PAID'
        ];
    }

    $arrears = $tenant_billed - $tenant_paid;

    $total_billed_ever += $tenant_billed;
    $total_paid_ever += $tenant_paid;
    $total_arrears_ever += $arrears;
    $total_credit_ever += $tenant_credit;

    $tenant_details[] = [
        'id' => $t['id'],
        'name' => $t['name'],
        'unit' => $t['unit'],
        'monthly_fixed' => $monthly_fixed,
        'move_in' => $move_in->format('d M Y'),
        'total_billed' => $tenant_billed,
        'total_paid' => $tenant_paid,
        'arrears' => $arrears,
        'credit' => $tenant_credit,
        'history' => $history
    ];
}

include '../includes/header.php';
?>

<div class="container py-4 py-md-5">
    <div class="d-flex flex-column flex-md-row justify-content-between align-items-center gap-3 mb-4">
        <a href="dashboard.php" class="btn btn-outline-dark px-4 py-2 rounded-pill">Back to Dashboard</a>
        <button onclick="exportFiltered()" class="btn btn-success px-5 py-2 rounded-pill fw-bold shadow-sm">
            Export Detailed CSV
        </button>
    </div>

    <h2 class="text-center text-success mb-5 fw-bold display-5">RENE.C.AGENCY — DETAILED PAYMENT HISTORY</h2>

    <!-- Summary Cards -->
    <div class="row g-4 mb-5">
        <div class="col-6 col-md-3">
            <div class="card h-100 border-0 shadow-sm text-white bg-primary">
                <div class="card-body text-center p-4">
                    <small>Total Billed</small>
                    <h4 class="fw-bold mb-0">KSh <?= number_format($total_billed_ever) ?></h4>
                </div>
            </div>
        </div>
        <div class="col-6 col-md-3">
            <div class="card h-100 border-0 shadow-sm text-white bg-success">
                <div class="card-body text-center p-4">
                    <small>Collected</small>
                    <h4 class="fw-bold mb-0">KSh <?= number_format($total_paid_ever) ?></h4>
                </div>
            </div>
        </div>
        <div class="col-6 col-md-3">
            <div class="card h-100 border-0 shadow-sm text-white bg-danger">
                <div class="card-body text-center p-4">
                    <small>ARREARS</small>
                    <h4 class="fw-bold mb-0">KSh <?= number_format($total_arrears_ever) ?></h4>
                </div>
            </div>
        </div>
        <div class="col-6 col-md-3">
            <div class="card h-100 border-0 shadow-sm text-white bg-teal" style="background:#0d9488;">
                <div class="card-body text-center p-4">
                    <small>Total Credit</small>
                    <h4 class="fw-bold mb-0">KSh <?= number_format($total_credit_ever) ?></h4>
                </div>
            </div>
        </div>
    </div>

    <!-- Filters -->
    <div class="card shadow-sm mb-4">
        <div class="card-body">
            <div class="row g-3 align-items-end">
                <div class="col-12 col-md-4">
                    <label class="form-label small fw-bold text-muted">Search Tenant / Unit</label>
                    <input type="text" id="searchText" class="form-control form-control-lg" placeholder="Name or unit">
                </div>
                <div class="col-12 col-md-3">
                    <label class="form-label small fw-bold text-muted">Status</label>
                    <select id="statusFilter" class="form-select form-select-lg">
                        <option value="all">All</option>
                        <option value="owes">OWES</option>
                        <option value="credit">In Credit</option>
                        <option value="cleared">Cleared</option>
                    </select>
                </div>
                <div class="col-12 col-md-3">
                    <label class="form-label small fw-bold text-muted">Arrears</label>
                    <select id="arrearsFilter" class="form-select form-select-lg">
                        <option value="any">Any</option>
                        <option value="gt5k">> 5K</option>
                        <option value="gt10k">> 10K</option>
                        <option value="gt20k">> 20K</option>
                        <option value="gt50k">> 50K</option>
                    </select>
                </div>
                <div class="col-12 col-md-2">
                    <button onclick="clearFilters()" class="btn btn-outline-secondary w-100 h-100">Clear</button>
                </div>
            </div>
        </div>
    </div>

    <!-- Accordion -->
    <div class="accordion" id="tenantsAccordion">
        <?php foreach ($tenant_details as $i => $td): ?>
        <div class="accordion-item tenant-row"
             data-name="<?= strtolower($td['name']) ?>"
             data-unit="<?= strtolower($td['unit']) ?>"
             data-status="<?= $td['arrears'] > 0 ? 'owes' : ($td['credit'] > 0 ? 'credit' : 'cleared') ?>"
             data-arrears="<?= $td['arrears'] ?>">
            <h2 class="accordion-header">
                <button class="accordion-button <?= $i === 0 ? '' : 'collapsed' ?> fw-bold" type="button" data-bs-toggle="collapse"
                        data-bs-target="#collapse<?= $td['id'] ?>" style="font-size:1.1rem;">
                    <div class="w-100 d-flex justify-content-between align-items-center pe-3">
                        <span><strong>#<?= $i+1 ?>.</strong> <?= htmlspecialchars($td['name']) ?> — <?= htmlspecialchars($td['unit']) ?></span>
                        <span class="text-end">
                            <small class="me-3">Billed: KSh <?= number_format($td['total_billed']) ?></small>
                            <small class="me-3 text-success">Paid: KSh <?= number_format($td['total_paid']) ?></small>
                            <?php if ($td['arrears'] > 0): ?>
                                <span class="badge bg-danger fs-6">OWES KSh <?= number_format($td['arrears']) ?></span>
                            <?php elseif ($td['credit'] > 0): ?>
                                <span class="badge bg-success fs-6">CREDIT KSh <?= number_format($td['credit']) ?></span>
                            <?php else: ?>
                                <span class="badge bg-secondary fs-6">CLEARED</span>
                            <?php endif; ?>
                        </span>
                    </div>
                </button>
            </h2>
            <div id="collapse<?= $td['id'] ?>" class="accordion-collapse collapse <?= $i === 0 ? 'show' : '' ?>" data-bs-parent="#tenantsAccordion">
                <div class="accordion-body p-0">
                    <div class="table-responsive">
                        <table class="table table-sm table-hover mb-0">
                            <thead class="table-light">
                                <tr>
                                    <th>Month</th>
                                    <th class="text-end">Billed</th>
                                    <th class="text-end">Paid</th>
                                    <th class="text-end">Credit Used</th>
                                    <th class="text-center">Balance</th>
                                    <th>Status</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($td['history'] as $h): ?>
                                <tr>
                                    <td><?= $h['month'] ?></td>
                                    <td class="text-end">KSh <?= number_format($h['billed']) ?></td>
                                    <td class="text-end text-success fw-bold"><?= $h['paid'] > 0 ? 'KSh '.number_format($h['paid']) : '—' ?></td>
                                    <td class="text-end text-info"><?= $h['credit_used'] > 0 ? 'KSh '.number_format($h['credit_used']) : '—' ?></td>
                                    <td class="text-center fw-bold <?= $h['arrears'] > 0 ? 'text-danger' : 'text-success' ?>">
                                        <?= $h['arrears'] > 0 ? 'OWES KSh '.number_format($h['arrears']) : 'PAID' ?>
                                    </td>
                                    <td>
                                        <span class="badge bg-<?= $h['status'] ?>"><?= $h['arrears'] > 0 ? 'OWES' : 'PAID' ?></span>
                                    </td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
        <?php endforeach; ?>
    </div>

    <?php if ($total_arrears_ever > 0): ?>
        <div class="alert alert-danger text-center mt-5 fs-4 fw-bold py-4 rounded-4 shadow">
            URGENT: KSh <?= number_format($total_arrears_ever) ?> IN TOTAL ARREARS!
        </div>
    <?php else: ?>
        <div class="alert alert-success text-center mt-5 fs-4 fw-bold py-4 rounded-4 shadow">
            ALL TENANTS ARE PAID OR IN CREDIT — EXCELLENT!
        </div>
    <?php endif; ?>
</div>

<script>
function applyFilters() {
    const text = document.getElementById('searchText').value.toLowerCase();
    const status = document.getElementById('statusFilter').value;
    const arrears = document.getElementById('arrearsFilter').value;
    const limits = {gt5k:5000, gt10k:10000, gt20k:20000, gt50k:50000};

    document.querySelectorAll('.tenant-row').forEach(row => {
        const name = row.dataset.name;
        const unit = row.dataset.unit;
        const rowStatus = row.dataset.status;
        const arrearsVal = parseFloat(row.dataset.arrears) || 0;

        let show = true;
        if (text && !name.includes(text) && !unit.includes(text)) show = false;
        if (status !== 'all' && rowStatus !== status) show = false;
        if (arrears !== 'any' && arrearsVal <= (limits[arrears] ?? 0)) show = false;

        row.style.display = show ? '' : 'none';
    });
}

function exportFiltered() {
    const filters = {
        text: document.getElementById('searchText').value,
        status: document.getElementById('statusFilter').value,
        arrears: document.getElementById('arrearsFilter').value
    };
    const params = new URLSearchParams({export: 'csv', filters: JSON.stringify(filters)});
    window.location = '?' + params.toString();
}

function clearFilters() {
    document.getElementById('searchText').value = '';
    document.getElementById('statusFilter').value = 'all';
    document.getElementById('arrearsFilter').value = 'any';
    applyFilters();
}

['searchText', 'statusFilter', 'arrearsFilter'].forEach(id => {
    document.getElementById(id).addEventListener(id === 'searchText' ? 'keyup' : 'change', applyFilters);
});

applyFilters();
</script>

<?php include '../includes/footer.php'; ?>