<?php
session_start();
require_once '../includes/functions.php';

if (!isset($_SESSION['tenant_id'])) {
    redirect('../tenant/login.php');
}

$tenant_id = $_SESSION['tenant_id'];

// Fetch tenant data
$stmt = $pdo->prepare("
    SELECT t.name, t.move_in_date, t.phone,
           COALESCE(p.name, 'Not Assigned') AS property_name,
           COALESCE(p.rent_amount, 0) + COALESCE(p.water_bill, 0) AS base_rent
    FROM tenants t
    LEFT JOIN properties p ON t.property_id = p.id
    WHERE t.id = ?
");
$stmt->execute([$tenant_id]);
$tenant = $stmt->fetch();

if (!$tenant) { session_destroy(); redirect('../tenant/login.php'); }

// Global garbage fee
$garbage_fee = $pdo->query("SELECT COALESCE(setting_value, 500) FROM settings WHERE setting_key='global_garbage_fee'")->fetchColumn() ?: 500;
$monthly_rent = $tenant['base_rent'] + $garbage_fee;

// === MONTHS BILLED (MOVE-IN → TODAY) ===
$move_in = new DateTime($tenant['move_in_date']);
$first_bill = ($move_in->format('d') === '01')
    ? $move_in->format('Y-m-01')
    : (clone $move_in)->modify('first day of next month')->format('Y-m-01');

$start = new DateTime($first_bill);
$end = new DateTime('first day of next month');
$period = new DatePeriod($start, new DateInterval('P1M'), $end);
$months_billed = iterator_count($period);

// TOTAL BILLED = MONTHLY × MONTHS (YOUR EXACT RULE)
$total_billed = $monthly_rent * $months_billed;

// Paid & Credit from DB
$totals = $pdo->query("SELECT COALESCE(SUM(amount_paid),0), COALESCE(SUM(credit_carried),0) FROM payments WHERE tenant_id = $tenant_id")->fetch(PDO::FETCH_NUM);
$total_paid = $totals[0];
$total_credit = $totals[1];

$arrears = $total_billed - $total_paid;
$net_balance = $arrears - $total_credit;
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>My Rent • <?= htmlspecialchars($tenant['name']) ?> | Rene.C.Agency</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.11.3/font/bootstrap-icons.min.css" rel="stylesheet">
    <link href="https://cdn.jsdelivr.net/npm/@tabler/icons-webfont@latest/dist/tabler-icons.min.css" rel="stylesheet">
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700;800&display=swap" rel="stylesheet">
    <style>
        :root {--primary:#6366f1;--success:#10b981;--danger:#ef4444;--warning:#f59e0b;--info:#0ea5e9;--dark:#1e293b;}
        body {font-family:'Inter',sans-serif;background:linear-gradient(135deg,#667eea,#764ba2);min-height:100vh;color:#1e293b;}
        .glass {background:rgba(255,255,255,0.15);backdrop-filter:blur(20px);border-radius:24px;border:1px solid rgba(255,255,255,0.2);box-shadow:0 20px 40px rgba(0,0,0,0.2);}
        .header {background:linear-gradient(135deg,#6366f1,#8b5cf6);color:white;}
        .stat-card {background:white;border-radius:20px;padding:2rem;box-shadow:0 10px 30px rgba(0,0,0,0.1);transition:0.3s;}
        .stat-card:hover {transform:translateY(-12px);box-shadow:0 25px 50px rgba(0,0,0,0.2);}
        .arrears {background:linear-gradient(135deg,#ef4444,#dc2626);color:white;}
        .credit {background:linear-gradient(135deg,#0ea5e9,#0284c7);color:white;}
        .paidup {background:linear-gradient(135deg,#10b981,#059669);color:white;}
        .btn-report {
            background: linear-gradient(135deg, #1e293b, #334155);
            color: white;
            font-size: 1.3rem;
            padding: 1rem 3rem;
            border-radius: 50px;
            font-weight: 700;
            box-shadow: 0 10px 30px rgba(0,0,0,0.3);
            transition: all 0.4s;
        }
        .btn-report:hover {
            transform: translateY(-5px);
            box-shadow: 0 20px 40px rgba(0,0,0,0.4);
            background: linear-gradient(135deg, #334155, #1e293b);
        }
        .table-modern thead {background:linear-gradient(135deg,#1e293b,#334155);color:white;}
        .badge-status {padding:0.6em 1.2em;border-radius:50px;font-weight:600;}
    </style>
</head>
<body>

<div class="container py-5">
    <div class="row justify-content-center">
        <div class="col-xl-11 col-lg-12">

            <!-- Header -->
            <div class="glass header shadow-lg mb-5 p-5 text-center">
                <h1 class="display-4 fw-bold mb-3">Welcome, <?= htmlspecialchars($tenant['name']) ?>!</h1>
                <p class="lead opacity-90">
                    <strong><?= htmlspecialchars($tenant['property_name']) ?></strong> • 
                    Monthly Rent: <strong>KSh <?= number_format($monthly_rent) ?></strong>
                </p>
                <div class="d-flex justify-content-center gap-4 mt-4 flex-wrap">
                    <a href="logout.php" class="btn btn-outline-light btn-lg rounded-pill px-5">Logout</a>
                    <a href="tel:+254700000000" class="btn btn btn-success btn-lg rounded-pill px-5">Contact Landlord</a>
                </div>
            </div>

            <!-- Stats -->
            <div class="row g-4 mb-5">
                <div class="col-md-6 col-lg-3"><div class="stat-card text-center h-100%"><h2 class="text-primary fw-bold">KSh <?= number_format($total_billed) ?></h2><p>Total Billed</p><small><?= $months_billed ?> months</small></div></div>
                <div class="col-md-6 col-lg-3"><div class="stat-card text-center h-100%"><h2 class="text-success fw-bold">KSh <?= number_format($total_paid) ?></h2><p>Total Paid</p></div></div>
                <div class="col-md-6 col-lg-3"><div class="stat-card text-white text-center h-100 <?= $arrears>0?'arrears':'paidup' ?>"><h2 class="fw-bold">KSh <?= number_format($arrears) ?></h2><p><?= $arrears>0?'ARREARS':'NO ARREARS' ?></p></div></div>
                <div class="col-md-6 col-lg-3"><div class="stat-card text-white text-center h-100 <?= $total_credit>0?'credit':'bg-secondary' ?>"><h2 class="fw-bold">KSh <?= number_format($total_credit) ?></h2><p>Credit Balance</p></div></div>
            </div>

            <!-- Monthly Table -->
            <div class="glass shadow-lg mb-5">
                <div class="p-4 border-bottom">
                    <h3 class="fw-bold mb-1">Monthly Rent Statement</h3>
                    <p class="text-white-75 mb-0">From <?= date('F Y', strtotime($first_bill)) ?> → <?= date('F Y') ?> • <?= $months_billed ?> months</p>
                </div>
                <div class="table-responsive">
                    <table class="table table-hover mb-0 table-modern">
                        <thead><tr>
                            <th>Month</th>
                            <th class="text-end">Billed</th>
                            <th class="text-end">Paid</th>
                            <th class="text-end">Credit Used</th>
                            <th class="text-center">Balance</th>
                            <th>Status</th>
                        </tr></thead>
                        <tbody>
                            <?php foreach ($period as $dt):
                                $due_date = $dt->format('Y-m-01');
                                $display = $dt->format('M Y');

                                $stmt = $pdo->prepare("SELECT * FROM payments WHERE tenant_id = ? AND due_date = ?");
                                $stmt->execute([$tenant_id, $due_date]);
                                $p = $stmt->fetch();

                                if (!$p) {
                                    $days = $dt->format('t');
                                    $prorated = ($due_date === $first_bill)
                                        ? round($monthly_rent * ($days - ($move_in->format('j')-1)) / $days)
                                        : $monthly_rent;
                                    $pdo->prepare("INSERT INTO payments (tenant_id, amount, due_date, status) VALUES (?, ?, ?, 'pending')")
                                        ->execute([$tenant_id, $prorated, $due_date]);
                                    $p = ['amount'=>$prorated, 'amount_paid'=>0, 'credit_carried'=>0];
                                }

                                $credit_used = min($p['credit_carried']??0, max(0, $p['amount'] - $p['amount_paid']));
                                $balance = $p['amount'] - $p['amount_paid'] - $credit_used;
                            ?>
                            <tr>
                                <td class="fw-600"><?= $display ?></td>
                                <td class="text-end">KSh <?= number_format($p['amount']) ?></td>
                                <td class="text-end text-success fw-bold">KSh <?= number_format($p['amount_paid']) ?></td>
                                <td class="text-end text-info"><?= $credit_used>0?'KSh '.number_format($credit_used):'—' ?></td>
                                <td class="text-center fw-bold <?= $balance>0?'text-danger':'text-success' ?>">
                                    <?= $balance>0?'KSh '.number_format($balance):'Cleared' ?>
                                </td>
                                <td>
                                    <span class="badge-status <?= $balance>0?'bg-danger':'bg-success' ?>">
                                        <?= $balance>0?'OWES':'PAID' ?>
                                    </span>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- BIG BUTTON – IMPOSSIBLE TO MISS -->
            <div class="text-center my-5">
                <a href="reports.php" target="_blank" class="btn btn-report btn-lg">
                    <i class="ti ti-download fs-3 align-middle me-3"></i>
                    VIEW FULL REPORT & DOWNLOAD PDF
                </a>
            </div>

            <!-- Final Message -->
            <div class="text-center">
                <div class="glass p-5 d-inline-block">
                    <?php if ($arrears > 0): ?>
                        <h2 class="text-danger fw-bold">Please clear KSh <?= number_format($arrears) ?> arrears</h2>
                    <?php elseif ($total_credit > 0): ?>
                        <h2 class="text-info fw-bold">You have KSh <?= number_format($total_credit) ?> credit</h2>
                    <?php else: ?>
                        <h2 class="text-success fw-bold">You are fully paid up!</h2>
                    <?php endif; ?>
                </div>
            </div>

            <div class="text-center mt-5 text-white-50">
                <p>&copy; <?= date('Y') ?> Rene.C.Agency • Professional • Transparent • Trusted</p>
            </div>

        </div>
    </div>
</div>

<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>