<?php
session_start();
session_regenerate_id(true);

// Enforce HTTPS
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    header("Location: https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
    exit();
}

// Disable debug in production
if ($_SERVER['SERVER_NAME'] !== 'localhost') {
    ini_set('display_errors', 0);
    error_reporting(0);
}

// Check if admin is logged in
if (!isset($_SESSION['student_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: login.php?error=" . urlencode("Access denied. Admins only."));
    exit();
}

// Database connection
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "school_db";

try {
    $conn = new PDO("mysql:host=$servername;dbname=$dbname", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    error_log("Connection failed: " . $e->getMessage());
    $error = "Connection failed. Please contact support.";
}

// Fetch admin's student details
$admin_student = null;
if (isset($_SESSION['student_id'])) {
    try {
        $stmt = $conn->prepare("SELECT name, grade, contact, admission_no, programme, stage FROM students WHERE id = :student_id");
        $stmt->execute([':student_id' => (int)$_SESSION['student_id']]);
        $admin_student = $stmt->fetch(PDO::FETCH_ASSOC);
    } catch(PDOException $e) {
        error_log("Admin student fetch error: " . $e->getMessage());
        $error = "Error fetching profile data.";
    }
}

// CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// Fetch table statistics
$tables = [
    'academics' => ['name' => 'Academics', 'rows' => 0, 'page' => 'manage_academics.php', 'icon' => 'bi-book'],
    'accommodation_bookings' => ['name' => 'Accommodation Bookings', 'rows' => 0, 'page' => 'accommodation.php', 'icon' => 'bi-building'],
    'clearance_forms' => ['name' => 'Clearance Forms', 'rows' => 0, 'page' => 'manage_clearance.php', 'icon' => 'bi-check-circle'],
    'contact_messages' => ['name' => 'Contact Messages', 'rows' => 1, 'page' => 'manage_contact_messages.php', 'icon' => 'bi-envelope'],
    'financials' => ['name' => 'Financials', 'rows' => 1, 'page' => 'manage_financials.php', 'icon' => 'bi-wallet'],
    'news' => ['name' => 'News', 'rows' => 2, 'page' => 'manage_news.php', 'icon' => 'bi-newspaper'],
    'resources' => ['name' => 'Resources', 'rows' => 0, 'page' => 'manage_resources.php', 'icon' => 'bi-file-earmark'],
    'rooms' => ['name' => 'Rooms', 'rows' => 6, 'page' => 'accommodation.php', 'icon' => 'bi-building'],
    'special_exams_applications' => ['name' => 'Special Exams', 'rows' => 0, 'page' => 'manage_special_exams.php', 'icon' => 'bi-pencil-square'],
    'students' => ['name' => 'Students', 'rows' => 1, 'page' => 'manage_students.php', 'icon' => 'bi-people'],
    'student_requests' => ['name' => 'Student Requests', 'rows' => 0, 'page' => 'manage_student_requests.php', 'icon' => 'bi-file-text'],
    'timetables' => ['name' => 'Timetables', 'rows' => 2, 'page' => 'manage_timetables.php', 'icon' => 'bi-calendar'],
    'transactions' => ['name' => 'Transactions', 'rows' => 2, 'page' => 'manage_transactions.php', 'icon' => 'bi-currency-exchange'],
    'transfer_forms' => ['name' => 'Transfer Forms', 'rows' => 0, 'page' => 'manage_transfers.php', 'icon' => 'bi-arrow-left-right'],
    'users' => ['name' => 'Users', 'rows' => 2, 'page' => 'manage_users.php', 'icon' => 'bi-person-circle'],
];

// Fetch row counts dynamically
try {
    foreach ($tables as $table => &$info) {
        $stmt = $conn->query("SELECT COUNT(*) FROM $table");
        $info['rows'] = (int)$stmt->fetchColumn();
    }
    unset($info); // Unset reference
} catch(PDOException $e) {
    error_log("Table counts fetch error: " . $e->getMessage());
    $error = "Error fetching table counts.";
}

// Fetch pending actions
$pending_counts = [];
try {
    $pending_counts['accommodation_bookings'] = $conn->query("SELECT COUNT(*) FROM accommodation_bookings WHERE status = 'pending'")->fetchColumn();
    $pending_counts['clearance_forms'] = $conn->query("SELECT COUNT(*) FROM clearance_forms WHERE status = 'pending'")->fetchColumn();
    $pending_counts['special_exams_applications'] = $conn->query("SELECT COUNT(*) FROM special_exams_applications WHERE status = 'pending'")->fetchColumn();
    $pending_counts['student_requests'] = $conn->query("SELECT COUNT(*) FROM student_requests WHERE status = 'pending'")->fetchColumn();
    $pending_counts['transfer_forms'] = $conn->query("SELECT COUNT(*) FROM transfer_forms WHERE status = 'pending'")->fetchColumn();
} catch(PDOException $e) {
    error_log("Pending counts fetch error: " . $e->getMessage());
    $error = "Error fetching pending actions.";
}

// Handle logout
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['logout']) && $_POST['csrf_token'] === $csrf_token) {
    session_unset();
    session_destroy();
    header("Location: login.php");
    exit();
}

// Handle redirect messages
$message = '';
$alert_type = '';
if (isset($_GET['success'])) {
    $message = htmlspecialchars($_GET['success']);
    $alert_type = 'alert-success';
} elseif (isset($_GET['error'])) {
    $message = htmlspecialchars($_GET['error']);
    $alert_type = 'alert-danger';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="Admin dashboard for Gakero E.L.C.K High School, managing all school operations.">
    <meta name="keywords" content="Gakero E.L.C.K High School, admin dashboard, school management, accommodation, students, financials">
    <meta name="author" content="Gakero E.L.C.K High School">
    <title>Admin Dashboard - Gakero E.L.C.K High School</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600&display=swap" rel="stylesheet">
    <style>
        * { box-sizing: border-box; }
        body { font-family: 'Poppins', sans-serif; background-color: #f5f6f5; color: #333; margin: 0; overflow-x: hidden; width: 100%; }
        .offcanvas { width: 250px; }
        .navbar { background-color: #003087; padding: 8px 0; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1); z-index: 1001; position: sticky; top: 0; }
        .navbar-brand { font-size: 1.25rem; font-weight: 600; color: #fff !important; }
        .nav-link { font-size: 0.9rem; color: #fff !important; padding: 6px 12px; transition: color 0.3s ease, background-color 0.3s ease; }
        .nav-link:hover { color: #a9d6e5 !important; background-color: rgba(255, 255, 255, 0.1); border-radius: 4px; }
        .navbar-toggler { border-color: rgba(255, 255, 255, 0.5); }
        .sidebar { position: fixed; top: 48px; left: 0; width: 250px; height: calc(100vh - 48px); background-color: #003087; color: #fff; overflow-y: auto; z-index: 1000; box-shadow: 2px 0 4px rgba(0, 0, 0, 0.1); transition: transform 0.3s ease; }
        .sidebar-header { border-bottom: 1px solid rgba(255, 255, 255, 0.2); }
        .sidebar .nav-link { font-size: 0.95rem; padding: 10px 15px; color: #fff; transition: background-color 0.3s ease, padding-left 0.3s ease; }
        .sidebar .nav-link:hover { background-color: rgba(255, 255, 255, 0.1); padding-left: 20px; }
        .profile-card { background: rgba(255, 255, 255, 0.05); border-radius: 8px; margin: 15px; padding: 15px; }
        .avatar { width: 80px; height: 80px; display: flex; align-items: center; justify-content: center; border-radius: 50%; }
        main { margin-left: 250px; padding: 15px; min-height: calc(100vh - 48px); background-color: #fff; border-radius: 8px; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05); z-index: 999; width: calc(100% - 250px); }
        footer { background-color: #003087; color: #fff; padding: 20px 0; margin-top: 20px; z-index: 999; width: 100%; }
        footer a { color: #a9d6e5; text-decoration: none; transition: color 0.3s ease; }
        footer a:hover { color: #fff; }
        .hero-section {margin-left: 150px; background: linear-gradient(135deg, #003087, #005f73); border-radius: 8px; padding: 20px; margin-bottom: 2rem; }
        .hero-section h1 { font-size: 1.75rem; font-weight: 600; }
        .hero-section p { font-size: 1rem; }
        section{margin-left: 150px;}
        .section-title { font-size: 1.75rem; color: #003087; margin-bottom: 1rem; position: relative; }
        .section-title::after { content: ''; display: block; width: 50px; height: 3px; background: #003087; margin-top: 0.5rem; }
        .card { border: none; border-radius: 8px; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1); transition: transform 0.3s ease, box-shadow 0.3s ease; }
        .card:hover { transform: translateY(-5px); box-shadow: 0 6px 16px rgba(0, 0, 0, 0.2); }
        .card-title { font-size: 1.25rem; color: #003087; }
        .card-text { font-size: 0.9rem; color: #555; }
        .btn-primary { background-color: #003087; border: none; padding: 0.75rem 1.5rem; border-radius: 8px; transition: background-color 0.3s ease; }
        .btn-primary:hover { background-color: #00205b; }
        .alert { border-radius: 8px; font-size: 0.9rem; }
        .badge-pending { background-color: #ffc107; color: #333; }
        @media (max-width: 991px) {
            .sidebar { display: none; }
            main { margin-left: 0; width: 100%; }
            .navbar-brand { font-size: 1.1rem; }
            .nav-link { font-size: 0.85rem; }
            .hero-section { padding: 2rem 0; }
            .section-title { font-size: 1.5rem; }
        }
        @media (min-width: 992px) {
            .sidebar { display: block; }
            main { margin-left: 250px; width: calc(100% - 250px); }
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg bg-primary text-white">
        <div class="container-fluid">
            <a class="navbar-brand" href="index.php">Gakero E.L.C.K High School</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item"><a class="nav-link" href="index.php">Home</a></li>
                    <li class="nav-item"><a class="nav-link" href="about.php">About</a></li>
                    <li class="nav-item"><a class="nav-link" href="contact.php">Contact</a></li>
                    <li class="nav-item"><a class="nav-link" href="news.php">News</a></li>
                    <li class="nav-item">
                        <form method="POST" action="admin_page.php" style="display:inline;">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                            <button type="submit" name="logout" class="nav-link btn btn-link text-white">Logout</button>
                        </form>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="d-flex">
        <nav class="sidebar bg-primary text-white" id="sidebar">
            <div class="sidebar-header text-center p-3">
                <h4 class="mb-0">Gakero E.L.C.K</h4>
            </div>
            <div class="profile-card">
                <div class="text-center">
                    <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2" role="img" aria-label="Admin profile icon">
                        <i class="bi bi-person-circle fs-2"></i>
                    </div>
                    <h6 class="mb-1"><?php echo htmlspecialchars($admin_student['name'] ?? 'Admin'); ?></h6>
                    <small class="text-white-50">ID: <?php echo htmlspecialchars($admin_student['admission_no'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Programme: <?php echo htmlspecialchars($admin_student['programme'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Stage: <?php echo htmlspecialchars($admin_student['stage'] ?? 'N/A'); ?></small>
                </div>
            </div>
            <h5 class="p-3">MAIN NAVIGATION</h5>
            <ul class="nav flex-column p-3">
                <li class="nav-item">
                    <a class="nav-link text-white active" href="admin_page.php"><i class="bi bi-house-door me-2"></i>Dashboard</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Manage Accommodation</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_academics.php"><i class="bi bi-book me-2"></i>Manage Academics</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_clearance.php"><i class="bi bi-check-circle me-2"></i>Manage Clearance</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_contact_messages.php"><i class="bi bi-envelope me-2"></i>Manage Contact Messages</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_financials.php"><i class="bi bi-wallet me-2"></i>Manage Financials</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>Manage News</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_resources.php"><i class="bi bi-file-earmark me-2"></i>Manage Resources</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_special_exams.php"><i class="bi bi-pencil-square me-2"></i>Manage Special Exams</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_students.php"><i class="bi bi-people me-2"></i>Manage Students</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_student_requests.php"><i class="bi bi-file-text me-2"></i>Manage Student Requests</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_timetables.php"><i class="bi bi-calendar me-2"></i>Manage Timetables</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_transactions.php"><i class="bi bi-currency-exchange me-2"></i>Manage Transactions</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_transfers.php"><i class="bi bi-arrow-left-right me-2"></i>Manage Transfers</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_users.php"><i class="bi bi-person-circle me-2"></i>Manage Users</a>
                </li>
                <li class="nav-item">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                        <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                    </form>
                </li>
            </ul>
        </nav>

        <main class="container">
            <button class="btn btn-primary d-lg-none mb-3" type="button" data-bs-toggle="offcanvas" data-bs-target="#sidebarOffcanvas" aria-controls="sidebarOffcanvas">
                <i class="bi bi-list"></i> Menu
            </button>
            <div class="offcanvas offcanvas-start bg-primary text-white" tabindex="-1" id="sidebarOffcanvas" aria-labelledby="sidebarOffcanvasLabel">
                <div class="offcanvas-header">
                    <h5 class="offcanvas-title" id="sidebarOffcanvasLabel">Gakero E.L.C.K</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
                </div>
                <div class="offcanvas-body">
                    <div class="profile-card">
                        <div class="text-center">
                            <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2" role="img" aria-label="Admin profile icon">
                                <i class="bi bi-person-circle fs-2"></i>
                            </div>
                            <h6 class="mb-1"><?php echo htmlspecialchars($admin_student['name'] ?? 'Admin'); ?></h6>
                            <small class="text-white-50">ID: <?php echo htmlspecialchars($admin_student['admission_no'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Programme: <?php echo htmlspecialchars($admin_student['programme'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Stage: <?php echo htmlspecialchars($admin_student['stage'] ?? 'N/A'); ?></small>
                        </div>
                    </div>
                    <h5>MAIN NAVIGATION</h5>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white active" href="admin_page.php"><i class="bi bi-house-door me-2"></i>Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Manage Accommodation</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_academics.php"><i class="bi bi-book me-2"></i>Manage Academics</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_clearance.php"><i class="bi bi-check-circle me-2"></i>Manage Clearance</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_contact_messages.php"><i class="bi bi-envelope me-2"></i>Manage Contact Messages</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_financials.php"><i class="bi bi-wallet me-2"></i>Manage Financials</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>Manage News</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_resources.php"><i class="bi bi-file-earmark me-2"></i>Manage Resources</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_special_exams.php"><i class="bi bi-pencil-square me-2"></i>Manage Special Exams</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_students.php"><i class="bi bi-people me-2"></i>Manage Students</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_student_requests.php"><i class="bi bi-file-text me-2"></i>Manage Student Requests</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_timetables.php"><i class="bi bi-calendar me-2"></i>Manage Timetables</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_transactions.php"><i class="bi bi-currency-exchange me-2"></i>Manage Transactions</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_transfers.php"><i class="bi bi-arrow-left-right me-2"></i>Manage Transfers</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_users.php"><i class="bi bi-person-circle me-2"></i>Manage Users</a>
                        </li>
                        <li class="nav-item">
                            <form method="POST">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                                <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                            </form>
                        </li>
                    </ul>
                </div>
            </div>

            <header class="hero-section text-white text-center py-4 rounded mb-4">
                <h1 class="display-5">Admin Dashboard</h1>
                <p class="lead">Manage all operations for Gakero E.L.C.K High School</p>
                <small class="text-white-50">Total Records: <?php echo array_sum(array_column($tables, 'rows')); ?> across <?php echo count($tables); ?> tables</small>
            </header>

            <?php if ($message): ?>
                <div class="alert <?php echo $alert_type; ?> alert-dismissible fade show" role="alert">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>
            <?php if (isset($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <section class="mb-5">
                <h2 class="section-title">Management Dashboard</h2>
                <div class="row g-4">
                    <?php foreach ($tables as $table => $info): ?>
                        <div class="col-md-4 col-lg-3">
                            <div class="card h-100">
                                <div class="card-body text-center">
                                    <i class="bi <?php echo htmlspecialchars($info['icon']); ?> fs-2 text-primary mb-2"></i>
                                    <h5 class="card-title"><?php echo htmlspecialchars($info['name']); ?></h5>
                                    <p class="card-text">
                                        <?php echo $info['rows']; ?> record<?php echo $info['rows'] != 1 ? 's' : ''; ?>
                                        <?php if (isset($pending_counts[$table]) && $pending_counts[$table] > 0): ?>
                                            <br><span class="badge badge-pending"><?php echo $pending_counts[$table]; ?> pending</span>
                                        <?php endif; ?>
                                    </p>
                                    <a href="<?php echo htmlspecialchars($info['page']); ?>" class="btn btn-primary btn-sm">Manage <?php echo htmlspecialchars($info['name']); ?></a>
                                </div>
                            </div>
                        </div>
                    <?php endforeach; ?>
                </div>
            </section>
        </main>
    </div>

    <footer class="bg-dark text-white text-center py-4">
        <div class="container">
            <p>&copy; <?php echo date("Y"); ?> Gakero E.L.C.K High School. All rights reserved.</p>
            <p>Kisii County, Gucha Sub-county, Misesi Location</p>
            <p>Email: <a href="mailto:info@gakeroelckhigh.edu" class="text-white">info@gakeroelckhigh.edu</a> | Phone: <a href="tel:+1234567890" class="text-white">(123) 456-7890</a></p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>