<?php
session_start();
session_regenerate_id(true); // Prevent session fixation

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// Enforce HTTPS
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    header("Location: https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
    exit();
}

// Disable debug in production
if ($_SERVER['SERVER_NAME'] !== 'localhost') {
    ini_set('display_errors', 0);
    error_reporting(0);
}

// Check if user is logged in
if (!isset($_SESSION['student_id']) || !in_array($_SESSION['role'], ['student', 'admin'])) {
    header("Location: login.php");
    exit();
}

// Database connection
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "school_db";

try {
    $conn = new PDO("mysql:host=$servername;dbname=$dbname", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    error_log("Connection failed: " . $e->getMessage());
    $error = "Connection failed. Please contact support.";
}

// Fetch user's student details
$user = [];
try {
    $stmt = $conn->prepare("SELECT name, grade, contact, admission_no, programme, stage FROM students WHERE id = :student_id");
    $stmt->execute([':student_id' => (int)$_SESSION['student_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        $error = "User data not found.";
    }
} catch(PDOException $e) {
    error_log("User fetch error: " . $e->getMessage());
    $error = "Error fetching user data.";
}

// Fetch resources
$resources = [];
try {
    $resource_stmt = $conn->query("SELECT id, file_name, description, upload_date FROM resources ORDER BY upload_date DESC");
    $resources = $resource_stmt->fetchAll(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    error_log("Resources fetch error: " . $e->getMessage());
    $error = "Error fetching resources.";
}

// Handle upload resource (admin only)
if ($_SESSION['role'] === 'admin' && $_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['csrf_token'] === $csrf_token) {
    if ($_POST['action'] === 'upload_resource') {
        $description = filter_input(INPUT_POST, 'description', FILTER_SANITIZE_STRING);
        $target_dir = "Uploads/";
        $target_file = $target_dir . basename($_FILES["file"]["name"]);
        $uploadOk = 1;
        $fileType = strtolower(pathinfo($target_file, PATHINFO_EXTENSION));

        // Check if file is an actual file
        if (isset($_FILES["file"]["tmp_name"]) && is_uploaded_file($_FILES["file"]["tmp_name"])) {
            // Check file size (e.g., 5MB max)
            if ($_FILES["file"]["size"] > 5000000) {
                $error = "Sorry, your file is too large.";
                $uploadOk = 0;
            }

            // Allow certain file formats
            if (!in_array($fileType, ['pdf', 'doc', 'docx', 'zip'])) {
                $error = "Sorry, only PDF, DOC, DOCX & ZIP files are allowed.";
                $uploadOk = 0;
            }

            // Check if $uploadOk is set to 0 by an error
            if ($uploadOk == 0) {
                $error = $error ?: "Sorry, your file was not uploaded.";
            } else {
                if (move_uploaded_file($_FILES["file"]["tmp_name"], $target_file)) {
                    try {
                        $stmt = $conn->prepare("INSERT INTO resources (file_name, description) VALUES (:file_name, :description)");
                        $stmt->execute([':file_name' => basename($_FILES["file"]["name"]), ':description' => $description]);
                        $success = "Resource uploaded successfully.";
                        header("Location: downloads.php");
                        exit();
                    } catch(PDOException $e) {
                        error_log("Upload resource error: " . $e->getMessage());
                        $error = "Error uploading resource.";
                    }
                } else {
                    $error = "Sorry, there was an error uploading your file.";
                }
            }
        } else {
            $error = "No file was uploaded.";
        }
    }
}

// Handle logout
if (isset($_POST['logout']) && $_POST['csrf_token'] === $csrf_token) {
    session_unset();
    session_destroy();
    header("Location: login.php");
    exit();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="Download resources at Cypher University.">
    <meta name="keywords" content="Cypher University, student portal, downloads">
    <meta name="author" content="Cypher University">
    <title>Downloads - Cypher University</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600&display=swap" rel="stylesheet">
    <style>
        /* Ensure box-sizing is applied globally */
        * {
            box-sizing: border-box;
        }

        /* General Styling */
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f5f6f5;
            color: #333;
            margin: 0;
            overflow-x: hidden;
        }

        /* Sticky Navbar */
        .navbar {
            background-color: #003087;
            padding: 8px 0;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            z-index: 1001;
            position: sticky;
            top: 0;
        }
        .navbar-brand {
            font-size: 1.25rem;
            font-weight: 600;
            color: #fff !important;
        }
        .nav-link {
            font-size: 0.9rem;
            color: #fff !important;
            padding: 6px 12px;
            transition: color 0.3s ease, background-color 0.3s ease;
        }
        .nav-link:hover {
            color: #a9d6e5 !important;
            background-color: rgba(255, 255, 255, 0.1);
            border-radius: 4px;
        }
        .navbar-toggler {
            border-color: rgba(255, 255, 255, 0.5);
        }

        /* Fixed Sidebar */
        .sidebar {
            position: fixed;
            top: 48px;
            left: 0;
            width: 250px;
            height: calc(100vh - 48px);
            background-color: #003087;
            color: #fff;
            overflow-y: auto;
            z-index: 1000;
            box-shadow: 2px 0 4px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease;
        }
        .sidebar-header {
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
        }
        .sidebar .nav-link {
            font-size: 0.95rem;
            padding: 10px 15px;
            color: #fff;
            transition: background-color 0.3s ease, padding-left 0.3s ease;
        }
        .sidebar .nav-link:hover {
            background-color: rgba(255, 255, 255, 0.1);
            padding-left: 20px;
        }
        .profile-card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 8px;
            margin: 15px;
            padding: 15px;
        }
        .avatar {
            width: 80px;
            height: 80px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
        }

        /* Main Content */
        main {
            margin-left: 250px;
            padding: 15px;
            min-height: calc(100vh - 48px);
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
            z-index: 999;
            width: calc(100% - 250px);
        }
        

        /* Static Footer */
        footer {
            background-color: #003087;
            color: #fff;
            padding: 20px 0;
            margin-top: 20px;
            z-index: 999;
            width: 100%;
        }
        footer a {
            color: #a9d6e5;
            text-decoration: none;
            transition: color 0.3s ease;
        }
        footer a:hover {
            color: #fff;
        }

        /* Hero Section */
        .hero-section {
            background: linear-gradient(135deg, #003087, #005f73);
            border-radius: 8px;
            padding: 20px;
            margin-left: 150px;
            margin-bottom: 2rem;
        }
        .hero-section h1 {
            font-size: 1.75rem;
            font-weight: 600;
        }
        .hero-section p {
            font-size: 1rem;
        }

        /* Section Titles */
        .section-title {
            font-size: 1.75rem;
            color: #003087;
            margin-bottom: 1rem;
            position: relative;
        }
        section{
            margin-left: 150px;
        }
        .section-title::after {
            content: '';
            display: block;
            width: 50px;
            height: 3px;
            background: #003087;
            margin-top: 0.5rem;
        }

        /* Cards */
        .card {
            border: none;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.2);
        }
        .card-title {
            font-size: 1.25rem;
            color: #003087;
            margin-bottom: 0.75rem;
        }
        .card-body {
            padding: 1.5rem;
        }

        /* Buttons */
        .btn-primary {
            background-color: #003087;
            border: none;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            transition: background-color 0.3s ease;
        }
        .btn-primary:hover {
            background-color: #00205b;
        }
        .btn-outline-primary {
            border-color: #003087;
            color: #003087;
        }
        .btn-outline-primary:hover {
            background-color: #003087;
            color: #fff;
        }

        /* Tables */
        .table {
            margin-top: 1rem;
        }
        .table th {
            background-color: #e9ecef;
            color: #003087;
        }

        /* Forms */
        .form-control {
            border-radius: 8px;
            border: 1px solid #ced4da;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-control:focus {
            border-color: #003087;
            box-shadow: 0 0 5px rgba(0, 48, 135, 0.5);
        }
        .form-label {
            color: #003087;
        }

        /* Alerts */
        .alert {
            border-radius: 8px;
            font-size: 0.9rem;
        }

        /* Responsive Adjustments */
        @media (max-width: 991px) {
            .sidebar {
                display: none;
            }
            main {
                margin-left: 0;
                width: 100%;
            }
            .navbar-brand {
                font-size: 1.1rem;
            }
            .nav-link {
                font-size: 0.85rem;
            }
            .hero-section {
                padding: 2rem 0;
            }
            .hero-section h1 {
                font-size: 1.5rem;
            }
            .section-title {
                font-size: 1.5rem;
            }
        }

        @media (min-width: 992px) {
            .sidebar {
                display: block;
            }
            main {
                margin-left: 250px;
                width: calc(100% - 250px);
            }
        }
    </style>
</head>
<body>
    <!-- Sticky Navbar -->
    <nav class="navbar navbar-expand-lg bg-primary text-white">
        <div class="container-fluid">
            <a class="navbar-brand" href="#">Cypher University</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php">Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="about.php">About</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="contact.php">Contact</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="news.php">News</a>
                    </li>
                    <li class="nav-item">
                        <form method="POST" class="d-inline">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                            <button type="submit" name="logout" class="nav-link bg-transparent border-0">Logout</button>
                        </form>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="d-flex">
        <!-- Fixed Sidebar -->
        <nav class="sidebar bg-primary text-white" id="sidebar">
            <div class="sidebar-header text-center p-3">
                <h4 class="mb-0">Cypher University</h4>
            </div>
            <div class="profile-card p-3">
                <div class="text-center">
                    <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2">
                        <i class="bi bi-person-circle fs-2"></i>
                    </div>
                    <h6 class="mb-1"><?php echo htmlspecialchars($user['name'] ?? ($_SESSION['role'] === 'admin' ? 'Admin' : 'Student')); ?></h6>
                    <small class="text-white-50">Admission No: <?php echo htmlspecialchars($user['admission_no'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Programme: <?php echo htmlspecialchars($user['programme'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Stage: <?php echo htmlspecialchars($user['stage'] ?? 'N/A'); ?></small>
                </div>
            </div>
            <h5 class="p-3">MAIN NAVIGATION</h5>
            <ul class="nav flex-column p-3">
                <li class="nav-item">
                    <a class="nav-link text-white" href="<?php echo $_SESSION['role'] === 'admin' ? 'admin_page.php' : 'student_dashboard.php'; ?>"><i class="bi bi-house me-2"></i>Dashboard</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="financials.php"><i class="bi bi-wallet me-2"></i>Financials</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="academics.php"><i class="bi bi-book me-2"></i>Academics</a>
                </li>
                <?php if ($_SESSION['role'] === 'admin'): ?>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="manage_students.php"><i class="bi bi-people me-2"></i>Manage Students</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>Manage News</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link text-white" href="manage_timetables.php"><i class="bi bi-calendar me-2"></i>Manage Timetables</a>
                    </li>
                <?php endif; ?>
                <li class="nav-item">
                    <a class="nav-link text-white" href="student_requests.php"><i class="bi bi-file-text me-2"></i>Student Requests</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Accommodation</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="special_exams.php"><i class="bi bi-pencil-square me-2"></i>Special Exams</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white active" href="downloads.php"><i class="bi bi-download me-2"></i>Downloads</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="clearance.php"><i class="bi bi-check-circle me-2"></i>Students Clearance</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="transfer.php"><i class="bi bi-arrow-left-right me-2"></i>Inter-School Transfer</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="settings.php"><i class="bi bi-gear me-2"></i>Settings</a>
                </li>
                <li class="nav-item">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                        <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                    </form>
                </li>
            </ul>
            <h5 class="p-3">PERIODIC ACTIVITIES</h5>
            <ul class="nav flex-column p-3">
                <li class="nav-item">
                    <a class="nav-link text-white" href="lecturers_evaluation.php"><i class="bi bi-person-check me-2"></i>Lecturers Evaluation</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="msu_election.php"><i class="bi bi-box-seam me-2"></i>MSU Election</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="notice_board.php"><i class="bi bi-megaphone me-2"></i>Notice Board</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="https://facebook.com/Cypher" target="_blank"><i class="bi bi-facebook me-2"></i>MSU Official Facebook</a>
                </li>
            </ul>
        </nav>

        <!-- Main Content -->
        <main class="container">
            <button class="btn btn-primary d-lg-none mb-3" type="button" data-bs-toggle="offcanvas" data-bs-target="#sidebarOffcanvas" aria-controls="sidebarOffcanvas">
                <i class="bi bi-list"></i> Menu
            </button>
            <div class="offcanvas offcanvas-start bg-primary text-white" tabindex="-1" id="sidebarOffcanvas" aria-labelledby="sidebarOffcanvasLabel">
                <div class="offcanvas-header">
                    <h5 class="offcanvas-title" id="sidebarOffcanvasLabel">Cypher University</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
                </div>
                <div class="offcanvas-body">
                    <div class="profile-card p-3">
                        <div class="text-center">
                            <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2">
                                <i class="bi bi-person-circle fs-2"></i>
                            </div>
                            <h6 class="mb-1"><?php echo htmlspecialchars($user['name'] ?? ($_SESSION['role'] === 'admin' ? 'Admin' : 'Student')); ?></h6>
                            <small class="text-white-50">Admission No: <?php echo htmlspecialchars($user['admission_no'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Programme: <?php echo htmlspecialchars($user['programme'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Stage: <?php echo htmlspecialchars($user['stage'] ?? 'N/A'); ?></small>
                        </div>
                    </div>
                    <h5>MAIN NAVIGATION</h5>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="<?php echo $_SESSION['role'] === 'admin' ? 'admin_page.php' : 'student_dashboard.php'; ?>"><i class="bi bi-house me-2"></i>Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="financials.php"><i class="bi bi-wallet me-2"></i>Financials</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="academics.php"><i class="bi bi-book me-2"></i>Academics</a>
                        </li>
                        <?php if ($_SESSION['role'] === 'admin'): ?>
                            <li class="nav-item">
                                <a class="nav-link text-white" href="manage_students.php"><i class="bi bi-people me-2"></i>Manage Students</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>Manage News</a>
                            </li>
                            <li class="nav-item">
                                <a class="nav-link text-white" href="manage_timetables.php"><i class="bi bi-calendar me-2"></i>Manage Timetables</a>
                            </li>
                        <?php endif; ?>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="student_requests.php"><i class="bi bi-file-text me-2"></i>Student Requests</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Accommodation</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="special_exams.php"><i class="bi bi-pencil-square me-2"></i>Special Exams</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white active" href="downloads.php"><i class="bi bi-download me-2"></i>Downloads</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="clearance.php"><i class="bi bi-check-circle me-2"></i>Students Clearance</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="transfer.php"><i class="bi bi-arrow-left-right me-2"></i>Inter-School Transfer</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="settings.php"><i class="bi bi-gear me-2"></i>Settings</a>
                        </li>
                        <li class="nav-item">
                            <form method="POST">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                                <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                            </form>
                        </li>
                    </ul>
                    <h5>PERIODIC ACTIVITIES</h5>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="lecturers_evaluation.php"><i class="bi bi-person-check me-2"></i>Lecturers Evaluation</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="msu_election.php"><i class="bi bi-box-seam me-2"></i>MSU Election</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="notice_board.php"><i class="bi bi-megaphone me-2"></i>Notice Board</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="https://facebook.com/Cypher" target="_blank"><i class="bi bi-facebook me-2"></i>MSU Official Facebook</a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Hero Section -->
            <header class="hero-section text-white text-center py-4 rounded mb-4">
                <h1 class="display-5">Downloads</h1>
                <p class="lead">Download resources</p>
                <small class="text-white-50">Online</small>
            </header>

            <?php if (isset($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>
            <?php if (isset($success)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($success); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <?php if ($_SESSION['role'] === 'admin'): ?>
                <!-- Upload Resource Form -->
                <section class="mb-5">
                    <h2 class="section-title">Upload Resource</h2>
                    <div class="card">
                        <div class="card-body">
                            <form method="POST" enctype="multipart/form-data">
                                <input type="hidden" name="action" value="upload_resource">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                                <div class="mb-3">
                                    <label for="file" class="form-label">File</label>
                                    <input type="file" id="file" name="file" class="form-control" required>
                                </div>
                                <div class="mb-3">
                                    <label for="description" class="form-label">Description</label>
                                    <textarea id="description" name="description" class="form-control" required></textarea>
                                </div>
                                <button type="submit" class="btn btn-primary">Upload</button>
                            </form>
                        </div>
                    </div>
                </section>
            <?php endif; ?>

            <!-- Resources List -->
            <section class="mb-5">
                <h2 class="section-title">Available Resources</h2>
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th scope="col">File Name</th>
                                <th scope="col">Description</th>
                                <th scope="col">Upload Date</th>
                                <th scope="col">Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($resources)): ?>
                                <?php foreach ($resources as $resource): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($resource['file_name']); ?></td>
                                        <td><?php echo htmlspecialchars($resource['description']); ?></td>
                                        <td><?php echo date('F j, Y H:i', strtotime($resource['upload_date'])); ?></td>
                                        <td>
                                            <a href="Uploads/<?php echo htmlspecialchars($resource['file_name']); ?>" class="btn btn-outline-primary btn-sm" download>Download</a>
                                        </td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr><td colspan="4" class="text-center">No resources available.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </section>
        </main>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white text-center py-4">
        <div class="container">
            <p>&copy; <?php echo date("Y"); ?> Cypher University. Powered by AppKings Solutions Limited.</p>
            <p>Email: <a href="mailto:info@Cypher.ac.ke" class="text-white">info@Cypher.ac.ke</a> | Phone: <a href="tel:+254123456789" class="text-white">+254 123 456 789</a></p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>