<?php
session_start();
session_regenerate_id(true);

// Enforce HTTPS
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    header("Location: https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
    exit();
}

// Enable debug mode for troubleshooting (set to false in production)
$debug_mode = true;
if (!$debug_mode && $_SERVER['SERVER_NAME'] !== 'localhost') {
    ini_set('display_errors', 0);
    error_reporting(0);
}

// Check if admin is logged in
if (!isset($_SESSION['student_id']) || $_SESSION['role'] !== 'admin') {
    header("Location: login.php");
    exit();
}

// Database connection
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "school_db";

try {
    $conn = new PDO("mysql:host=$servername;dbname=$dbname", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    error_log("Connection failed: " . $e->getMessage());
    $error = "Connection failed: " . $e->getMessage();
}

// CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// Fetch admin's student details
try {
    $stmt = $conn->prepare("SELECT name, grade, contact FROM students WHERE id = :student_id");
    $stmt->execute([':student_id' => $_SESSION['student_id']]);
    $admin_student = $stmt->fetch(PDO::FETCH_ASSOC);
} catch(PDOException $e) {
    error_log("Admin student fetch error: " . $e->getMessage());
    $error = "Error fetching admin's student data: " . $e->getMessage();
}

// Fetch financial record
$id = filter_input(INPUT_GET, 'id', FILTER_VALIDATE_INT);
$record = null;
if ($id) {
    try {
        $stmt = $conn->prepare("SELECT f.*, s.name AS student_name FROM financials f LEFT JOIN students s ON f.student_id = s.id WHERE f.id = :id");
        $stmt->execute([':id' => $id]);
        $record = $stmt->fetch(PDO::FETCH_ASSOC);
        if (!$record) {
            $error = "Financial record not found.";
        }
    } catch(PDOException $e) {
        error_log("Fetch financial error: " . $e->getMessage());
        $error = "Error fetching financial record: " . $e->getMessage();
    }
} else {
    $error = "Invalid financial ID.";
}

// Fetch all students for dropdown
$students = [];
try {
    $stmt = $conn->query("SELECT id, name FROM students ORDER BY name");
    $students = $stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
} catch(PDOException $e) {
    error_log("Students fetch error: " . $e->getMessage());
    $error = "Error fetching students: " . $e->getMessage();
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'edit_financial' && $_POST['csrf_token'] === $csrf_token) {
    $id = filter_input(INPUT_POST, 'id', FILTER_VALIDATE_INT);
    $student_id = filter_input(INPUT_POST, 'student_id', FILTER_VALIDATE_INT);
    $total_billed = filter_input(INPUT_POST, 'total_billed', FILTER_VALIDATE_FLOAT);
    $total_paid = filter_input(INPUT_POST, 'total_paid', FILTER_VALIDATE_FLOAT);
    $balance = filter_input(INPUT_POST, 'balance', FILTER_VALIDATE_FLOAT);
    $due_date = filter_input(INPUT_POST, 'due_date', FILTER_SANITIZE_STRING);

    if ($id && $student_id && $total_billed !== false && $total_paid !== false && $balance !== false && $due_date) {
        try {
            $stmt = $conn->prepare("UPDATE financials SET student_id = :student_id, total_billed = :total_billed, total_paid = :total_paid, balance = :balance, due_date = :due_date WHERE id = :id");
            $stmt->execute([
                ':student_id' => $student_id,
                ':total_billed' => $total_billed,
                ':total_paid' => $total_paid,
                ':balance' => $balance,
                ':due_date' => $due_date,
                ':id' => $id
            ]);
            $success = "Financial record updated successfully.";
            header("Location: manage_financials.php?success=" . urlencode($success));
            exit();
        } catch(PDOException $e) {
            error_log("Edit financial error: " . $e->getMessage());
            $error = "Error updating financial record: " . $e->getMessage();
        }
    } else {
        $error = "Invalid financial details.";
    }
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="Edit financial record for Gakero E.L.C.K High School.">
    <meta name="keywords" content="Gakero E.L.C.K High School, admin, edit financial">
    <meta name="author" content="Gakero E.L.C.K High School">
    <title>Edit Financial Record - Gakero E.L.C.K High School</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-9ndCyUaIbzAi2FUVXJi0CjmCapSmO7SnpJef0486qhLnuZ2cdeRhO02iuK6FUUVM" crossorigin="anonymous">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600&display=swap" rel="stylesheet">
    <style>
        * { box-sizing: border-box; }
        body { font-family: 'Poppins', sans-serif; background-color: #f5f6f5; color: #333; margin: 0; overflow-x: hidden; }
        .navbar { background-color: #003087; padding: 8px 0; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1); z-index: 900; position: sticky; top: 0; }
        .navbar-brand { font-size: 1.25rem; font-weight: 600; color: #fff !important; }
        .nav-link { font-size: 0.9rem; color: #fff !important; padding: 6px 12px; transition: color 0.3s ease, background-color 0.3s ease; }
        .nav-link:hover { color: #a9d6e5 !important; background-color: rgba(255, 255, 255, 0.1); border-radius: 4px; }
        .navbar-toggler { border-color: rgba(255, 255, 255, 0.5); }
        .sidebar { position: fixed; top: 48px; left: 0; width: 250px; height: calc(100vh - 48px); background-color: #003087; color: #fff; overflow-y: auto; z-index: 800; box-shadow: 2px 0 4px rgba(0, 0, 0, 0.1); transition: transform 0.3s ease; }
        .sidebar-header { border-bottom: 1px solid rgba(255, 255, 255, 0.2); }
        .sidebar .nav-link { font-size: 0.95rem; padding: 10px 15px; color: #fff; transition: background-color 0.3s ease, padding-left 0.3s ease; }
        .sidebar .nav-link:hover { background-color: rgba(255, 255, 255, 0.1); padding-left: 20px; }
        .profile-card { background: rgba(255, 255, 255, 0.05); border-radius: 8px; margin: 15px; padding: 15px; }
        .avatar { width: 80px; height: 80px; display: flex; align-items: center; justify-content: center; border-radius: 50%; }
        main { margin-left: 250px; padding: 15px; min-height: calc(100vh - 48px); background-color: #fff; border-radius: 8px; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05); z-index: 700; width: calc(100% - 250px); }
        footer { background-color: #003087; color: #fff; padding: 20px 0; margin-top: 20px; z-index: 600; width: 100%; }
        footer a { color: #a9d6e5; text-decoration: none; transition: color 0.3s ease; }
        footer a:hover { color: #fff; }
        .hero-section { background: linear-gradient(135deg, #003087, #005f73); border-radius: 8px; padding: 20px; margin-bottom: 2rem; }
        .hero-section h1 { font-size: 1.75rem; font-weight: 600; }
        .hero-section p { font-size: 1rem; }
        .section-title { font-size: 1.75rem; color: #003087; margin-bottom: 1rem; position: relative; }
        .section-title::after { content: ''; display: block; width: 50px; height: 3px; background: #003087; margin-top: 0.5rem; }
        .card { border: none; border-radius: 8px; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1); transition: transform 0.3s ease, box-shadow 0.3s ease; }
        .card:hover { transform: translateY(-5px); box-shadow: 0 6px 16px rgba(0, 0, 0, 0.2); }
        .card-title { font-size: 1.25rem; color: #003087; }
        .btn-primary { background-color: #003087; border: none; padding: 0.75rem 1.5rem; border-radius: 8px; transition: background-color 0.3s ease; }
        .btn-primary:hover { background-color: #00205b; }
        .btn-outline-primary { border-color: #003087; color: #003087; }
        .btn-outline-primary:hover { background-color: #003087; color: #fff; }
        .form-control { border-radius: 8px; border: 1px solid #ced4da; transition: border-color 0.3s ease, box-shadow 0.3s ease; }
        .form-control:focus { border-color: #003087; box-shadow: 0 0 5px rgba(0, 48, 135, 0.5); }
        .form-label { color: #003087; }
        .alert { border-radius: 8px; font-size: 0.9rem; animation: fadeOut 5s forwards; }
        @keyframes fadeOut {
            0% { opacity: 1; }
            80% { opacity: 1; }
            100% { opacity: 0; display: none; }
        }
        @media (max-width: 991px) {
            .sidebar { display: none; }
            main { margin-left: 0; width: 100%; }
            .navbar-brand { font-size: 1.1rem; }
            .nav-link { font-size: 0.85rem; }
            .hero-section { padding: 2rem 0; }
            .hero-section h1 { font-size: 1.5rem; }
            .section-title { font-size: 1.5rem; }
        }
        @media (min-width: 992px) {
            .sidebar { display: block; }
            main { margin-left: 250px; width: calc(100% - 250px); }
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg bg-primary text-white">
        <div class="container-fluid">
            <a class="navbar-brand" href="index.php">Gakero E.L.C.K High School</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item"><a class="nav-link" href="index.php">Home</a></li>
                    <li class="nav-item"><a class="nav-link" href="about.php">About</a></li>
                    <li class="nav-item"><a class="nav-link" href="contact.php">Contact</a></li>
                    <li class="nav-item"><a class="nav-link" href="news.php">News</a></li>
                    <li class="nav-item">
                        <form method="POST" action="manage_financials.php" style="display:inline;">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                            <button type="submit" name="logout" class="nav-link btn btn-link text-white">Logout</button>
                        </form>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="d-flex">
        <nav class="sidebar bg-primary text-white" id="sidebar">
            <div class="sidebar-header text-center p-3">
                <h4 class="mb-0">Gakero E.L.C.K</h4>
            </div>
            <div class="profile-card">
                <div class="text-center">
                    <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2" role="img" aria-label="Admin profile icon">
                        <i class="bi bi-person-circle fs-2"></i>
                    </div>
                    <h6 class="mb-1"><?php echo htmlspecialchars($admin_student['name'] ?? 'Admin'); ?></h6>
                    <small class="text-white-50">ID: <?php echo htmlspecialchars($_SESSION['student_id'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Grade: <?php echo htmlspecialchars($admin_student['grade'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Contact: <?php echo htmlspecialchars($admin_student['contact'] ?? 'N/A'); ?></small>
                </div>
            </div>
            <ul class="nav flex-column p-3">
                <li class="nav-item"><a class="nav-link text-white" href="admin_page.php"><i class="bi bi-house-door me-2"></i>Dashboard</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Manage Accommodation</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_academics.php"><i class="bi bi-book me-2"></i>Manage Academics</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_clearance.php"><i class="bi bi-check-circle me-2"></i>Manage Clearance</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_contact_messages.php"><i class="bi bi-envelope me-2"></i>Manage Contact Messages</a></li>
                <li class="nav-item"><a class="nav-link text-white active" href="manage_financials.php"><i class="bi bi-wallet me-2"></i>Manage Financials</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>Manage News</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_resources.php"><i class="bi bi-file-earmark me-2"></i>Manage Resources</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_special_exams.php"><i class="bi bi-pencil-square me-2"></i>Manage Special Exams</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_students.php"><i class="bi bi-people me-2"></i>Manage Students</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_student_requests.php"><i class="bi bi-file-text me-2"></i>Manage Student Requests</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_timetables.php"><i class="bi bi-calendar me-2"></i>Manage Timetables</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_transactions.php"><i class="bi bi-currency-exchange me-2"></i>Manage Transactions</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_transfers.php"><i class="bi bi-arrow-left-right me-2"></i>Manage Transfers</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_users.php"><i class="bi bi-person-circle me-2"></i>Manage Users</a></li>
                <li class="nav-item"><a class="nav-link text-white" href="manage_rooms.php"><i class="bi bi-door-open me-2"></i>Manage Rooms</a></li>
                <li class="nav-item">
                    <form method="POST" action="manage_financials.php">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                        <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                    </form>
                </li>
            </ul>
        </nav>

        <main class="flex-grow-1 p-4">
            <button class="btn btn-primary d-lg-none mb-3" type="button" data-bs-toggle="offcanvas" data-bs-target="#sidebarOffcanvas" aria-controls="sidebarOffcanvas">
                <i class="bi bi-list"></i> Menu
            </button>
            <div class="offcanvas offcanvas-start bg-primary text-white" tabindex="-1" id="sidebarOffcanvas" aria-labelledby="sidebarOffcanvasLabel">
                <div class="offcanvas-header">
                    <h5 class="offcanvas-title" id="sidebarOffcanvasLabel">Gakero E.L.C.K</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
                </div>
                <div class="offcanvas-body">
                    <div class="profile-card">
                        <div class="text-center">
                            <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2" role="img" aria-label="Admin profile icon">
                                <i class="bi bi-person-circle fs-2"></i>
                            </div>
                            <h6 class="mb-1"><?php echo htmlspecialchars($admin_student['name'] ?? 'Admin'); ?></h6>
                            <small class="text-white-50">ID: <?php echo htmlspecialchars($_SESSION['student_id'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Grade: <?php echo htmlspecialchars($admin_student['grade'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Contact: <?php echo htmlspecialchars($admin_student['contact'] ?? 'N/A'); ?></small>
                        </div>
                    </div>
                    <ul class="nav flex-column">
                        <li class="nav-item"><a class="nav-link text-white" href="admin_page.php"><i class="bi bi-house-door me-2"></i>Dashboard</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Manage Accommodation</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_academics.php"><i class="bi bi-book me-2"></i>Manage Academics</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_clearance.php"><i class="bi bi-check-circle me-2"></i>Manage Clearance</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_contact_messages.php"><i class="bi bi-envelope me-2"></i>Manage Contact Messages</a></li>
                        <li class="nav-item"><a class="nav-link text-white active" href="manage_financials.php"><i class="bi bi-wallet me-2"></i>Manage Financials</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>Manage News</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_resources.php"><i class="bi bi-file-earmark me-2"></i>Manage Resources</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_special_exams.php"><i class="bi bi-pencil-square me-2"></i>Manage Special Exams</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_students.php"><i class="bi bi-people me-2"></i>Manage Students</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_student_requests.php"><i class="bi bi-file-text me-2"></i>Manage Student Requests</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_timetables.php"><i class="bi bi-calendar me-2"></i>Manage Timetables</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_transactions.php"><i class="bi bi-currency-exchange me-2"></i>Manage Transactions</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_transfers.php"><i class="bi bi-arrow-left-right me-2"></i>Manage Transfers</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_users.php"><i class="bi bi-person-circle me-2"></i>Manage Users</a></li>
                        <li class="nav-item"><a class="nav-link text-white" href="manage_rooms.php"><i class="bi bi-door-open me-2"></i>Manage Rooms</a></li>
                        <li class="nav-item">
                            <form method="POST" action="manage_financials.php">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                                <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                            </form>
                        </li>
                    </ul>
                </div>
            </div>

            <header class="hero-section text-white text-center py-4 rounded mb-4">
                <h1 class="display-5">Edit Financial Record</h1>
                <p class="lead">Update financial details for <?php echo htmlspecialchars($record['student_name'] ?? 'Unknown'); ?></p>
            </header>

            <?php if (isset($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>
            <?php if (isset($success)): ?>
                <div class="alert alert-success alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($success); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <?php if ($record): ?>
                <section class="mb-5">
                    <h2 class="section-title">Edit Financial Record</h2>
                    <div class="card">
                        <div class="card-body">
                            <form method="POST">
                                <input type="hidden" name="action" value="edit_financial">
                                <input type="hidden" name="id" value="<?php echo htmlspecialchars($record['id']); ?>">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                                <div class="mb-3">
                                    <label for="student_id" class="form-label">Student</label>
                                    <select id="student_id" name="student_id" class="form-control" required>
                                        <option value="">Select a student</option>
                                        <?php foreach ($students as $student): ?>
                                            <option value="<?php echo htmlspecialchars($student['id']); ?>" <?php echo $student['id'] == $record['student_id'] ? 'selected' : ''; ?>>
                                                <?php echo htmlspecialchars($student['name']); ?>
                                            </option>
                                        <?php endforeach; ?>
                                    </select>
                                </div>
                                <div class="mb-3">
                                    <label for="total_billed" class="form-label">Total Billed</label>
                                    <input type="number" step="0.01" id="total_billed" name="total_billed" class="form-control" value="<?php echo htmlspecialchars($record['total_billed']); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="total_paid" class="form-label">Total Paid</label>
                                    <input type="number" step="0.01" id="total_paid" name="total_paid" class="form-control" value="<?php echo htmlspecialchars($record['total_paid']); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="balance" class="form-label">Balance</label>
                                    <input type="number" step="0.01" id="balance" name="balance" class="form-control" value="<?php echo htmlspecialchars($record['balance']); ?>" required>
                                </div>
                                <div class="mb-3">
                                    <label for="due_date" class="form-label">Due Date</label>
                                    <input type="date" id="due_date" name="due_date" class="form-control" value="<?php echo htmlspecialchars($record['due_date']); ?>" required>
                                </div>
                                <button type="submit" class="btn btn-primary">Save Changes</button>
                                <a href="manage_financials.php" class="btn btn-outline-primary">Cancel</a>
                            </form>
                        </div>
                    </div>
                </section>
            <?php else: ?>
                <div class="alert alert-danger">No financial record found.</div>
            <?php endif; ?>
        </main>
    </div>

    <footer class="bg-dark text-white text-center py-4">
        <div class="container">
            <p>&copy; <?php echo date("Y"); ?> Gakero E.L.C.K High School. All rights reserved.</p>
            <p>Kisii County, Gucha Sub-county, Misesi Location</p>
            <p>Email: <a href="mailto:info@gakeroelckhigh.edu" class="text-white">info@gakeroelckhigh.edu</a> | Phone: <a href="tel:+1234567890" class="text-white">(123) 456-7890</a></p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js" integrity="sha384-geWF76RCwLtnZ8qwWowPQNguL3RmwHVBC9FhGdlKrxdiJJigb/j/68SIy3Te4Bkz" crossorigin="anonymous"></script>
</body>
</html>