<?php
session_start();
session_regenerate_id(true);

// Enforce HTTPS
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    header("Location: https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
    exit();
}

// Disable debug in production
if ($_SERVER['SERVER_NAME'] !== 'localhost') {
    ini_set('display_errors', 0);
    error_reporting(E_ALL & ~E_NOTICE);
}

// Check if user is a student
if (!isset($_SESSION['student_id']) || $_SESSION['role'] !== 'student') {
    $redirect = isset($_SESSION['role']) && $_SESSION['role'] === 'admin' ? 'admin_page.php' : 'login.php';
    header("Location: $redirect?error=" . urlencode("Access denied. Students only."));
    exit();
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// Database connection
$servername = "localhost";
$username = "root";
$password = "";
$dbname = "school_db";

try {
    $conn = new PDO("mysql:host=$servername;dbname=$dbname", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    error_log("Connection failed: " . $e->getMessage());
    $error = "Connection failed. Please contact support.";
}

// Fetch student's details
$user = [];
try {
    $stmt = $conn->prepare("SELECT name, grade, contact, admission_no, programme, stage FROM students WHERE id = :student_id");
    $stmt->execute([':student_id' => (int)$_SESSION['student_id']]);
    $user = $stmt->fetch(PDO::FETCH_ASSOC);
    if (!$user) {
        error_log("No user data found for student_id: " . $_SESSION['student_id']);
        $error = "User data not found.";
    }
} catch(PDOException $e) {
    error_log("User fetch error: " . $e->getMessage());
    $error = "Error fetching user data.";
}

// Fetch student's applications
$applications = [];
try {
    $application_stmt = $conn->prepare("SELECT id, course_name, exam_code, reason, status, submitted_date FROM special_exams_applications WHERE student_id = :student_id ORDER BY submitted_date DESC");
    $application_stmt->execute([':student_id' => (int)$_SESSION['student_id']]);
    $applications = $application_stmt->fetchAll(PDO::FETCH_ASSOC) ?: [];
} catch(PDOException $e) {
    error_log("Applications fetch error: " . $e->getMessage());
    $error = "Error fetching applications.";
}

// Handle submit application
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['action']) && $_POST['action'] === 'submit_application' && $_POST['csrf_token'] === $csrf_token) {
    $course_name = trim(filter_input(INPUT_POST, 'course_name', FILTER_SANITIZE_STRING));
    $exam_code = trim(filter_input(INPUT_POST, 'exam_code', FILTER_SANITIZE_STRING));
    $reason = trim(filter_input(INPUT_POST, 'reason', FILTER_SANITIZE_STRING));

    if ($course_name && $exam_code && $reason) {
        try {
            $stmt = $conn->prepare("INSERT INTO special_exams_applications (student_id, course_name, exam_code, reason, status, submitted_date) VALUES (:student_id, :course_name, :exam_code, :reason, 'pending', NOW())");
            $stmt->execute([
                ':student_id' => (int)$_SESSION['student_id'],
                ':course_name' => $course_name,
                ':exam_code' => $exam_code,
                ':reason' => $reason
            ]);
            $success = "Application submitted successfully.";
            header("Location: special_exams.php?success=" . urlencode($success));
            exit();
        } catch(PDOException $e) {
            error_log("Submit application error: " . $e->getMessage());
            $error = "Error submitting application.";
        }
    } else {
        $error = "All fields are required.";
    }
}

// Handle logout
if (isset($_POST['logout']) && $_POST['csrf_token'] === $csrf_token) {
    session_unset();
    session_destroy();
    header("Location: login.php");
    exit();
}

// Handle redirect messages
$message = '';
$alert_type = '';
if (isset($_GET['success'])) {
    $message = htmlspecialchars($_GET['success']);
    $alert_type = 'alert-success';
} elseif (isset($_GET['error'])) {
    $message = htmlspecialchars($_GET['error']);
    $alert_type = 'alert-danger';
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="Special Exams Application for Gakero E.L.C.K High School, apply and view special exam requests.">
    <meta name="keywords" content="Gakero E.L.C.K High School, special exams, student portal">
    <meta name="author" content="Gakero E.L.C.K High School">
    <title>Special Exams Application - Gakero E.L.C.K High School</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600&display=swap" rel="stylesheet">
    <style>
        * { box-sizing: border-box; }
        body { font-family: 'Poppins', sans-serif; background-color: #f5f6f5; color: #333; margin: 0; overflow-x: hidden; }
        .navbar { background-color: #003087; padding: 8px 0; box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1); z-index: 1001; position: sticky; top: 0; }
        .navbar-brand { font-size: 1.25rem; font-weight: 600; color: #fff !important; }
        .nav-link { font-size: 0.9rem; color: #fff !important; padding: 6px 12px; transition: color 0.3s ease, background-color 0.3s ease; }
        .nav-link:hover { color: #a9d6e5 !important; background-color: rgba(255, 255, 255, 0.1); border-radius: 4px; }
        .navbar-toggler { border-color: rgba(255, 255, 255, 0.5); }
        .sidebar { position: fixed; top: 48px; left: 0; width: 250px; height: calc(100vh - 48px); background-color: #003087; color: #fff; overflow-y: auto; z-index: 1000; box-shadow: 2px 0 4px rgba(0, 0, 0, 0.1); transition: transform 0.3s ease; }
        .sidebar-header { border-bottom: 1px solid rgba(255, 255, 255, 0.2); }
        .sidebar .nav-link { font-size: 0.95rem; padding: 10px 15px; color: #fff; transition: background-color 0.3s ease, padding-left 0.3s ease; }
        .sidebar .nav-link:hover { background-color: rgba(255, 255, 255, 0.1); padding-left: 20px; }
        .profile-card { background: rgba(255, 255, 255, 0.05); border-radius: 8px; margin: 15px; padding: 15px; }
        .avatar { width: 80px; height: 80px; display: flex; align-items: center; justify-content: center; border-radius: 50%; }
        main { margin-left: 250px; padding: 15px; min-height: calc(100vh - 48px); background-color: #fff; border-radius: 8px; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05); z-index: 999; width: calc(100% - 250px); }
        footer { background-color: #003087; color: #fff; padding: 20px 0; margin-top: 20px; z-index: 999; width: 100%; }
        footer a { color: #a9d6e5; text-decoration: none; transition: color 0.3s ease; }
        footer a:hover { color: #fff; }
        .hero-section { background: linear-gradient(135deg, #003087, #005f73); border-radius: 8px; padding: 20px; margin-bottom: 2rem; }
        .hero-section h1 { font-size: 1.75rem; font-weight: 600; }
        .hero-section p { font-size: 1rem; }
        .section-title { font-size: 1.75rem; color: #003087; margin-bottom: 1rem; position: relative; }
        .section-title::after { content: ''; display: block; width: 50px; height: 3px; background: #003087; margin-top: 0.5rem; }
        .card { border: none; border-radius: 8px; box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1); transition: transform 0.3s ease, box-shadow 0.3s ease; }
        .card:hover { transform: translateY(-5px); box-shadow: 0 6px 16px rgba(0, 0, 0, 0.2); }
        .card-title { font-size: 1.25rem; color: #003087; margin-bottom: 0.75rem; }
        .card-body { padding: 1.5rem; }
        .btn-primary { background-color: #003087; border: none; padding: 0.75rem 1.5rem; border-radius: 8px; transition: background-color 0.3s ease; }
        .btn-primary:hover { background-color: #00205b; }
        .table { margin-top: 1rem; }
        .table th { background-color: #e9ecef; color: #003087; }
        .form-control { border-radius: 8px; border: 1px solid #ced4da; transition: border-color 0.3s ease, box-shadow 0.3s ease; cursor: text; }
        .form-control:focus { border-color: #003087; box-shadow: 0 0 5px rgba(0, 48, 135, 0.5); outline: 2px solid #003087; outline-offset: 2px; }
        .form-label { color: #003087; }
        .alert { border-radius: 8px; font-size: 0.9rem; }
        @media (max-width: 991px) {
            .sidebar { display: none; }
            main { margin-left: 0; width: 100%; }
            .navbar-brand { font-size: 1.1rem; }
            .nav-link { font-size: 0.85rem; }
            .hero-section { padding: 2rem 0; }
            .hero-section h1 { font-size: 1.5rem; }
            .section-title { font-size: 1.5rem; }
        }
        @media (min-width: 992px) {
            .sidebar { display: block; }
            main { margin-left: 250px; width: calc(100% - 250px); }
        }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg bg-primary text-white">
        <div class="container-fluid">
            <a class="navbar-brand" href="index.php">Gakero E.L.C.K High School</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item"><a class="nav-link" href="index.php">Home</a></li>
                    <li class="nav-item"><a class="nav-link" href="about.php">About</a></li>
                    <li class="nav-item"><a class="nav-link" href="contact.php">Contact</a></li>
                    <li class="nav-item"><a class="nav-link" href="news.php">News</a></li>
                    <li class="nav-item">
                        <form method="POST" action="special_exams.php" style="display:inline;">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                            <button type="submit" name="logout" class="nav-link btn btn-link text-white">Logout</button>
                        </form>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="d-flex">
        <nav class="sidebar bg-primary text-white" id="sidebar">
            <div class="sidebar-header text-center p-3">
                <h4 class="mb-0">Gakero E.L.C.K</h4>
            </div>
            <div class="profile-card">
                <div class="text-center">
                    <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2" role="img" aria-label="Student profile icon">
                        <i class="bi bi-person-circle fs-2"></i>
                    </div>
                    <h6 class="mb-1"><?php echo htmlspecialchars($user['name'] ?? 'Student'); ?></h6>
                    <small class="text-white-50">ID: <?php echo htmlspecialchars($user['admission_no'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Programme: <?php echo htmlspecialchars($user['programme'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Stage: <?php echo htmlspecialchars($user['stage'] ?? 'N/A'); ?></small>
                </div>
            </div>
            <h5 class="p-3">MAIN NAVIGATION</h5>
            <ul class="nav flex-column p-3">
                <li class="nav-item">
                    <a class="nav-link text-white" href="student_dashboard.php"><i class="bi bi-house-door me-2"></i>Dashboard</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Accommodation</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_academics.php"><i class="bi bi-book me-2"></i>Academics</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_clearance.php"><i class="bi bi-check-circle me-2"></i>Clearance</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_contact_messages.php"><i class="bi bi-envelope me-2"></i>Contact Messages</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_financials.php"><i class="bi bi-wallet me-2"></i>Financials</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>News</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="manage_resources.php"><i class="bi bi-file-earmark me-2"></i>Resources</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white active" href="special_exams.php"><i class="bi bi-pencil-square me-2"></i>Special Exams</a>
                </li>
                <li class="nav-item">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                        <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                    </form>
                </li>
            </ul>
        </nav>

        <main class="flex-grow-1 p-4">
            <button class="btn btn-primary d-lg-none mb-3" type="button" data-bs-toggle="offcanvas" data-bs-target="#sidebarOffcanvas" aria-controls="sidebarOffcanvas">
                <i class="bi bi-list"></i> Menu
            </button>
            <div class="offcanvas offcanvas-start bg-primary text-white" tabindex="-1" id="sidebarOffcanvas" aria-labelledby="sidebarOffcanvasLabel">
                <div class="offcanvas-header">
                    <h5 class="offcanvas-title" id="sidebarOffcanvasLabel">Gakero E.L.C.K</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
                </div>
                <div class="offcanvas-body">
                    <div class="profile-card">
                        <div class="text-center">
                            <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2" role="img" aria-label="Student profile icon">
                                <i class="bi bi-person-circle fs-2"></i>
                            </div>
                            <h6 class="mb-1"><?php echo htmlspecialchars($user['name'] ?? 'Student'); ?></h6>
                            <small class="text-white-50">ID: <?php echo htmlspecialchars($user['admission_no'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Programme: <?php echo htmlspecialchars($user['programme'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Stage: <?php echo htmlspecialchars($user['stage'] ?? 'N/A'); ?></small>
                        </div>
                    </div>
                    <h5>MAIN NAVIGATION</h5>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="student_dashboard.php"><i class="bi bi-house-door me-2"></i>Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2"></i>Accommodation</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_academics.php"><i class="bi bi-book me-2"></i>Academics</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_clearance.php"><i class="bi bi-check-circle me-2"></i>Clearance</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_contact_messages.php"><i class="bi bi-envelope me-2"></i>Contact Messages</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_financials.php"><i class="bi bi-wallet me-2"></i>Financials</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_news.php"><i class="bi bi-newspaper me-2"></i>News</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="manage_resources.php"><i class="bi bi-file-earmark me-2"></i>Resources</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white active" href="special_exams.php"><i class="bi bi-pencil-square me-2"></i>Special Exams</a>
                        </li>
                        <li class="nav-item">
                            <form method="POST">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                                <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2"></i>Logout</button>
                            </form>
                        </li>
                    </ul>
                </div>
            </div>

            <header class="hero-section text-white text-center py-4 rounded mb-4">
                <h1 class="display-5">Special Exams Application</h1>
                <p class="lead">Apply and view special exam requests</p>
                <small class="text-white-50">Online</small>
            </header>

            <?php if ($message): ?>
                <div class="alert <?php echo $alert_type; ?> alert-dismissible fade show" role="alert">
                    <?php echo $message; ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>
            <?php if (isset($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <section class="mb-5">
                <h2 class="section-title">Apply for Special Exam</h2>
                <div class="card">
                    <div class="card-body">
                        <form method="POST" id="submitApplicationForm">
                            <input type="hidden" name="action" value="submit_application">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                            <div class="mb-3">
                                <label for="course_name" class="form-label">Course Name</label>
                                <input type="text" id="course_name" name="course_name" class="form-control" required aria-describedby="course_name_help">
                                <small id="course_name_help" class="form-text text-muted">Enter the course name (e.g., Mathematics).</small>
                            </div>
                            <div class="mb-3">
                                <label for="exam_code" class="form-label">Exam Code</label>
                                <input type="text" id="exam_code" name="exam_code" class="form-control" required aria-describedby="exam_code_help">
                                <small id="exam_code_help" class="form-text text-muted">Enter the exam code (e.g., MATH2023).</small>
                            </div>
                            <div class="mb-3">
                                <label for="reason" class="form-label">Reason</label>
                                <textarea id="reason" name="reason" class="form-control" rows="5" required aria-describedby="reason_help"></textarea>
                                <small id="reason_help" class="form-text text-muted">Provide the reason for the special exam.</small>
                            </div>
                            <button type="submit" class="btn btn-primary">Submit Application</button>
                        </form>
                    </div>
                </div>
            </section>

            <section class="mb-5">
                <h2 class="section-title">Your Applications</h2>
                <div class="table-responsive">
                    <table class="table table-bordered table-striped">
                        <thead>
                            <tr>
                                <th scope="col">Course Name</th>
                                <th scope="col">Exam Code</th>
                                <th scope="col">Reason</th>
                                <th scope="col">Status</th>
                                <th scope="col">Submitted Date</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php if (!empty($applications)): ?>
                                <?php foreach ($applications as $application): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($application['course_name']); ?></td>
                                        <td><?php echo htmlspecialchars($application['exam_code']); ?></td>
                                        <td><?php echo htmlspecialchars(substr($application['reason'], 0, 50)) . (strlen($application['reason']) > 50 ? '...' : ''); ?></td>
                                        <td><?php echo htmlspecialchars($application['status']); ?></td>
                                        <td><?php echo date('F j, Y H:i', strtotime($application['submitted_date'])); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            <?php else: ?>
                                <tr><td colspan="5" class="text-center">No applications found.</td></tr>
                            <?php endif; ?>
                        </tbody>
                    </table>
                </div>
            </section>
        </main>
    </div>

    <footer class="bg-dark text-white text-center py-4">
        <div class="container">
            <p>&copy; <?php echo date("Y"); ?> Gakero E.L.C.K High School. All rights reserved.</p>
            <p>Kisii County, Gucha Sub-county, Misesi Location</p>
            <p>Email: <a href="mailto:info@gakeroelckhigh.edu" class="text-white">info@gakeroelckhigh.edu</a> | Phone: <a href="tel:+1234567890" class="text-white">(123) 456-7890</a></p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
    <script>
        // Client-side validation for submit form
        document.getElementById('submitApplicationForm').addEventListener('submit', function(e) {
            const courseName = document.getElementById('course_name').value.trim();
            const examCode = document.getElementById('exam_code').value.trim();
            const reason = document.getElementById('reason').value.trim();
            if (!courseName || !examCode || !reason) {
                e.preventDefault();
                alert('Please fill in all fields correctly.');
            }
        });

        // Focus on first input
        document.addEventListener('DOMContentLoaded', function () {
            const firstInput = document.querySelector('input, textarea');
            if (firstInput) {
                firstInput.focus();
            }
        });
    </script>
</body>
</html>