<?php
session_start();
session_regenerate_id(true); // Prevent session fixation

// Enforce HTTPS
if (!isset($_SERVER['HTTPS']) || $_SERVER['HTTPS'] !== 'on') {
    header("Location: https://" . $_SERVER['HTTP_HOST'] . $_SERVER['REQUEST_URI']);
    exit();
}

// Disable debug in production
if ($_SERVER['SERVER_NAME'] !== 'localhost') {
    ini_set('display_errors', 0);
    error_reporting(0);
}

// Load configuration (assumes config.php is outside web root)
//reire_once '/path/to/config.php'; // Contains $servername, $username, $password, $dbname
// Example config.php content:
 $servername = "localhost";
 $username = "root";
 $password = "";
 $dbname = "school_db";

// Check if student is logged in
if (!isset($_SESSION['student_id']) || $_SESSION['role'] !== 'student') {
    header("Location: student_login.php");
    exit();
}

// Database connection
try {
    $conn = new PDO("mysql:host=$servername;dbname=$dbname", $username, $password);
    $conn->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch(PDOException $e) {
    error_log("Connection failed: " . $e->getMessage());
    die("Unable to connect to the database. Please try again later.");
}

// Helper function for executing queries
function executeQuery($conn, $query, $params = []) {
    try {
        $stmt = $conn->prepare($query);
        $stmt->execute($params);
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    } catch(PDOException $e) {
        error_log("Query error: " . $e->getMessage());
        return false;
    }
}

// Fetch student details
$student = executeQuery($conn, "SELECT name, grade, contact, admission_no, id_passport, gender, dob, phone, email, postal_address, programme, stage FROM students WHERE id = :student_id", [':student_id' => $_SESSION['student_id']]);
if (!$student) {
    $error = "Student data not found. Please contact support.";
} else {
    $student = $student[0]; // Fetch first row
}

// Fetch financial details
$financials = executeQuery($conn, "SELECT total_billed, total_paid, balance FROM financials WHERE student_id = :student_id", [':student_id' => $_SESSION['student_id']]);
if (!$financials) {
    $error = "Financial data not available. Please contact the finance office.";
} else {
    $financials = $financials[0];
}

// Fetch recent news (latest 3)
$news = executeQuery($conn, "SELECT id, title, content, image, date_posted FROM news ORDER BY date_posted DESC LIMIT 3");
if ($news === false) {
    $error = "Error fetching news. Please try again later.";
}

// Fetch timetable for student's grade
if (isset($student['grade'])) {
    $timetable = executeQuery($conn, "SELECT day, period, subject, teacher FROM timetables WHERE class = :grade ORDER BY FIELD(day, 'Monday', 'Tuesday', 'Wednesday', 'Thursday', 'Friday'), period", [':grade' => $student['grade']]);
    if ($timetable === false) {
        $error = "Error fetching timetable. Please try again later.";
    }
} else {
    $timetable = [];
    $error = "Grade information not available for timetable.";
}

// Generate CSRF token
if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
$csrf_token = $_SESSION['csrf_token'];

// Handle logout
if (isset($_POST['logout']) && $_POST['csrf_token'] === $csrf_token) {
    error_log("User {$_SESSION['student_id']} logged out.");
    session_unset();
    session_destroy();
    header("Location: student_login.php");
    exit();
}

// Dynamic footer contact info (could be fetched from DB or config)
$contact_email = "info@Cypher.ac.ke";
$contact_phone = "+254123456789";
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="description" content="Student portal for Cypher University, providing access to financials, academics, and student services.">
    <meta name="keywords" content="Cypher University, student portal, financials, academics">
    <meta name="author" content="Cypher University">
    <title>Student Portal - Cypher University</title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/css/bootstrap.min.css" rel="stylesheet" integrity="sha384-QWTKZyjpPEjISv5WaRU9OFeRpok6YctnYmDr5pNlyT2bRjXh0JMhjY6hW+ALEwIH" crossorigin="anonymous">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css">
    <link href="https://fonts.googleapis.com/css2?family=Poppins:wght@300;400;600&display=swap" rel="stylesheet">
    <style>
        /* General Styling */
        body {
            font-family: 'Poppins', sans-serif;
            background-color: #f5f6f5;
            color: #333;
            margin: 0;
            overflow-x: hidden;
        }

        /* Compact Navbar */
        .navbar {
            background-color: #003087;
            padding: 8px 0;
            box-shadow: 0 2px 4px rgba(0, 0, 0, 0.1);
            z-index: 1001;
        }
        .navbar-brand {
            font-size: 1.25rem;
            font-weight: 600;
            color: #fff !important;
        }
        .nav-link {
            font-size: 0.9rem;
            color: #fff !important;
            padding: 6px 12px;
            transition: color 0.3s ease, background-color 0.3s ease;
        }
        .nav-link:hover {
            color: #a9d6e5 !important;
            background-color: rgba(255, 255, 255, 0.1);
            border-radius: 4px;
        }
        .navbar-toggler {
            border-color: rgba(255, 255, 255, 0.5);
        }

        /* Fixed Sidebar */
        .sidebar {
            position: fixed;
            top: 48px;
            left: 0;
            width: 250px;
            height: calc(100vh - 48px);
            background-color: #003087;
            color: #fff;
            overflow-y: auto;
            z-index: 1000;
            box-shadow: 2px 0 4px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease;
        }
        .sidebar-header {
            border-bottom: 1px solid rgba(255, 255, 255, 0.2);
        }
        .sidebar .nav-link {
            font-size: 0.95rem;
            padding: 10px 15px;
            color: #fff;
            transition: background-color 0.3s ease, padding-left 0.3s ease;
        }
        .sidebar .nav-link:hover {
            background-color: rgba(255, 255, 255, 0.1);
            padding-left: 20px;
        }
        .profile-card {
            background: rgba(255, 255, 255, 0.05);
            border-radius: 8px;
            margin: 15px;
            padding: 15px;
        }
        .avatar {
            width: 80px;
            height: 80px;
            display: flex;
            align-items: center;
            justify-content: center;
            border-radius: 50%;
        }

        /* Scrollable Main Content */
        main {
            margin-left: 250px;
            padding: 20px;
            min-height: calc(100vh - 48px - 80px);
            overflow-y: auto;
            background-color: #fff;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.05);
        }

        /* Static Footer */
        footer {
            position: relative;
            background-color: #003087;
            color: #fff;
            padding: 20px 0;
            margin-top: 20px;
        }
        footer a {
            color: #a9d6e5;
            text-decoration: none;
            transition: color 0.3s ease;
        }
        footer a:hover {
            color: #fff;
        }

        /* Hero Section */
        .hero-section {
            background: linear-gradient(135deg, #003087, #005f73);
            border-radius: 8px;
            padding: 20px;
            margin-bottom: 2rem;
        }
        .hero-section h1 {
            font-size: 1.75rem;
            font-weight: 600;
        }
        .hero-section p {
            font-size: 1rem;
        }

        /* Section Titles */
        .section-title {
            font-size: 1.75rem;
            color: #003087;
            margin-bottom: 1rem;
            position: relative;
        }
        .section-title::after {
            content: '';
            display: block;
            width: 50px;
            height: 3px;
            background: #003087;
            margin-top: 0.5rem;
        }

        /* Cards */
        .card {
            border: none;
            border-radius: 8px;
            box-shadow: 0 2px 8px rgba(0, 0, 0, 0.1);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .card:hover {
            transform: translateY(-5px);
            box-shadow: 0 6px 16px rgba(0, 0, 0, 0.2);
        }
        .card-title {
            font-size: 1.25rem;
            color: #003087;
            margin-bottom: 0.75rem;
        }
        .card-img-top {
            height: 200px;
            object-fit: cover;
            border-top-left-radius: 8px;
            border-top-right-radius: 8px;
        }
        .card-body {
            padding: 1.5rem;
        }
        .card-text {
            font-size: 1rem;
            color: #555;
        }

        /* Buttons */
        .btn-primary {
            background-color: #003087;
            border: none;
            padding: 0.75rem 1.5rem;
            border-radius: 8px;
            transition: background-color 0.3s ease;
        }
        .btn-primary:hover {
            background-color: #00205b;
        }
        .btn-outline-primary {
            border-color: #003087;
            color: #003087;
        }
        .btn-outline-primary:hover {
            background-color: #003087;
            color: #fff;
        }

        /* Tables */
        .table {
            margin-top: 1rem;
        }
        .table th {
            background-color: #e9ecef;
            color: #003087;
        }

        /* News Section */
        .news-section {
            background: linear-gradient(135deg, #ffffff, #e9ecef);
            padding: 3rem 0;
            border-radius: 10px;
            box-shadow: 0 4px 12px rgba(0, 0, 0, 0.15);
        }
        .news-card .card {
            border: none;
            border-radius: 8px;
        }

        /* Responsive Adjustments */
        @media (max-width: 991px) {
            .sidebar {
                display: none;
            }
            main {
                margin-left: 0;
            }
            .navbar-brand {
                font-size: 1.1rem;
            }
            .nav-link {
                font-size: 0.85rem;
            }
            .hero-section {
                padding: 2rem 0;
            }
            .hero-section h1 {
                font-size: 1.5rem;
            }
            .section-title {
                font-size: 1.5rem;
            }
            .card-img-top {
                height: 150px;
            }
        }
    </style>
</head>
<body>
    <!-- Sticky Navigation Bar -->
    <nav class="navbar navbar-expand-lg navbar-dark sticky-top" role="navigation">
        <div class="container">
            <a class="navbar-brand" href="index.php">Cypher University</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php">Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="about.php">About</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="contact.php">Contact</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link" href="news.php">News</a>
                    </li>
                    <li class="nav-item">
                        <form method="POST" action="student_dashboard.php" style="display:inline;">
                            <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                            <button type="submit" name="logout" class="nav-link btn btn-link text-white">Logout</button>
                        </form>
                    </li>
                </ul>
            </div>
        </div>
    </nav>

    <div class="d-flex">
        <!-- Fixed Sidebar -->
        <nav class="sidebar bg-primary text-white" id="sidebar" role="navigation" aria-label="Main navigation">
            <div class="sidebar-header text-center p-3">
                <h4 class="mb-0">Cypher University</h4>
            </div>
            <div class="profile-card">
                <div class="text-center">
                    <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2">
                        <i class="bi bi-person-circle fs-2" aria-hidden="true"></i>
                    </div>
                    <h6 class="mb-1"><?php echo htmlspecialchars($student['name'] ?? 'Student'); ?></h6>
                    <small class="text-white-50">Admission No: <?php echo htmlspecialchars($student['admission_no'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Programme: <?php echo htmlspecialchars($student['programme'] ?? 'N/A'); ?></small><br>
                    <small class="text-white-50">Stage: <?php echo htmlspecialchars($student['stage'] ?? 'N/A'); ?></small>
                </div>
            </div>
            <h5 class="p-3">MAIN NAVIGATION</h5>
            <ul class="nav flex-column p-3">
                <li class="nav-item">
                    <a class="nav-link text-white" href="student_dashboard.php"><i class="bi bi-house me-2" aria-hidden="true"></i>Dashboard</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="financials.php"><i class="bi bi-wallet me-2" aria-hidden="true"></i>Financials</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="academics.php"><i class="bi bi-book me-2" aria-hidden="true"></i>Academics</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="student_requests.php"><i class="bi bi-file-text me-2" aria-hidden="true"></i>Student Requests</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2" aria-hidden="true"></i>Accommodation</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="special_exams.php"><i class="bi bi-pencil-square me-2" aria-hidden="true"></i>Special Exams</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="downloads.php"><i class="bi bi-download me-2" aria-hidden="true"></i>Downloads</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="clearance.php"><i class="bi bi-check-circle me-2" aria-hidden="true"></i>Students Clearance</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="transfer.php"><i class="bi bi-arrow-left-right me-2" aria-hidden="true"></i>Inter-School Transfer</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="settings.php"><i class="bi bi-gear me-2" aria-hidden="true"></i>Settings</a>
                </li>
                <li class="nav-item">
                    <form method="POST">
                        <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                        <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2" aria-hidden="true"></i>Logout</button>
                    </form>
                </li>
            </ul>
            <h5 class="p-3">PERIODIC ACTIVITIES</h5>
            <ul class="nav flex-column p-3">
                <li class="nav-item">
                    <a class="nav-link text-white" href="lecturers_evaluation.php"><i class="bi bi-person-check me-2" aria-hidden="true"></i>Lecturers Evaluation</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="msu_election.php"><i class="bi bi-box-seam me-2" aria-hidden="true"></i>MSU Election</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="notice_board.php"><i class="bi bi-megaphone me-2" aria-hidden="true"></i>Notice Board</a>
                </li>
                <li class="nav-item">
                    <a class="nav-link text-white" href="https://facebook.com/Cypher" target="_blank" rel="noopener"><i class="bi bi-facebook me-2" aria-hidden="true"></i>MSU Official Facebook</a>
                </li>
            </ul>
        </nav>

        <!-- Scrollable Main Content -->
        <main class="flex-grow-1 p-4" role="main">
            <button class="btn btn-primary d-lg-none mb-3" type="button" data-bs-toggle="offcanvas" data-bs-target="#sidebarOffcanvas" aria-controls="sidebarOffcanvas">
                <i class="bi bi-list" aria-hidden="true"></i> Menu
            </button>
            <div class="offcanvas offcanvas-start bg-primary text-white" tabindex="-1" id="sidebarOffcanvas" aria-labelledby="sidebarOffcanvasLabel">
                <div class="offcanvas-header">
                    <h5 class="offcanvas-title" id="sidebarOffcanvasLabel">Cypher University</h5>
                    <button type="button" class="btn-close btn-close-white" data-bs-dismiss="offcanvas" aria-label="Close"></button>
                </div>
                <div class="offcanvas-body">
                    <div class="profile-card">
                        <div class="text-center">
                            <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2">
                                <i class="bi bi-person-circle fs-2" aria-hidden="true"></i>
                            </div>
                            <h6 class="mb-1"><?php echo htmlspecialchars($student['name'] ?? 'Student'); ?></h6>
                            <small class="text-white-50">Admission No: <?php echo htmlspecialchars($student['admission_no'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Programme: <?php echo htmlspecialchars($student['programme'] ?? 'N/A'); ?></small><br>
                            <small class="text-white-50">Stage: <?php echo htmlspecialchars($student['stage'] ?? 'N/A'); ?></small>
                        </div>
                    </div>
                    <h5>MAIN NAVIGATION</h5>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="student_dashboard.php"><i class="bi bi-house me-2" aria-hidden="true"></i>Dashboard</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="financials.php"><i class="bi bi-wallet me-2" aria-hidden="true"></i>Financials</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="academics.php"><i class="bi bi-book me-2" aria-hidden="true"></i>Academics</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="student_requests.php"><i class="bi bi-file-text me-2" aria-hidden="true"></i>Student Requests</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="accommodation.php"><i class="bi bi-building me-2" aria-hidden="true"></i>Accommodation</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="special_exams.php"><i class="bi bi-pencil-square me-2" aria-hidden="true"></i>Special Exams</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="downloads.php"><i class="bi bi-download me-2" aria-hidden="true"></i>Downloads</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="clearance.php"><i class="bi bi-check-circle me-2" aria-hidden="true"></i>Students Clearance</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="transfer.php"><i class="bi bi-arrow-left-right me-2" aria-hidden="true"></i>Inter-School Transfer</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="settings.php"><i class="bi bi-gear me-2" aria-hidden="true"></i>Settings</a>
                        </li>
                        <li class="nav-item">
                            <form method="POST">
                                <input type="hidden" name="csrf_token" value="<?php echo htmlspecialchars($csrf_token); ?>">
                                <button type="submit" name="logout" class="nav-link text-white bg-transparent border-0"><i class="bi bi-box-arrow-right me-2" aria-hidden="true"></i>Logout</button>
                            </form>
                        </li>
                    </ul>
                    <h5>PERIODIC ACTIVITIES</h5>
                    <ul class="nav flex-column">
                        <li class="nav-item">
                            <a class="nav-link text-white" href="lecturers_evaluation.php"><i class="bi bi-person-check me-2" aria-hidden="true"></i>Lecturers Evaluation</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="msu_election.php"><i class="bi bi-box-seam me-2" aria-hidden="true"></i>MSU Election</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="notice_board.php"><i class="bi bi-megaphone me-2" aria-hidden="true"></i>Notice Board</a>
                        </li>
                        <li class="nav-item">
                            <a class="nav-link text-white" href="https://facebook.com/Cypher" target="_blank" rel="noopener"><i class="bi bi-facebook me-2" aria-hidden="true"></i>MSU Official Facebook</a>
                        </li>
                    </ul>
                </div>
            </div>

            <!-- Hero Section -->
            <header class="hero-section bg-primary text-white text-center py-4 rounded mb-4">
                <h1 class="display-5">Welcome, <?php echo htmlspecialchars($student['name'] ?? 'Student'); ?>!</h1>
                <p class="lead">Your student portal for Cypher University</p>
                <small class="text-white-50">Online</small>
            </header>

            <?php if (isset($error)): ?>
                <div class="alert alert-danger alert-dismissible fade show" role="alert">
                    <?php echo htmlspecialchars($error); ?>
                    <button type="button" class="btn-close" data-bs-dismiss="alert" aria-label="Close"></button>
                </div>
            <?php endif; ?>

            <!-- Financials Section -->
            <div class="section mb-5">
                <h2 class="section-title">Financial Overview</h2>
                <?php if (isset($financials)): ?>
                    <div class="row g-4">
                        <div class="col-md-4">
                            <div class="card text-center">
                                <div class="card-body">
                                    <h5 class="card-title">Total Billed</h5>
                                    <p class="card-text">Ksh. <?php echo number_format($financials['total_billed'], 2); ?></p>
                                    <a href="financials.php" class="btn btn-outline-primary btn-sm">View Details</a>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card text-center">
                                <div class="card-body">
                                    <h5 class="card-title">Total Paid</h5>
                                    <p class="card-text">Ksh. <?php echo number_format($financials['total_paid'], 2); ?></p>
                                    <a href="financials.php" class="btn btn-outline-primary btn-sm">View Details</a>
                                </div>
                            </div>
                        </div>
                        <div class="col-md-4">
                            <div class="card text-center">
                                <div class="card-body">
                                    <h5 class="card-title">Balance</h5>
                                    <p class="card-text">Ksh. <?php echo number_format($financials['balance'], 2); ?></p>
                                    <a href="financials.php" class="btn btn-outline-primary btn-sm">View Details</a>
                                </div>
                            </div>
                        </div>
                    </div>
                <?php else: ?>
                    <p class="text-muted">Financial data not available. Please contact the finance office.</p>
                <?php endif; ?>
            </div>

            <!-- User Profile Section -->
            <div class="section mb-5">
                <h2 class="section-title">User Profile</h2>
                <div class="card">
                    <div class="card-body">
                        <div class="row">
                            <div class="col-md-3 text-center">
                                <div class="avatar bg-light text-primary rounded-circle mx-auto mb-2">
                                    <i class="bi bi-person-circle fs-1" aria-hidden="true"></i>
                                </div>
                                <h5><?php echo htmlspecialchars($student['name'] ?? 'Student'); ?></h5>
                                <p><strong>Programme:</strong> <?php echo htmlspecialchars($student['programme'] ?? 'N/A'); ?></p>
                                <p><strong>My Stage:</strong> <?php echo htmlspecialchars($student['stage'] ?? 'N/A'); ?></p>
                            </div>
                            <div class="col-md-9">
                                <h5>Personal Information</h5>
                                <div class="row">
                                    <div class="col-md-6">
                                        <p><strong>Admission No:</strong> <?php echo htmlspecialchars($student['admission_no'] ?? 'N/A'); ?></p>
                                        <p><strong>ID/Passport:</strong> <?php echo htmlspecialchars($student['id_passport'] ?? 'N/A'); ?></p>
                                        <p><strong>Full Name:</strong> <?php echo htmlspecialchars($student['name'] ?? 'N/A'); ?></p>
                                        <p><strong>Gender:</strong> <?php echo htmlspecialchars($student['gender'] ?? 'N/A'); ?></p>
                                    </div>
                                    <div class="col-md-6">
                                        <p><strong>Date of Birth:</strong> <?php echo htmlspecialchars($student['dob'] ?? 'N/A'); ?></p>
                                        <p><strong>Phone Number:</strong> <?php echo htmlspecialchars($student['phone'] ?? 'N/A'); ?></p>
                                        <p><strong>Email Address:</strong> <?php echo htmlspecialchars($student['email'] ?? 'N/A'); ?></p>
                                        <p><strong>Postal Address:</strong> <?php echo htmlspecialchars($student['postal_address'] ?? 'N/A'); ?></p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Timetable -->
            <div class="section mb-5">
                <h2 class="section-title">Your Timetable</h2>
                <?php if (!empty($timetable)): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-striped">
                            <thead>
                                <tr>
                                    <th scope="col">Day</th>
                                    <th scope="col">Period</th>
                                    <th scope="col">Subject</th>
                                    <th scope="col">Teacher</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($timetable as $slot): ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($slot['day']); ?></td>
                                        <td><?php echo htmlspecialchars($slot['period']); ?></td>
                                        <td><?php echo htmlspecialchars($slot['subject']); ?></td>
                                        <td><?php echo htmlspecialchars($slot['teacher']); ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-muted">No timetable available for your grade.</p>
                <?php endif; ?>
            </div>

            <!-- Recent News -->
            <div class="section mb-5 news-section">
                <h2 class="section-title">Recent News</h2>
                <div class="row g-4">
                    <?php if (!empty($news)): ?>
                        <?php foreach ($news as $item): ?>
                            <div class="col-md-4 news-card">
                                <div class="card h-100">
                                    <?php if (!empty($item['image']) && filter_var($item['image'], FILTER_VALIDATE_URL)): ?>
                                        <img src="<?php echo htmlspecialchars($item['image']); ?>" class="card-img-top" alt="News image for <?php echo htmlspecialchars($item['title']); ?>" loading="lazy">
                                    <?php endif; ?>
                                    <div class="card-body">
                                        <h5 class="card-title"><?php echo htmlspecialchars($item['title']); ?></h5>
                                        <p class="card-text"><?php echo htmlspecialchars(substr($item['content'], 0, 100)) . (strlen($item['content']) > 100 ? '...' : ''); ?></p>
                                        <p class="card-text"><small class="text-muted">Posted on: <?php echo date('F j, Y', strtotime($item['date_posted'])); ?></small></p>
                                        <a href="news-detail.php?id=<?php echo htmlspecialchars($item['id']); ?>" class="btn btn-outline-primary btn-sm">Read More</a>
                                    </div>
                                </div>
                            </div>
                        <?php endforeach; ?>
                    <?php else: ?>
                        <div class="col-12 text-center">
                            <p class="text-muted">No news articles available at this time.</p>
                        </div>
                    <?php endif; ?>
                </div>
            </div>
        </main>
    </div>

    <!-- Footer -->
    <footer class="bg-dark text-white text-center py-4" role="contentinfo">
        <div class="container">
            <p>&copy; <?php echo date("Y"); ?> Cypher University. Powered by AppKings Solutions Limited.</p>
            <p>Email: <a href="mailto:<?php echo htmlspecialchars($contact_email); ?>" class="text-white"><?php echo htmlspecialchars($contact_email); ?></a> | Phone: <a href="tel:<?php echo htmlspecialchars($contact_phone); ?>" class="text-white"><?php echo htmlspecialchars($contact_phone); ?></a></p>
        </div>
    </footer>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js" integrity="sha384-YvpcrYf0tY3lHB60NNkmXc5s9fDVZLESaAA55NDzOxhy9GkcIdslK1eN7N6jIeHz" crossorigin="anonymous"></script>
</body>
</html>