<?php
// login.php - Enhanced Professional Login UI with Centered Layout, Audit Logging, and Role-Based Redirection
session_start();
include 'db.php';

if (isset($_SESSION['user_id'])) {
    // Redirect based on role
    $role = $_SESSION['role'];
    if ($role == 'sales') {
        header('Location: sales_dashboard.php');
    } elseif ($role == 'tech') {
        header('Location: tech_dashboard.php');
    } elseif ($role == 'technician') {
        header('Location: technician_dashboard.php');
    } elseif ($role == 'superuser') {
        header('Location: main.php');
    }
    exit;
}

if ($_SERVER['REQUEST_METHOD'] == 'POST') {
    if (!hash_equals($_SESSION['csrf_token'] ?? '', $_POST['csrf_token'] ?? '')) {
        $error = "Invalid request";
    } else {
        $username = trim($_POST['username']);
        $password = $_POST['password'];

        // Basic input validation
        if (empty($username) || empty($password)) {
            $error = "Username and password are required";
        } else {
            $stmt = $conn->prepare("SELECT * FROM users WHERE username = ? AND password = ?");
            $stmt->bind_param("ss", $username, $password);
            $stmt->execute();
            $result = $stmt->get_result();

            if ($result->num_rows > 0) {
                $row = $result->fetch_assoc();
                $_SESSION['user_id'] = $row['id'];
                $_SESSION['username'] = $row['username'];
                $_SESSION['role'] = $row['role'];
                $_SESSION['region_id'] = $row['region_id'];
                $_SESSION['csrf_token'] = bin2hex(random_bytes(32));

                log_audit($conn, $row['id'], 'Login', 'User logged in');

                // Role-based redirection
                if ($row['role'] == 'sales') {
                    header("Location: sales_dashboard.php");
                } elseif ($row['role'] == 'tech') {
                    header("Location: tech_dashboard.php");
                } elseif ($row['role'] == 'technician') {
                    header("Location: technician_dashboard.php");
                } elseif ($row['role'] == 'superuser') {
                    header("Location: main.php");
                }
                exit();
            } else {
                $error = "Invalid username or password.";
                log_audit($conn, null, 'Failed Login', "Failed login attempt for username: $username");
            }
            $stmt->close();
        }
    }
}

if (!isset($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Vehicle Tracking System - Login</title>
    <link href="https://cdn.jsdelivr.net/npm/tailwindcss@2.2.19/dist/tailwind.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css" rel="stylesheet">
    <style>
        .gradient-bg {
            background: linear-gradient(135deg, #1e3a8a, #60a5fa);
        }
        .login-card {
            animation: fadeIn 0.5s ease-in-out;
        }
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(20px); }
            to { opacity: 1; transform: translateY(0); }
        }
        .input-focus {
            transition: all 0.3s ease;
        }
        .input-focus:focus {
            border-color: #2563eb;
            box-shadow: 0 0 0 4px rgba(37, 99, 235, 0.1);
        }
        .btn-primary:hover {
            transform: translateY(-2px);
            transition: all 0.3s ease;
        }
        .toggle-password:hover {
            cursor: pointer;
            color: #2563eb;
        }
    </style>
</head>
<body class="gradient-bg min-h-screen flex items-center justify-center p-4">
    <div class="login-card max-w-md w-full bg-white rounded-2xl shadow-2xl p-8">
        <div class="text-center mb-6">
            <i class="fas fa-car fa-3x text-blue-600 mb-4"></i>
            <h3 class="text-2xl font-bold text-gray-800">Vehicle Tracking System</h3>
            <p class="text-gray-500">Secure Login Portal</p>
        </div>
        <?php if (isset($error)): ?>
            <div class="bg-red-100 border-l-4 border-red-500 text-red-700 p-4 rounded mb-6" role="alert">
                <p><?php echo htmlspecialchars($error); ?></p>
            </div>
        <?php endif; ?>
        <form method="POST" id="loginForm" onsubmit="handleSubmit(event)">
            <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
            <div class="mb-5">
                <label for="username" class="block text-sm font-medium text-gray-700 mb-2">
                    <i class="fas fa-user mr-2"></i>Username
                </label>
                <input
                    type="text"
                    id="username"
                    name="username"
                    class="input-focus w-full p-3 border border-gray-300 rounded-lg focus:outline-none"
                    placeholder="Enter your username"
                    required
                    aria-required="true"
                    aria-describedby="username-error"
                >
                <p id="username-error" class="text-red-500 text-sm mt-1 hidden">Username is required</p>
            </div>
            <div class="mb-5 relative">
                <label for="password" class="block text-sm font-medium text-gray-700 mb-2">
                    <i class="fas fa-lock mr-2"></i>Password
                </label>
                <input
                    type="password"
                    id="password"
                    name="password"
                    class="input-focus w-full p-3 border border-gray-300 rounded-lg focus:outline-none pr-10"
                    placeholder="Enter your password"
                    required
                    aria-required="true"
                    aria-describedby="password-error"
                >
                <i class="fas fa-eye toggle-password absolute right-3 top-10 text-gray-500" onclick="togglePassword()"></i>
                <p id="password-error" class="text-red-500 text-sm mt-1 hidden">Password is required</p>
            </div>
            <div class="flex items-center justify-between mb-5">
                <div class="flex items-center">
                    <input
                        id="rememberMe"
                        type="checkbox"
                        class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
                    >
                    <label for="rememberMe" class="ml-2 text-sm text-gray-600">Remember me</label>
                </div>
                <a href="#" class="text-sm text-blue-600 hover:underline">Forgot password?</a>
            </div>
            <button
                type="submit"
                class="btn-primary w-full bg-blue-600 text-white p-3 rounded-lg font-semibold hover:bg-blue-700 disabled:opacity-50"
                id="submitBtn"
            >
                <i class="fas fa-sign-in-alt mr-2"></i>Sign In
            </button>
        </form>
        <div class="text-center mt-6">
            <p class="text-sm text-gray-500">
                Don't have an account? <a href="#" class="text-blue-600 font-semibold hover:underline">Contact Admin</a>
            </p>
        </div>
    </div>
    <script>
        function togglePassword() {
            const passwordInput = document.getElementById('password');
            const toggleIcon = document.querySelector('.toggle-password');
            if (passwordInput.type === 'password') {
                passwordInput.type = 'text';
                toggleIcon.classList.remove('fa-eye');
                toggleIcon.classList.add('fa-eye-slash');
            } else {
                passwordInput.type = 'password';
                toggleIcon.classList.remove('fa-eye-slash');
                toggleIcon.classList.add('fa-eye');
            }
        }

        function handleSubmit(event) {
            event.preventDefault();
            const username = document.getElementById('username').value.trim();
            const password = document.getElementById('password').value;
            const usernameError = document.getElementById('username-error');
            const passwordError = document.getElementById('password-error');
            const submitBtn = document.getElementById('submitBtn');

            usernameError.classList.add('hidden');
            passwordError.classList.add('hidden');

            let isValid = true;
            if (!username) {
                usernameError.classList.remove('hidden');
                isValid = false;
            }
            if (!password) {
                passwordError.classList.remove('hidden');
                isValid = false;
            }

            if (isValid) {
                submitBtn.disabled = true;
                submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i>Signing In...';
                document.getElementById('loginForm').submit();
            }
        }
    </script>
</body>
</html>