<?php
// main.php - Superuser Dashboard with Integrated Sales, Tech, and Technician Functionalities
include 'auth.php';

// Verify role
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'superuser') {
    echo "Access Denied.";
    exit();
}

// Verify session user_id
if (!isset($_SESSION['user_id']) || !is_numeric($_SESSION['user_id'])) {
    die("Error: Invalid or missing user ID in session.");
}

$user_id = $_SESSION['user_id'];

// Define log_audit and send_notification if not in auth.php
if (!function_exists('log_audit')) {
    function log_audit($conn, $user_id, $action, $details) {
        $stmt = $conn->prepare("INSERT INTO audit_logs (user_id, action, details, timestamp) VALUES (?, ?, ?, NOW())");
        if ($stmt === false) return;
        $stmt->bind_param("iss", $user_id, $action, $details);
        $stmt->execute();
        $stmt->close();
    }
}

if (!function_exists('send_notification')) {
    function send_notification($conn, $user_id, $message, $related_id = null, $related_tab = null) {
        $stmt = $conn->prepare("INSERT INTO notifications (user_id, message, related_id, related_tab, created_at) VALUES (?, ?, ?, ?, NOW())");
        if ($stmt === false) return;
        $stmt->bind_param("isis", $user_id, $message, $related_id, $related_tab);
        $stmt->execute();
        $stmt->close();
    }
}

// Handle Mark Notification Read
if (isset($_GET['mark_read'])) {
    $notif_id = $_GET['mark_read'];
    $tab = isset($_GET['tab']) ? $_GET['tab'] : 'notifications';
    $client_id = isset($_GET['client_id']) ? $_GET['client_id'] : null;
    $stmt = $conn->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("ii", $notif_id, $user_id);
    $stmt->execute();
    $stmt->close();
    $redirect_url = "main.php?tab=" . urlencode($tab);
    if ($client_id) $redirect_url .= "&highlight=" . urlencode($client_id);
    header("Location: $redirect_url");
    exit();
}

// Handle Add Client (Sales)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['add_client'])) {
    $name = $_POST['name'];
    $contact_email = $_POST['contact_email'];
    $contact_phone = $_POST['contact_phone'];
    $address = $_POST['address'];
    $details = $_POST['details'];
    $region_id = $_POST['region_id'];
    $created_by = $user_id;

    $stmt = $conn->prepare("INSERT INTO clients (name, contact_email, contact_phone, address, details, region_id, created_by, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, 'New', NOW())");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("sssssii", $name, $contact_email, $contact_phone, $address, $details, $region_id, $created_by);
    if ($stmt->execute()) {
        $client_id = $stmt->insert_id;
        log_audit($conn, $user_id, 'Add Client', "Client ID: $client_id");

        // Notify technicians in region with no pending tasks
        $tech_sql = "SELECT u.id FROM users u 
                     LEFT JOIN clients c ON u.id = c.assigned_tech_id AND c.status IN ('Assignment Pending', 'Assigned', 'In Progress', 'Details Submitted') 
                     WHERE u.role = 'technician' AND u.region_id = ? AND c.id IS NULL GROUP BY u.id";
        $tech_stmt = $conn->prepare($tech_sql);
        if ($tech_stmt === false) die("Prepare failed: " . $conn->error);
        $tech_stmt->bind_param("i", $region_id);
        $tech_stmt->execute();
        $tech_result = $tech_stmt->get_result();
        while ($tech_row = $tech_result->fetch_assoc()) {
            $message = "New installation available: Client $name (ID: $client_id) in your region.";
            send_notification($conn, $tech_row['id'], $message, $client_id, 'available');
        }
        $tech_stmt->close();
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Client added successfully.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=sales_addclient");
    exit();
}

// Handle Update Client (Sales)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_client'])) {
    $client_id = $_POST['client_id'];
    $name = $_POST['name'];
    $contact_email = $_POST['contact_email'];
    $contact_phone = $_POST['contact_phone'];
    $address = $_POST['address'];
    $details = $_POST['details'];
    $region_id = $_POST['region_id'];

    $stmt = $conn->prepare("UPDATE clients SET name=?, contact_email=?, contact_phone=?, address=?, details=?, region_id=? WHERE id=? AND status='New'");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("sssssii", $name, $contact_email, $contact_phone, $address, $details, $region_id, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Update Client', "Client ID: $client_id");
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Client updated successfully.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=sales_clients");
    exit();
}

// Handle Export Clients to CSV (Sales)
if (isset($_GET['export_clients'])) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="clients.csv"');
    $output = fopen('php://output', 'w');
    fputcsv($output, ['ID', 'Name', 'Email', 'Phone', 'Address', 'Details', 'Region', 'Status', 'Created At']);
    $sql = "SELECT c.id, c.name, c.contact_email, c.contact_phone, c.address, c.details, r.name AS region, c.status, c.created_at 
            FROM clients c JOIN regions r ON c.region_id = r.id";
    $result = $conn->query($sql);
    if ($result === false) die("Query failed: " . $conn->error);
    while ($row = $result->fetch_assoc()) {
        fputcsv($output, $row);
    }
    log_audit($conn, $user_id, 'Export Clients', 'Exported all clients to CSV');
    exit();
}

// Handle Technician Complete Installation
if (isset($_GET['complete_technician'])) {
    $client_id = $_GET['complete_technician'];
    $client_sql = "SELECT c.name, r.name AS region_name FROM clients c JOIN regions r ON c.region_id = r.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $region_name = $client_row['region_name'] ?? 'Unknown Region';
    $client_stmt->close();

    // Verify the client is assigned to the superuser and in Details Submitted status
    $verify_sql = "SELECT status, assigned_tech_id FROM clients WHERE id = ? AND assigned_tech_id = ? AND status = 'Details Submitted'";
    $verify_stmt = $conn->prepare($verify_sql);
    if ($verify_stmt === false) die("Prepare failed: " . $conn->error);
    $verify_stmt->bind_param("ii", $client_id, $user_id);
    $verify_stmt->execute();
    $verify_result = $verify_stmt->get_result();
    if ($verify_result->num_rows === 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: Cannot complete. Client not assigned to you or details not submitted.'];
        $verify_stmt->close();
        header("Location: main.php?tab=technician_assigned");
        exit();
    }
    $verify_stmt->close();

    // Update client status to Completed
    $stmt = $conn->prepare("UPDATE clients SET status = 'Completed', updated_at = NOW() WHERE id = ?");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("i", $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Complete Installation', "Client: $client_name in $region_name");
        $message = "Installation for client $client_name in $region_name marked as completed.";
        send_notification($conn, $user_id, $message, $client_id, 'technician_completed');
        // Notify sales team
        $client_sql = "SELECT created_by FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) die("Prepare failed: " . $conn->error);
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        if ($client_row['created_by']) {
            $message = "Installation for client $client_name completed by superuser (ID: $user_id).";
            send_notification($conn, $client_row['created_by'], $message, $client_id, 'sales_clients');
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => "Installation for client $client_name marked as completed."];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=technician_completed&highlight=$client_id");
    exit();
}

// Handle Approve Assignment (Tech)
if (isset($_GET['approve'])) {
    $client_id = $_GET['approve'];
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET approved = 1, status = 'Assigned' WHERE id = ? AND status = 'Assignment Pending'");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("i", $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Approve Assignment', "Client: $client_name");
        $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) die("Prepare failed: " . $conn->error);
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Assignment approved for client $client_name by technician $tech_name.";
        if ($client_row['created_by']) send_notification($conn, $client_row['created_by'], $message, $client_id, 'sales_clients');
        if ($client_row['assigned_tech_id']) send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'technician_assigned');
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Assignment approved.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=tech_pending");
    exit();
}

// Handle Direct Technician Assignment (Tech)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['assign_technician'])) {
    $client_id = $_POST['client_id'];
    $tech_id = $_POST['technician_id'];
    $client_sql = "SELECT c.name FROM clients c WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $client_stmt->close();
    $tech_sql = "SELECT username FROM users WHERE id = ?";
    $tech_stmt = $conn->prepare($tech_sql);
    if ($tech_stmt === false) die("Prepare failed: " . $conn->error);
    $tech_stmt->bind_param("i", $tech_id);
    $tech_stmt->execute();
    $tech_result = $tech_stmt->get_result();
    $tech_row = $tech_result->fetch_assoc();
    $tech_name = $tech_row['username'] ?? 'Unknown Technician';
    $tech_stmt->close();
    $pending_sql = "SELECT COUNT(*) as count FROM clients WHERE assigned_tech_id = ? AND status IN ('Assignment Pending', 'Assigned', 'In Progress')";
    $pending_stmt = $conn->prepare($pending_sql);
    if ($pending_stmt === false) die("Prepare failed: " . $conn->error);
    $pending_stmt->bind_param("i", $tech_id);
    $pending_stmt->execute();
    $pending_result = $pending_stmt->get_result();
    $pending_row = $pending_result->fetch_assoc();
    if ($pending_row['count'] > 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => "Cannot assign: Technician $tech_name has pending tasks."];
        $pending_stmt->close();
        header("Location: main.php?tab=tech_allclients");
        exit();
    }
    $pending_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET assigned_tech_id = ?, status = 'Assigned', approved = 1 WHERE id = ? AND status = 'New' AND assigned_tech_id IS NULL");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("ii", $tech_id, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Assign Technician', "Client: $client_name to Technician: $tech_name");
        $client_sql = "SELECT created_by FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) die("Prepare failed: " . $conn->error);
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Technician $tech_name assigned to client $client_name.";
        if ($client_row['created_by']) send_notification($conn, $client_row['created_by'], $message, $client_id, 'sales_clients');
        send_notification($conn, $tech_id, $message, $client_id, 'technician_assigned');
        $_SESSION['toast'] = ['type' => 'success', 'message' => "Technician $tech_name assigned to client $client_name."];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=tech_allclients");
    exit();
}

// Handle Complete Installation (Tech)
if (isset($_GET['complete'])) {
    $client_id = $_GET['complete'];
    $check_sql = "SELECT COUNT(*) AS count FROM installations WHERE client_id = ?";
    $check_stmt = $conn->prepare($check_sql);
    $check_stmt->bind_param("i", $client_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    $check_row = $check_result->fetch_assoc();
    if ($check_row['count'] == 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: No installation record found for this client.'];
        $check_stmt->close();
        header("Location: main.php?tab=tech_submitted");
        exit();
    }
    $check_stmt->close();
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET status = 'Completed' WHERE id = ? AND status = 'Details Submitted'");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("i", $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Complete Installation', "Client: $client_name");
        $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) die("Prepare failed: " . $conn->error);
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Installation completed for client $client_name by technician $tech_name.";
        if ($client_row['created_by']) send_notification($conn, $client_row['created_by'], $message, $client_id, 'sales_clients');
        if ($client_row['assigned_tech_id']) send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'technician_completed');
        $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
        $tech_result = $conn->query($tech_sql);
        if ($tech_result === false) die("Query failed: " . $conn->error);
        while ($tech_row = $tech_result->fetch_assoc()) {
            send_notification($conn, $tech_row['id'], $message, $client_id, 'tech_completed');
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Installation completed.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=tech_submitted");
    exit();
}

// Handle Update Status (Tech)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_status'])) {
    $client_id = $_POST['client_id'];
    $new_status = $_POST['status'];
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    if ($new_status == 'Completed') {
        $check_sql = "SELECT COUNT(*) AS count FROM installations WHERE client_id = ?";
        $check_stmt = $conn->prepare($check_sql);
        $check_stmt->bind_param("i", $client_id);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        $check_row = $check_result->fetch_assoc();
        if ($check_row['count'] == 0) {
            $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: Cannot mark as Completed. No installation record found.'];
            $check_stmt->close();
            header("Location: main.php?tab=tech_allclients");
            exit();
        }
        $check_stmt->close();
    }
    $stmt = $conn->prepare("UPDATE clients SET status = ?, assigned_tech_id = CASE WHEN ? = 'Cancelled' THEN NULL ELSE assigned_tech_id END WHERE id = ?");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("ssi", $new_status, $new_status, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Update Status', "Client: $client_name to $new_status");
        if ($new_status == 'Completed') {
            $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
            $client_stmt = $conn->prepare($client_sql);
            if ($client_stmt === false) die("Prepare failed: " . $conn->error);
            $client_stmt->bind_param("i", $client_id);
            $client_stmt->execute();
            $client_result = $client_stmt->get_result();
            $client_row = $client_result->fetch_assoc();
            $client_stmt->close();
            $message = "Installation status updated to Completed for client $client_name by technician $tech_name.";
            if ($client_row['created_by']) send_notification($conn, $client_row['created_by'], $message, $client_id, 'sales_clients');
            if ($client_row['assigned_tech_id']) send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'technician_completed');
            $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
            $tech_result = $conn->query($tech_sql);
            if ($tech_result === false) die("Query failed: " . $conn->error);
            while ($tech_row = $tech_result->fetch_assoc()) {
                send_notification($conn, $tech_row['id'], $message, $client_id, 'tech_completed');
            }
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Status updated.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=tech_allclients");
    exit();
}

// Handle Cancel Client (Tech)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['cancel_client'])) {
    $client_id = $_POST['client_id'];
    $cancel_reason = trim($_POST['cancel_reason']);
    if (empty($cancel_reason)) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: Cancellation reason is required.'];
        header("Location: main.php?tab=tech_allclients");
        exit();
    }
    $cancel_reason = htmlspecialchars($cancel_reason, ENT_QUOTES, 'UTF-8');
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET status = 'Cancelled', assigned_tech_id = NULL, approved = 0 WHERE id = ?");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("i", $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Cancel Client', "Client: $client_name, Reason: $cancel_reason");
        $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) die("Prepare failed: " . $conn->error);
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Client $client_name has been cancelled. Reason: $cancel_reason";
        if ($client_row['created_by']) send_notification($conn, $client_row['created_by'], $message, $client_id, 'sales_clients');
        if ($client_row['assigned_tech_id']) send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'technician_allclients');
        $_SESSION['toast'] = ['type' => 'success', 'message' => "Client $client_name cancelled."];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=tech_allclients");
    exit();
}

// Handle Assign to Self (Technician)
if (isset($_GET['assign_self'])) {
    $client_id = $_GET['assign_self'];
    $client_sql = "SELECT c.name, r.name AS region_name FROM clients c JOIN regions r ON c.region_id = r.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $region_name = $client_row['region_name'] ?? 'Unknown Region';
    $client_stmt->close();

    // Check if technician has pending tasks
    $pending_sql = "SELECT COUNT(*) as count FROM clients WHERE assigned_tech_id = ? AND status IN ('Assignment Pending', 'Assigned', 'In Progress', 'Details Submitted')";
    $pending_stmt = $conn->prepare($pending_sql);
    if ($pending_stmt === false) die("Prepare failed: " . $conn->error);
    $pending_stmt->bind_param("i", $user_id);
    $pending_stmt->execute();
    $pending_result = $pending_stmt->get_result();
    $pending_row = $pending_result->fetch_assoc();
    if ($pending_row['count'] > 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => "Cannot assign: You have pending tasks."];
        $pending_stmt->close();
        header("Location: main.php?tab=technician_available");
        exit();
    }
    $pending_stmt->close();

    // Assign client to superuser (technician) and set as Assigned
    $stmt = $conn->prepare("UPDATE clients SET assigned_tech_id = ?, status = 'Assigned', approved = 1 WHERE id = ? AND status = 'New' AND assigned_tech_id IS NULL");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("ii", $user_id, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Assign Self', "Client: $client_name in $region_name");
        $message = "You have assigned client $client_name in $region_name to yourself.";
        send_notification($conn, $user_id, $message, $client_id, 'technician_assigned');
        // Notify sales team
        $client_sql = "SELECT created_by FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) die("Prepare failed: " . $conn->error);
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        if ($client_row['created_by']) {
            $message = "Client $client_name has been assigned to superuser (ID: $user_id).";
            send_notification($conn, $client_row['created_by'], $message, $client_id, 'sales_clients');
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => "Assigned client $client_name to yourself."];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=technician_assigned&highlight=$client_id");
    exit();
}

// Handle Submit Installation Details (Technician)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['submit_details'])) {
    $client_id = $_POST['client_id'];
    $gadget_details = $_POST['gadget_details'];
    $vehicle_details = $_POST['vehicle_details'];
    $installation_date = $_POST['installation_date'];
    $notes = $_POST['notes'];

    $client_sql = "SELECT name FROM clients WHERE id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $client_stmt->close();

    $check_sql = "SELECT COUNT(*) AS count FROM installations WHERE client_id = ?";
    $check_stmt = $conn->prepare($check_sql);
    $check_stmt->bind_param("i", $client_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    $check_row = $check_result->fetch_assoc();
    if ($check_row['count'] > 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: Installation details already submitted for this client.'];
        $check_stmt->close();
        header("Location: main.php?tab=technician_assigned");
        exit();
    }
    $check_stmt->close();

    $stmt = $conn->prepare("INSERT INTO installations (client_id, gadget_details, vehicle_details, installation_date, notes, updated_at) VALUES (?, ?, ?, ?, ?, NOW())");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("issss", $client_id, $gadget_details, $vehicle_details, $installation_date, $notes);
    if ($stmt->execute()) {
        $update_stmt = $conn->prepare("UPDATE clients SET status = 'Details Submitted' WHERE id = ? AND status IN ('Assigned', 'In Progress')");
        if ($update_stmt === false) die("Prepare failed: " . $conn->error);
        $update_stmt->bind_param("i", $client_id);
        if ($update_stmt->execute()) {
            log_audit($conn, $user_id, 'Submit Details', "Client: $client_name");
            $message = "Installation details submitted for client $client_name by superuser.";
            send_notification($conn, $user_id, $message, $client_id, 'technician_assigned');
            $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
            $tech_result = $conn->query($tech_sql);
            if ($tech_result === false) die("Query failed: " . $conn->error);
            while ($tech_row = $tech_result->fetch_assoc()) {
                send_notification($conn, $tech_row['id'], $message, $client_id, 'tech_submitted');
            }
            $_SESSION['toast'] = ['type' => 'success', 'message' => 'Details submitted.'];
        } else {
            $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
        }
        $update_stmt->close();
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=technician_assigned");
    exit();
}

// Handle Update Installation Details (Technician)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_details'])) {
    $client_id = $_POST['client_id'];
    $gadget_details = $_POST['gadget_details'];
    $vehicle_details = $_POST['vehicle_details'];
    $installation_date = $_POST['installation_date'];
    $notes = $_POST['notes'];

    $client_sql = "SELECT name FROM clients WHERE id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) die("Prepare failed: " . $conn->error);
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $client_stmt->close();

    $stmt = $conn->prepare("UPDATE installations SET gadget_details = ?, vehicle_details = ?, installation_date = ?, notes = ?, updated_at = NOW() WHERE client_id = ?");
    if ($stmt === false) die("Prepare failed: " . $conn->error);
    $stmt->bind_param("ssssi", $gadget_details, $vehicle_details, $installation_date, $notes, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Update Details', "Client: $client_name");
        $message = "Installation details updated for client $client_name by superuser.";
        send_notification($conn, $user_id, $message, $client_id, 'technician_assigned');
        $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
        $tech_result = $conn->query($tech_sql);
        if ($tech_result === false) die("Query failed: " . $conn->error);
        while ($tech_row = $tech_result->fetch_assoc()) {
            send_notification($conn, $tech_row['id'], $message, $client_id, 'tech_submitted');
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Details updated successfully.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: main.php?tab=technician_assigned");
    exit();
}

// Handle Export Installations to Excel (Tech)
if (isset($_GET['export_installations'])) {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="installations.xls"');
    $sql = "SELECT c.id AS client_id, c.name, r.name AS region_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes, c.contact_email, c.contact_phone, c.address, c.details AS client_details, u.username AS sales_username, tu.username AS tech_username
            FROM clients c
            JOIN installations i ON c.id = i.client_id
            JOIN regions r ON c.region_id = r.id
            LEFT JOIN users u ON c.created_by = u.id
            LEFT JOIN users tu ON c.assigned_tech_id = tu.id
            WHERE c.status = 'Completed'";
    $params = [];
    $types = "";
    if (isset($_GET['from_date']) && !empty($_GET['from_date'])) {
        $sql .= " AND i.installation_date >= ?";
        $params[] = $_GET['from_date'];
        $types .= "s";
    }
    if (isset($_GET['to_date']) && !empty($_GET['to_date'])) {
        $sql .= " AND i.installation_date <= ?";
        $params[] = $_GET['to_date'];
        $types .= "s";
    }
    $stmt = $conn->prepare($sql);
    if (!empty($params)) $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    $headers = ['Client ID', 'Name', 'Region', 'Gadget Details', 'Vehicle Details', 'Installation Date', 'Notes', 'Contact Email', 'Contact Phone', 'Address', 'Client Details', 'Sales Person', 'Technician'];
    echo implode("\t", $headers) . "\n";
    while ($row = $result->fetch_assoc()) {
        $line = [
            $row['client_id'],
            $row['name'],
            $row['region_name'],
            $row['gadget_details'] ?? 'N/A',
            $row['vehicle_details'] ?? 'N/A',
            $row['installation_date'] ?? 'N/A',
            $row['notes'] ?? 'N/A',
            $row['contact_email'] ?? 'N/A',
            $row['contact_phone'] ?? 'N/A',
            $row['address'] ?? 'N/A',
            $row['client_details'] ?? 'N/A',
            $row['sales_username'] ?? 'N/A',
            $row['tech_username'] ?? 'N/A'
        ];
        echo implode("\t", array_map(function($v) { return str_replace(["\r", "\n", "\t"], ' ', $v); }, $line)) . "\n";
    }
    $stmt->close();
    $filter_details = "";
    if (isset($_GET['from_date']) || isset($_GET['to_date'])) {
        $filter_details = " with filters: " . (isset($_GET['from_date']) ? "from " . $_GET['from_date'] : "") . (isset($_GET['to_date']) ? " to " . $_GET['to_date'] : "");
    }
    log_audit($conn, $user_id, 'Export Installations', 'Exported completed installations to Excel' . $filter_details);
    exit();
}

// Handle Export Audit Logs to Excel (Tech)
if (isset($_GET['export_audit'])) {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="audit_logs.xls"');
    $sql = "SELECT a.id, u.username, a.action, a.details, a.timestamp FROM audit_logs a JOIN users u ON a.user_id = u.id";
    $params = [];
    $types = "";
    if (isset($_GET['from_date']) && !empty($_GET['from_date'])) {
        $sql .= " WHERE a.timestamp >= ?";
        $params[] = $_GET['from_date'] . " 00:00:00";
        $types .= "s";
    }
    if (isset($_GET['to_date']) && !empty($_GET['to_date'])) {
        $sql .= (empty($params) ? " WHERE" : " AND") . " a.timestamp <= ?";
        $params[] = $_GET['to_date'] . " 23:59:59";
        $types .= "s";
    }
    $sql .= " ORDER BY a.timestamp DESC";
    $stmt = $conn->prepare($sql);
    if (!empty($params)) $stmt->bind_param($types, ...$params);
    $stmt->execute();
    $result = $stmt->get_result();
    $headers = ['ID', 'User', 'Action', 'Details', 'Timestamp'];
    echo implode("\t", $headers) . "\n";
    while ($row = $result->fetch_assoc()) {
        $line = [
            $row['id'],
            $row['username'],
            $row['action'],
            $row['details'],
            $row['timestamp']
        ];
        echo implode("\t", array_map(function($v) { return str_replace(["\r", "\n", "\t"], ' ', $v); }, $line)) . "\n";
    }
    $stmt->close();
    $filter_details = "";
    if (isset($_GET['from_date']) || isset($_GET['to_date'])) {
        $filter_details = " with filters: " . (isset($_GET['from_date']) ? "from " . $_GET['from_date'] : "") . (isset($_GET['to_date']) ? " to " . $_GET['to_date'] : "");
    }
    log_audit($conn, $user_id, 'Export Audit Logs', 'Exported audit logs to Excel' . $filter_details);
    exit();
}

// Fetch regions
$regions_sql = "SELECT * FROM regions";
$regions_result = $conn->query($regions_sql);
if ($regions_result === false) die("Query failed: " . $conn->error);

// Fetch technicians
$tech_sql = "SELECT id, username FROM users WHERE role = 'technician' ORDER BY username";
$tech_result = $conn->query($tech_sql);
if ($tech_result === false) die("Query failed: " . $conn->error);
$technicians = [];
while ($tech_row = $tech_result->fetch_assoc()) {
    $technicians[] = $tech_row;
}

// Fetch all clients (Sales)
$clients_sql = "SELECT c.*, r.name AS region_name FROM clients c JOIN regions r ON c.region_id = r.id ORDER BY c.created_at DESC";
$clients_result = $conn->query($clients_sql);
if ($clients_result === false) die("Query failed: " . $conn->error);

// Fetch pending assignments (Tech)
$pending_sql = "SELECT c.*, u.username AS tech_name, r.name AS region_name FROM clients c
                JOIN users u ON c.assigned_tech_id = u.id
                JOIN regions r ON c.region_id = r.id
                WHERE c.status = 'Assignment Pending'";
$pending_result = $conn->query($pending_sql);
if ($pending_result === false) die("Query failed: " . $conn->error);

// Fetch details submitted (Tech)
$submitted_sql = "SELECT c.id, c.name, r.name AS region_name, u.username AS tech_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes, c.assigned_tech_id 
                  FROM clients c 
                  JOIN regions r ON c.region_id = r.id 
                  LEFT JOIN users u ON c.assigned_tech_id = u.id 
                  LEFT JOIN installations i ON c.id = i.client_id 
                  WHERE c.status = 'Details Submitted'";
$submitted_result = $conn->query($submitted_sql);
if ($submitted_result === false) die("Query failed: " . $conn->error);

// Fetch active clients (Tech)
$allclients_sql = "SELECT c.*, r.name AS region_name, u.username AS tech_name, s.username AS sales_name FROM clients c
                   LEFT JOIN regions r ON c.region_id = r.id
                   LEFT JOIN users u ON c.assigned_tech_id = u.id
                   LEFT JOIN users s ON c.created_by = s.id
                   WHERE c.status NOT IN ('Completed', 'Cancelled')
                   ORDER BY c.updated_at DESC LIMIT 100";
$allclients_result = $conn->query($allclients_sql);
if ($allclients_result === false) die("Query failed: " . $conn->error);

// Fetch completed installations (Tech/Technician)
$completed_sql = "SELECT c.id AS client_id, c.name, c.contact_email, c.contact_phone, c.address, c.details AS client_details, r.name AS region_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes, u.username AS sales_username, tu.username AS tech_username
                 FROM clients c
                 JOIN installations i ON c.id = i.client_id
                 JOIN regions r ON c.region_id = r.id
                 LEFT JOIN users u ON c.created_by = u.id
                 LEFT JOIN users tu ON c.assigned_tech_id = tu.id
                 WHERE c.status = 'Completed'
                 ORDER BY i.updated_at DESC LIMIT 100";
$completed_result = $conn->query($completed_sql);
if ($completed_result === false) die("Query failed: " . $conn->error);

// Fetch audit logs (Tech)
$audit_sql = "SELECT a.*, u.username FROM audit_logs a JOIN users u ON a.user_id = u.id ORDER BY a.timestamp DESC LIMIT 100";
$audit_result = $conn->query($audit_sql);
if ($audit_result === false) die("Query failed: " . $conn->error);

// Fetch notifications
$notif_sql = "SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC LIMIT 50";
$notif_stmt = $conn->prepare($notif_sql);
if ($notif_stmt === false) die("Prepare failed: " . $conn->error);
$notif_stmt->bind_param("i", $user_id);
$notif_stmt->execute();
$notif_result = $notif_stmt->get_result();
$notif_stmt->close();

// Fetch system stats with date filters
$from_date = isset($_GET['from_date']) && !empty($_GET['from_date']) ? $_GET['from_date'] : null;
$to_date = isset($_GET['to_date']) && !empty($_GET['to_date']) ? $_GET['to_date'] : null;

$total_users_sql = "SELECT COUNT(*) as count FROM users WHERE is_active = 1";
$total_users_result = $conn->query($total_users_sql);
$total_users = $total_users_result->fetch_assoc()['count'];

$total_clients_sql = "SELECT COUNT(*) as count FROM clients WHERE status NOT IN ('Cancelled')";
$params = [];
$types = "";
if ($from_date) {
    $total_clients_sql .= " AND created_at >= ?";
    $params[] = $from_date . " 00:00:00";
    $types .= "s";
}
if ($to_date) {
    $total_clients_sql .= " AND created_at <= ?";
    $params[] = $to_date . " 23:59:59";
    $types .= "s";
}
$total_clients_stmt = $conn->prepare($total_clients_sql);
if (!empty($params)) $total_clients_stmt->bind_param($types, ...$params);
$total_clients_stmt->execute();
$total_clients_result = $total_clients_stmt->get_result();
$total_clients = $total_clients_result->fetch_assoc()['count'];
$total_clients_stmt->close();

$completed_installations_sql = "SELECT COUNT(*) as count FROM clients WHERE status = 'Completed'";
$params = [];
$types = "";
if ($from_date) {
    $completed_installations_sql .= " AND updated_at >= ?";
    $params[] = $from_date . " 00:00:00";
    $types .= "s";
}
if ($to_date) {
    $completed_installations_sql .= " AND updated_at <= ?";
    $params[] = $to_date . " 23:59:59";
    $types .= "s";
}
$completed_installations_stmt = $conn->prepare($completed_installations_sql);
if (!empty($params)) $completed_installations_stmt->bind_param($types, ...$params);
$completed_installations_stmt->execute();
$completed_installations_result = $completed_installations_stmt->get_result();
$completed_installations = $completed_installations_result->fetch_assoc()['count'];
$completed_installations_stmt->close();

$pending_installations_sql = "SELECT COUNT(*) as count FROM clients WHERE status IN ('Assignment Pending', 'Assigned', 'In Progress', 'Details Submitted')";
$params = [];
$types = "";
if ($from_date) {
    $pending_installations_sql .= " AND updated_at >= ?";
    $params[] = $from_date . " 00:00:00";
    $types .= "s";
}
if ($to_date) {
    $pending_installations_sql .= " AND updated_at <= ?";
    $params[] = $to_date . " 23:59:59";
    $types .= "s";
}
$pending_installations_stmt = $conn->prepare($pending_installations_sql);
if (!empty($params)) $pending_installations_stmt->bind_param($types, ...$params);
$pending_installations_stmt->execute();
$pending_installations_result = $pending_installations_stmt->get_result();
$pending_installations = $pending_installations_result->fetch_assoc()['count'];
$pending_installations_stmt->close();

// Fetch clients by region
$clients_by_region_sql = "SELECT r.name AS region_name, COUNT(c.id) as count 
                          FROM regions r 
                          LEFT JOIN clients c ON r.id = c.region_id AND c.status NOT IN ('Cancelled')";
$params = [];
$types = "";
if ($from_date) {
    $clients_by_region_sql .= " AND c.created_at >= ?";
    $params[] = $from_date . " 00:00:00";
    $types .= "s";
}
if ($to_date) {
    $clients_by_region_sql .= " AND c.created_at <= ?";
    $params[] = $to_date . " 23:59:59";
    $types .= "s";
}
$clients_by_region_sql .= " GROUP BY r.id, r.name";
$clients_by_region_stmt = $conn->prepare($clients_by_region_sql);
if (!empty($params)) $clients_by_region_stmt->bind_param($types, ...$params);
$clients_by_region_stmt->execute();
$clients_by_region_result = $clients_by_region_stmt->get_result();
$clients_by_region = [];
$region_labels = [];
$region_data = [];
while ($row = $clients_by_region_result->fetch_assoc()) {
    $clients_by_region[] = $row;
    $region_labels[] = $row['region_name'];
    $region_data[] = $row['count'];
}
$clients_by_region_stmt->close();

// Fetch status distribution
$status_distribution_sql = "SELECT status, COUNT(*) as count FROM clients WHERE 1=1";
$params = [];
$types = "";
if ($from_date) {
    $status_distribution_sql .= " AND created_at >= ?";
    $params[] = $from_date . " 00:00:00";
    $types .= "s";
}
if ($to_date) {
    $status_distribution_sql .= " AND created_at <= ?";
    $params[] = $to_date . " 23:59:59";
    $types .= "s";
}
$status_distribution_sql .= " GROUP BY status";
$status_distribution_stmt = $conn->prepare($status_distribution_sql);
if (!empty($params)) $status_distribution_stmt->bind_param($types, ...$params);
$status_distribution_stmt->execute();
$status_distribution_result = $status_distribution_stmt->get_result();
$status_distribution = [];
$status_labels = ['New', 'Assignment Pending', 'Assigned', 'In Progress', 'Details Submitted', 'Completed', 'Cancelled'];
$status_data = array_fill(0, count($status_labels), 0);
$status_colors = ['#6c757d', '#ffc107', '#17a2b8', '#007bff', '#fd7e14', '#28a745', '#dc3545'];
while ($row = $status_distribution_result->fetch_assoc()) {
    $status_distribution[] = $row;
    $index = array_search($row['status'], $status_labels);
    if ($index !== false) {
        $status_data[$index] = $row['count'];
    }
}
$status_distribution_stmt->close();

// Fetch average installation time
$avg_installation_time_sql = "SELECT AVG(TIMESTAMPDIFF(DAY, c.created_at, c.updated_at)) as avg_days 
                              FROM clients c 
                              JOIN installations i ON c.id = i.client_id 
                              WHERE c.status = 'Completed'";
$params = [];
$types = "";
if ($from_date) {
    $avg_installation_time_sql .= " AND c.updated_at >= ?";
    $params[] = $from_date . " 00:00:00";
    $types .= "s";
}
if ($to_date) {
    $avg_installation_time_sql .= " AND c.updated_at <= ?";
    $params[] = $to_date . " 23:59:59";
    $types .= "s";
}
$avg_installation_time_stmt = $conn->prepare($avg_installation_time_sql);
if (!empty($params)) $avg_installation_time_stmt->bind_param($types, ...$params);
$avg_installation_time_stmt->execute();
$avg_installation_time_result = $avg_installation_time_stmt->get_result();
$avg_installation_time = $avg_installation_time_result->fetch_assoc()['avg_days'] ?? 0;
$avg_installation_time = round($avg_installation_time, 2);
$avg_installation_time_stmt->close();

// Fetch technician workload
$technician_workload_sql = "SELECT u.username, COUNT(c.id) as count 
                           FROM users u 
                           LEFT JOIN clients c ON u.id = c.assigned_tech_id AND c.status IN ('Assignment Pending', 'Assigned', 'In Progress', 'Details Submitted')";
$params = [];
$types = "";
if ($from_date) {
    $technician_workload_sql .= " AND c.updated_at >= ?";
    $params[] = $from_date . " 00:00:00";
    $types .= "s";
}
if ($to_date) {
    $technician_workload_sql .= " AND c.updated_at <= ?";
    $params[] = $to_date . " 23:59:59";
    $types .= "s";
}
$technician_workload_sql .= " WHERE u.role = 'technician' GROUP BY u.id, u.username";
$technician_workload_stmt = $conn->prepare($technician_workload_sql);
if (!empty($params)) $technician_workload_stmt->bind_param($types, ...$params);
$technician_workload_stmt->execute();
$technician_workload_result = $technician_workload_stmt->get_result();
$technician_workload = [];
$technician_labels = [];
$technician_data = [];
while ($row = $technician_workload_result->fetch_assoc()) {
    $technician_workload[] = $row;
    $technician_labels[] = $row['username'];
    $technician_data[] = $row['count'];
}
$technician_workload_stmt->close();

// Fetch recent activity
$recent_activity_sql = "SELECT a.*, u.username FROM audit_logs a JOIN users u ON a.user_id = u.id ORDER BY a.timestamp DESC LIMIT 10";
$recent_activity_result = $conn->query($recent_activity_sql);
if ($recent_activity_result === false) die("Query failed: " . $conn->error);

// Fetch all users
$users_sql = "SELECT u.id, u.username, u.role, u.is_active, r.name AS region_name 
              FROM users u LEFT JOIN regions r ON u.region_id = r.id";
$users_result = $conn->query($users_sql);
if ($users_result === false) die("Query failed: " . $conn->error);

// Fetch available installations (Technician)
$available_sql = "SELECT c.id, c.name, c.details, r.name AS region_name FROM clients c 
                  JOIN regions r ON c.region_id = r.id 
                  WHERE c.status = 'New' AND c.assigned_tech_id IS NULL";
$available_result = $conn->query($available_sql);
if ($available_result === false) die("Query failed: " . $conn->error);

// Fetch assigned installations (Technician)
$technician_assigned_sql = "SELECT c.*, r.name AS region_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes 
                            FROM clients c 
                            LEFT JOIN regions r ON c.region_id = r.id 
                            LEFT JOIN installations i ON c.id = i.client_id 
                            WHERE c.assigned_tech_id = ? AND c.status IN ('Assigned', 'In Progress', 'Details Submitted')";
$technician_assigned_stmt = $conn->prepare($technician_assigned_sql);
if ($technician_assigned_stmt === false) die("Prepare failed: " . $conn->error);
$technician_assigned_stmt->bind_param("i", $user_id);
$technician_assigned_stmt->execute();
$technician_assigned_result = $technician_assigned_stmt->get_result();
$technician_assigned_stmt->close();

// Delete notifications older than 7 days
$delete_notif_sql = "DELETE FROM notifications WHERE created_at < NOW() - INTERVAL 7 DAY AND user_id = ?";
$delete_notif_stmt = $conn->prepare($delete_notif_sql);
if ($delete_notif_stmt === false) die("Prepare failed: " . $conn->error);
$delete_notif_stmt->bind_param("i", $user_id);
$delete_notif_stmt->execute();
$delete_notif_stmt->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Superuser Dashboard - Vehicle Tracking System</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.0/dist/chart.umd.min.js"></script>
    <style>
        body { background-color: #f4f6f9; }
        .navbar-brand { font-weight: bold; }
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .nav-tabs {
            position: sticky;
            top: 56px;
            z-index: 1000;
            background-color: #f4f6f9;
            border-bottom: 1px solid #dee2e6;
        }
        .nav-tabs .nav-link {
            color: #333;
            font-weight: 500;
            border-radius: 0;
            padding: 10px 15px;
        }
        .nav-tabs .nav-link.active {
            background-color: #007bff;
            color: white;
            border-color: #dee2e6 #dee2e6 #f4f6f9;
        }
        .table { font-size: 0.9rem; }
        .btn-sm { font-size: 0.8rem; }
        .badge-pending { background-color: #ffc107; }
        .badge-inprogress { background-color: #17a2b8; }
        .badge-completed { background-color: #28a745; }
        .badge-secondary { background-color: #6c757d; }
        .toast-container { position: fixed; top: 20px; right: 20px; z-index: 1055; }
        .filter-form { background-color: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; }
        .notification-link { cursor: pointer; color: #007bff; text-decoration: underline; }
        .notification-link:hover { color: #0056b3; }
        .stats-card { background-color: #fff; padding: 20px; border-radius: 8px; margin-bottom: 20px; }
        .stats-card h6 { font-size: 1rem; color: #6c757d; }
        .stats-card p { font-size: 1.5rem; font-weight: bold; margin: 0; }
        .chart-container { max-width: 100%; margin-bottom: 20px; }
        /* Table styling */
        #clientsTable, #pendingTable, #submittedTable, #allClientsTable, #completedTable, #auditTable, #availableTable, #technicianAssignedTable, #usersTable, #recentActivityTable {
            table-layout: fixed;
            width: 100%;
        }
        #clientsTable th, #clientsTable td,
        #pendingTable th, #pendingTable td,
        #submittedTable th, #submittedTable td,
        #allClientsTable th, #allClientsTable td,
        #completedTable th, #completedTable td,
        #auditTable th, #auditTable td,
        #availableTable th, #availableTable td,
        #technicianAssignedTable th, #technicianAssignedTable td,
        #usersTable th, #usersTable td,
        #recentActivityTable th, #recentActivityTable td {
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            padding: 8px;
        }
        #clientsTable th:nth-child(1), #clientsTable td:nth-child(1) { width: 5%; }
        #clientsTable th:nth-child(2), #clientsTable td:nth-child(2) { width: 15%; }
        #clientsTable th:nth-child(3), #clientsTable td:nth-child(3) { width: 15%; }
        #clientsTable th:nth-child(4), #clientsTable td:nth-child(4) { width: 10%; }
        #clientsTable th:nth-child(5), #clientsTable td:nth-child(5) { width: 20%; white-space: normal; }
        #clientsTable th:nth-child(6), #clientsTable td:nth-child(6) { width: 20%; white-space: normal; }
        #clientsTable th:nth-child(7), #clientsTable td:nth-child(7) { width: 10%; }
        #clientsTable th:nth-child(8), #clientsTable td:nth-child(8) { width: 10%; }
        #clientsTable th:nth-child(9), #clientsTable td:nth-child(9) { width: 10%; }
        #pendingTable th:nth-child(1), #pendingTable td:nth-child(1) { width: 5%; }
        #pendingTable th:nth-child(2), #pendingTable td:nth-child(2) { width: 25%; }
        #pendingTable th:nth-child(3), #pendingTable td:nth-child(3) { width: 20%; }
        #pendingTable th:nth-child(4), #pendingTable td:nth-child(4) { width: 25%; }
        #pendingTable th:nth-child(5), #pendingTable td:nth-child(5) { width: 25%; }
        #submittedTable th:nth-child(1), #submittedTable td:nth-child(1) { width: 5%; }
        #submittedTable th:nth-child(2), #submittedTable td:nth-child(2) { width: 15%; }
        #submittedTable th:nth-child(3), #submittedTable td:nth-child(3) { width: 10%; }
        #submittedTable th:nth-child(4), #submittedTable td:nth-child(4) { width: 10%; }
        #submittedTable th:nth-child(5), #submittedTable td:nth-child(5) { width: 15%; white-space: normal; }
        #submittedTable th:nth-child(6), #submittedTable td:nth-child(6) { width: 15%; white-space: normal; }
        #submittedTable th:nth-child(7), #submittedTable td:nth-child(7) { width: 10%; }
        #submittedTable th:nth-child(8), #submittedTable td:nth-child(8) { width: 15%; white-space: normal; }
        #submittedTable th:nth-child(9), #submittedTable td:nth-child(9) { width: 15%; }
        #allClientsTable th:nth-child(1), #allClientsTable td:nth-child(1) { width: 5%; }
        #allClientsTable th:nth-child(2), #allClientsTable td:nth-child(2) { width: 10%; }
        #allClientsTable th:nth-child(3), #allClientsTable td:nth-child(3) { width: 10%; }
        #allClientsTable th:nth-child(4), #allClientsTable td:nth-child(4) { width: 8%; }
        #allClientsTable th:nth-child(5), #allClientsTable td:nth-child(5) { width: 12%; white-space: normal; }
        #allClientsTable th:nth-child(6), #allClientsTable td:nth-child(6) { width: 12%; white-space: normal; }
        #allClientsTable th:nth-child(7), #allClientsTable td:nth-child(7) { width: 8%; }
        #allClientsTable th:nth-child(8), #allClientsTable td:nth-child(8) { width: 8%; }
        #allClientsTable th:nth-child(9), #allClientsTable td:nth-child(9) { width: 8%; }
        #allClientsTable th:nth-child(10), #allClientsTable td:nth-child(10) { width: 8%; }
        #allClientsTable th:nth-child(11), #allClientsTable td:nth-child(11) { width: 5%; }
        #allClientsTable th:nth-child(12), #allClientsTable td:nth-child(12) { width: 10%; }
        #allClientsTable th:nth-child(13), #allClientsTable td:nth-child(13) { width: 10%; }
        #allClientsTable th:nth-child(14), #allClientsTable td:nth-child(14) { width: 6%; }
        #completedTable th:nth-child(1), #completedTable td:nth-child(1) { width: 5%; }
        #completedTable th:nth-child(2), #completedTable td:nth-child(2) { width: 10%; }
        #completedTable th:nth-child(3), #completedTable td:nth-child(3) { width: 8%; }
        #completedTable th:nth-child(4), #completedTable td:nth-child(4) { width: 8%; }
        #completedTable th:nth-child(5), #completedTable td:nth-child(5) { width: 8%; }
        #completedTable th:nth-child(6), #completedTable td:nth-child(6) { width: 8%; }
        #completedTable th:nth-child(7), #completedTable td:nth-child(7) { width: 15%; white-space: normal; }
        #completedTable th:nth-child(8), #completedTable td:nth-child(8) { width: 15%; white-space: normal; }
        #completedTable th:nth-child(9), #completedTable td:nth-child(9) { width: 15%; white-space: normal; }
        #completedTable th:nth-child(10), #completedTable td:nth-child(10) { width: 10%; }
        #completedTable th:nth-child(11), #completedTable td:nth-child(11) { width: 8%; }
        #completedTable th:nth-child(12), #completedTable td:nth-child(12) { width: 8%; }
        #completedTable th:nth-child(13), #completedTable td:nth-child(13) { width: 10%; }
        #auditTable th:nth-child(1), #auditTable td:nth-child(1) { width: 5%; }
        #auditTable th:nth-child(2), #auditTable td:nth-child(2) { width: 15%; }
        #auditTable th:nth-child(3), #auditTable td:nth-child(3) { width: 20%; }
        #auditTable th:nth-child(4), #auditTable td:nth-child(4) { width: 40%; white-space: normal; }
        #auditTable th:nth-child(5), #auditTable td:nth-child(5) { width: 20%; }
        #availableTable th:nth-child(1), #availableTable td:nth-child(1) { width: 5%; }
        #availableTable th:nth-child(2), #availableTable td:nth-child(2) { width: 25%; }
        #availableTable th:nth-child(3), #availableTable td:nth-child(3) { width: 35%; white-space: normal; }
        #availableTable th:nth-child(4), #availableTable td:nth-child(4) { width: 20%; }
        #availableTable th:nth-child(5), #availableTable td:nth-child(5) { width: 15%; }
        #technicianAssignedTable th:nth-child(1), #technicianAssignedTable td:nth-child(1) { width: 5%; }
        #technicianAssignedTable th:nth-child(2), #technicianAssignedTable td:nth-child(2) { width: 15%; }
        #technicianAssignedTable th:nth-child(3), #technicianAssignedTable td:nth-child(3) { width: 25%; white-space: normal; }
        #technicianAssignedTable th:nth-child(4), #technicianAssignedTable td:nth-child(4) { width: 15%; }
        #technicianAssignedTable th:nth-child(5), #technicianAssignedTable td:nth-child(5) { width: 10%; }
        #technicianAssignedTable th:nth-child(6), #technicianAssignedTable td:nth-child(6) { width: 30%; }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    <div class="container mt-5">
        <?php if (isset($_SESSION['toast'])) { ?>
            <div class="toast-container">
                <div class="toast align-items-center text-white bg-<?php echo htmlspecialchars($_SESSION['toast']['type']); ?> border-0" role="alert" aria-live="assertive" aria-atomic="true">
                    <div class="d-flex">
                        <div class="toast-body">
                            <?php echo htmlspecialchars($_SESSION['toast']['message']); ?>
                        </div>
                        <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                    </div>
                </div>
            </div>
            <?php unset($_SESSION['toast']); ?>
        <?php } ?>
    <h1 class="mb-4 text-center">Superuser Dashboard</h1>
        <ul class="nav nav-tabs" id="superuserTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (!isset($_GET['tab']) || $_GET['tab'] == 'notifications') ? 'active' : ''; ?>" id="notifications-tab" data-bs-toggle="tab" data-bs-target="#notifications" type="button" role="tab" aria-controls="notifications" aria-selected="<?php echo (!isset($_GET['tab']) || $_GET['tab'] == 'notifications') ? 'true' : 'false'; ?>">Notifications</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'system') ? 'active' : ''; ?>" id="system-tab" data-bs-toggle="tab" data-bs-target="#system" type="button" role="tab" aria-controls="system" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'system') ? 'true' : 'false'; ?>">System</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'sales_addclient') ? 'active' : ''; ?>" id="sales-addclient-tab" data-bs-toggle="tab" data-bs-target="#sales_addclient" type="button" role="tab" aria-controls="sales_addclient" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'sales_addclient') ? 'true' : 'false'; ?>">Add Client</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'sales_clients') ? 'active' : ''; ?>" id="sales-clients-tab" data-bs-toggle="tab" data-bs-target="#sales_clients" type="button" role="tab" aria-controls="sales_clients" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'sales_clients') ? 'true' : 'false'; ?>">Clients</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_pending') ? 'active' : ''; ?>" id="tech-pending-tab" data-bs-toggle="tab" data-bs-target="#tech_pending" type="button" role="tab" aria-controls="tech_pending" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_pending') ? 'true' : 'false'; ?>">Pending Assignments</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_submitted') ? 'active' : ''; ?>" id="tech-submitted-tab" data-bs-toggle="tab" data-bs-target="#tech_submitted" type="button" role="tab" aria-controls="tech_submitted" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_submitted') ? 'true' : 'false'; ?>">Details Submitted</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_allclients') ? 'active' : ''; ?>" id="tech-allclients-tab" data-bs-toggle="tab" data-bs-target="#tech_allclients" type="button" role="tab" aria-controls="tech_allclients" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_allclients') ? 'true' : 'false'; ?>">All Clients</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_completed') ? 'active' : ''; ?>" id="tech-completed-tab" data-bs-toggle="tab" data-bs-target="#tech_completed" type="button" role="tab" aria-controls="tech_completed" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_completed') ? 'true' : 'false'; ?>">Completed Installations</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_audit') ? 'active' : ''; ?>" id="tech-audit-tab" data-bs-toggle="tab" data-bs-target="#tech_audit" type="button" role="tab" aria-controls="tech_audit" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_audit') ? 'true' : 'false'; ?>">Audit Logs</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_available') ? 'active' : ''; ?>" id="technician-available-tab" data-bs-toggle="tab" data-bs-target="#technician_available" type="button" role="tab" aria-controls="technician_available" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_available') ? 'true' : 'false'; ?>">Available Installations</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_assigned') ? 'active' : ''; ?>" id="technician-assigned-tab" data-bs-toggle="tab" data-bs-target="#technician_assigned" type="button" role="tab" aria-controls="technician_assigned" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_assigned') ? 'true' : 'false'; ?>">Assigned Installations</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_completed') ? 'active' : ''; ?>" id="technician-completed-tab" data-bs-toggle="tab" data-bs-target="#technician_completed" type="button" role="tab" aria-controls="technician_completed" aria-selected="<?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_completed') ? 'true' : 'false'; ?>">Completed Installations</button>
            </li>
        </ul>

        <div class="tab-content mt-3" id="superuserTabContent">
            <!-- Notifications Tab -->
            <div class="tab-pane fade <?php echo (!isset($_GET['tab']) || $_GET['tab'] == 'notifications') ? 'show active' : ''; ?>" id="notifications" role="tabpanel" aria-labelledby="notifications-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Notifications</h5>
                    </div>
                    <div class="card-body">
                        <table id="notificationsTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>Message</th>
                                    <th>Received</th>
                                    <th>Action</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($notif = $notif_result->fetch_assoc()) { ?>
                                    <tr class="<?php echo $notif['is_read'] ? '' : 'table-primary'; ?>">
                                        <td>
                                            <?php if ($notif['related_id'] && $notif['related_tab']) { ?>
                                                <a class="notification-link" href="main.php?tab=<?php echo urlencode($notif['related_tab']); ?>&highlight=<?php echo $notif['related_id']; ?>&mark_read=<?php echo $notif['id']; ?>">
                                                    <?php echo htmlspecialchars($notif['message']); ?>
                                                </a>
                                            <?php } else { ?>
                                                <?php echo htmlspecialchars($notif['message']); ?>
                                            <?php } ?>
                                        </td>
                                        <td><?php echo htmlspecialchars($notif['created_at']); ?></td>
                                        <td>
                                            <?php if (!$notif['is_read']) { ?>
                                                <a href="main.php?tab=notifications&mark_read=<?php echo $notif['id']; ?>" class="btn btn-sm btn-outline-primary">Mark as Read</a>
                                            <?php } ?>
                                        </td>
                                    </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- System Tab -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'system') ? 'show active' : ''; ?>" id="system" role="tabpanel" aria-labelledby="system-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">System Statistics</h5>
                    </div>
                    <div class="card-body">
                        <form method="GET" class="filter-form mb-4">
                            <input type="hidden" name="tab" value="system">
                            <div class="row g-3 align-items-end">
                                <div class="col-md-3">
                                    <label for="from_date" class="form-label">From Date</label>
                                    <input type="date" class="form-control" id="from_date" name="from_date" value="<?php echo htmlspecialchars($from_date ?? ''); ?>">
                                </div>
                                <div class="col-md-3">
                                    <label for="to_date" class="form-label">To Date</label>
                                    <input type="date" class="form-control" id="to_date" name="to_date" value="<?php echo htmlspecialchars($to_date ?? ''); ?>">
                                </div>
                                <div class="col-md-3">
                                    <button type="submit" class="btn btn-primary">Apply Filter</button>
                                    <a href="main.php?tab=system" class="btn btn-secondary">Clear Filter</a>
                                </div>
                            </div>
                        </form>
                        <div class="row">
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <h6>Total Active Users</h6>
                                    <p><?php echo $total_users; ?></p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <h6>Total Clients</h6>
                                    <p><?php echo $total_clients; ?></p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <h6>Completed Installations</h6>
                                    <p><?php echo $completed_installations; ?></p>
                                </div>
                            </div>
                            <div class="col-md-3">
                                <div class="stats-card">
                                    <h6>Pending Installations</h6>
                                    <p><?php echo $pending_installations; ?></p>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="chart-container">
                                    <h6>Clients by Region</h6>
                                    <canvas id="regionChart"></canvas>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="chart-container">
                                    <h6>Client Status Distribution</h6>
                                    <canvas id="statusChart"></canvas>
                                </div>
                            </div>
                        </div>
                        <div class="row">
                            <div class="col-md-6">
                                <div class="stats-card">
                                    <h6>Average Installation Time</h6>
                                    <p><?php echo $avg_installation_time; ?> days</p>
                                </div>
                            </div>
                            <div class="col-md-6">
                                <div class="chart-container">
                                    <h6>Technician Workload</h6>
                                    <canvas id="technicianChart"></canvas>
                                </div>
                            </div>
                        </div>
                        <div class="card mt-4">
                            <div class="card-header">
                                <h6>Recent Activity</h6>
                            </div>
                            <div class="card-body">
                                <table id="recentActivityTable" class="table table-striped">
                                    <thead>
                                        <tr>
                                            <th>User</th>
                                            <th>Action</th>
                                            <th>Details</th>
                                            <th>Timestamp</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while ($activity = $recent_activity_result->fetch_assoc()) { ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($activity['username']); ?></td>
                                                <td><?php echo htmlspecialchars($activity['action']); ?></td>
                                                <td><?php echo htmlspecialchars($activity['details']); ?></td>
                                                <td><?php echo htmlspecialchars($activity['timestamp']); ?></td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Add Client Tab (Sales) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'sales_addclient') ? 'show active' : ''; ?>" id="sales_addclient" role="tabpanel" aria-labelledby="sales-addclient-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Add New Client</h5>
                    </div>
                    <div class="card-body">
                        <form method="POST">
                            <div class="mb-3">
                                <label for="name" class="form-label">Name</label>
                                <input type="text" class="form-control" id="name" name="name" required>
                            </div>
                            <div class="mb-3">
                                <label for="contact_email" class="form-label">Contact Email</label>
                                <input type="email" class="form-control" id="contact_email" name="contact_email" required>
                            </div>
                            <div class="mb-3">
                                <label for="contact_phone" class="form-label">Contact Phone</label>
                                <input type="text" class="form-control" id="contact_phone" name="contact_phone">
                            </div>
                            <div class="mb-3">
                                <label for="address" class="form-label">Address</label>
                                <textarea class="form-control" id="address" name="address" rows="4"></textarea>
                            </div>
                            <div class="mb-3">
                                <label for="details" class="form-label">Details</label>
                                <textarea class="form-control" id="details" name="details" rows="4"></textarea>
                            </div>
                            <div class="mb-3">
                                <label for="region_id" class="form-label">Region</label>
                                <select class="form-select" id="region_id" name="region_id" required>
                                    <option value="">Select Region</option>
                                    <?php while ($region = $regions_result->fetch_assoc()) { ?>
                                        <option value="<?php echo $region['id']; ?>"><?php echo htmlspecialchars($region['name']); ?></option>
                                    <?php } ?>
                                    <?php $regions_result->data_seek(0); ?>
                                </select>
                            </div>
                            <button type="submit" name="add_client" class="btn btn-primary">Add Client</button>
                        </form>
                    </div>
                </div>
            </div>

            <!-- Clients Tab (Sales) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'sales_clients') ? 'show active' : ''; ?>" id="sales_clients" role="tabpanel" aria-labelledby="sales-clients-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Clients</h5>
                        <a href="main.php?export_clients=1" class="btn btn-sm btn-outline-secondary float-end">Export to CSV</a>
                    </div>
                    <div class="card-body">
                        <table id="clientsTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Phone</th>
                                    <th>Address</th>
                                    <th>Details</th>
                                    <th>Region</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $clients_result->fetch_assoc()) { ?>
                                    <tr class="<?php echo isset($_GET['highlight']) && $_GET['highlight'] == $client['id'] ? 'table-primary' : ''; ?>">
                                        <td><?php echo $client['id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_email']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_phone'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['address'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['region_name']); ?></td>
                                        <td>
                                            <span class="badge <?php
                                                if ($client['status'] == 'New') echo 'badge-secondary';
                                                elseif ($client['status'] == 'Assignment Pending') echo 'badge-pending';
                                                elseif ($client['status'] == 'Assigned' || $client['status'] == 'In Progress') echo 'badge-inprogress';
                                                elseif ($client['status'] == 'Details Submitted') echo 'badge-primary';
                                                elseif ($client['status'] == 'Completed') echo 'badge-completed';
                                                else echo 'badge-danger';
                                            ?>">
                                                <?php echo htmlspecialchars($client['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($client['status'] == 'New') { ?>
                                                <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#editClientModal<?php echo $client['id']; ?>">Edit</button>
                                            <?php } ?>
                                        </td>
                                    </tr>
                                    <!-- Edit Client Modal -->
                                    <div class="modal fade" id="editClientModal<?php echo $client['id']; ?>" tabindex="-1" aria-labelledby="editClientModalLabel<?php echo $client['id']; ?>" aria-hidden="true">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title" id="editClientModalLabel<?php echo $client['id']; ?>">Edit Client</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <form method="POST">
                                                        <input type="hidden" name="client_id" value="<?php echo $client['id']; ?>">
                                                        <div class="mb-3">
                                                            <label for="name<?php echo $client['id']; ?>" class="form-label">Name</label>
                                                            <input type="text" class="form-control" id="name<?php echo $client['id']; ?>" name="name" value="<?php echo htmlspecialchars($client['name']); ?>" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="contact_email<?php echo $client['id']; ?>" class="form-label">Contact Email</label>
                                                            <input type="email" class="form-control" id="contact_email<?php echo $client['id']; ?>" name="contact_email" value="<?php echo htmlspecialchars($client['contact_email']); ?>" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="contact_phone<?php echo $client['id']; ?>" class="form-label">Contact Phone</label>
                                                            <input type="text" class="form-control" id="contact_phone<?php echo $client['id']; ?>" name="contact_phone" value="<?php echo htmlspecialchars($client['contact_phone'] ?? ''); ?>">
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="address<?php echo $client['id']; ?>" class="form-label">Address</label>
                                                            <textarea class="form-control" id="address<?php echo $client['id']; ?>" name="address" rows="4"><?php echo htmlspecialchars($client['address'] ?? ''); ?></textarea>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="details<?php echo $client['id']; ?>" class="form-label">Details</label>
                                                            <textarea class="form-control" id="details<?php echo $client['id']; ?>" name="details" rows="4"><?php echo htmlspecialchars($client['details'] ?? ''); ?></textarea>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="region_id<?php echo $client['id']; ?>" class="form-label">Region</label>
                                                            <select class="form-select" id="region_id<?php echo $client['id']; ?>" name="region_id" required>
                                                                <option value="">Select Region</option>
                                                                <?php
                                                                $regions_result->data_seek(0);
                                                                while ($region = $regions_result->fetch_assoc()) { ?>
                                                                    <option value="<?php echo $region['id']; ?>" <?php echo $region['id'] == $client['region_id'] ? 'selected' : ''; ?>><?php echo htmlspecialchars($region['name']); ?></option>
                                                                <?php } ?>
                                                                <?php $regions_result->data_seek(0); ?>
                                                            </select>
                                                        </div>
                                                        <button type="submit" name="update_client" class="btn btn-primary">Update Client</button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Pending Assignments Tab (Tech) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_pending') ? 'show active' : ''; ?>" id="tech_pending" role="tabpanel" aria-labelledby="tech-pending-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Pending Assignments</h5>
                    </div>
                    <div class="card-body">
                        <table id="pendingTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Technician</th>
                                    <th>Details</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $pending_result->fetch_assoc()) { ?>
                                    <tr>
                                        <td><?php echo $client['id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['tech_name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['details'] ?? 'N/A'); ?></td>
                                        <td>
                                            <a href="main.php?approve=<?php echo $client['id']; ?>" class="btn btn-sm btn-outline-success">Approve</a>
                                        </td>
                                    </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Details Submitted Tab (Tech) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_submitted') ? 'show active' : ''; ?>" id="tech_submitted" role="tabpanel" aria-labelledby="tech-submitted-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Details Submitted</h5>
                    </div>
                    <div class="card-body">
                        <table id="submittedTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Region</th>
                                    <th>Technician</th>
                                    <th>Gadget Details</th>
                                    <th>Vehicle Details</th>
                                    <th>Installation Date</th>
                                    <th>Notes</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $submitted_result->fetch_assoc()) { ?>
                                    <tr>
                                        <td><?php echo $client['id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['region_name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['tech_name'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['gadget_details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['vehicle_details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['installation_date'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['notes'] ?? 'N/A'); ?></td>
                                        <td>
                                            <a href="main.php?complete=<?php echo $client['id']; ?>" class="btn btn-sm btn-outline-success">Mark Completed</a>
                                        </td>
                                    </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- All Clients Tab (Tech) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_allclients') ? 'show active' : ''; ?>" id="tech_allclients" role="tabpanel" aria-labelledby="tech-allclients-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">All Clients</h5>
                    </div>
                    <div class="card-body">
                        <table id="allClientsTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Phone</th>
                                    <th>Address</th>
                                    <th>Details</th>
                                    <th>Region</th>
                                    <th>Status</th>
                                    <th>Sales</th>
                                    <th>Technician</th>
                                    <th>Approved</th>
                                    <th>Assign</th>
                                    <th>Status Update</th>
                                    <th>Cancel</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $allclients_result->fetch_assoc()) { ?>
                                    <tr class="<?php echo isset($_GET['highlight']) && $_GET['highlight'] == $client['id'] ? 'table-primary' : ''; ?>">
                                        <td><?php echo $client['id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_email']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_phone'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['address'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['region_name']); ?></td>
                                        <td>
                                            <span class="badge <?php
                                                if ($client['status'] == 'New') echo 'badge-secondary';
                                                elseif ($client['status'] == 'Assignment Pending') echo 'badge-pending';
                                                elseif ($client['status'] == 'Assigned' || $client['status'] == 'In Progress') echo 'badge-inprogress';
                                                elseif ($client['status'] == 'Details Submitted') echo 'badge-primary';
                                                else echo 'badge-danger';
                                            ?>">
                                                <?php echo htmlspecialchars($client['status']); ?>
                                            </span>
                                        </td>
                                        <td><?php echo htmlspecialchars($client['sales_name'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['tech_name'] ?? 'N/A'); ?></td>
                                        <td><?php echo $client['approved'] ? 'Yes' : 'No'; ?></td>
                                        <td>
                                            <?php if (!$client['assigned_tech_id'] && $client['status'] == 'New') { ?>
                                                <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#assignModal<?php echo $client['id']; ?>">Assign</button>
                                            <?php } ?>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-info" data-bs-toggle="modal" data-bs-target="#statusModal<?php echo $client['id']; ?>">Update</button>
                                        </td>
                                        <td>
                                            <button class="btn btn-sm btn-outline-danger" data-bs-toggle="modal" data-bs-target="#cancelModal<?php echo $client['id']; ?>">Cancel</button>
                                        </td>
                                    </tr>
                                    <!-- Assign Technician Modal -->
                                    <div class="modal fade" id="assignModal<?php echo $client['id']; ?>" tabindex="-1" aria-labelledby="assignModalLabel<?php echo $client['id']; ?>" aria-hidden="true">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title" id="assignModalLabel<?php echo $client['id']; ?>">Assign Technician</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <form method="POST">
                                                        <input type="hidden" name="client_id" value="<?php echo $client['id']; ?>">
                                                        <div class="mb-3">
                                                            <label for="technician_id<?php echo $client['id']; ?>" class="form-label">Technician</label>
                                                            <select class="form-select" id="technician_id<?php echo $client['id']; ?>" name="technician_id" required>
                                                                <option value="">Select Technician</option>
                                                                <?php foreach ($technicians as $tech) { ?>
                                                                    <option value="<?php echo $tech['id']; ?>"><?php echo htmlspecialchars($tech['username']); ?></option>
                                                                <?php } ?>
                                                            </select>
                                                        </div>
                                                        <button type="submit" name="assign_technician" class="btn btn-primary">Assign</button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <!-- Update Status Modal -->
                                    <div class="modal fade" id="statusModal<?php echo $client['id']; ?>" tabindex="-1" aria-labelledby="statusModalLabel<?php echo $client['id']; ?>" aria-hidden="true">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title" id="statusModalLabel<?php echo $client['id']; ?>">Update Status</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <form method="POST">
                                                        <input type="hidden" name="client_id" value="<?php echo $client['id']; ?>">
                                                        <div class="mb-3">
                                                            <label for="status<?php echo $client['id']; ?>" class="form-label">Status</label>
                                                            <select class="form-select" id="status<?php echo $client['id']; ?>" name="status" required>
                                                                <option value="New" <?php echo $client['status'] == 'New' ? 'selected' : ''; ?>>New</option>
                                                                <option value="Assignment Pending" <?php echo $client['status'] == 'Assignment Pending' ? 'selected' : ''; ?>>Assignment Pending</option>
                                                                <option value="Assigned" <?php echo $client['status'] == 'Assigned' ? 'selected' : ''; ?>>Assigned</option>
                                                                <option value="In Progress" <?php echo $client['status'] == 'In Progress' ? 'selected' : ''; ?>>In Progress</option>
                                                                <option value="Details Submitted" <?php echo $client['status'] == 'Details Submitted' ? 'selected' : ''; ?>>Details Submitted</option>
                                                                <option value="Completed" <?php echo $client['status'] == 'Completed' ? 'selected' : ''; ?>>Completed</option>
                                                                <option value="Cancelled" <?php echo $client['status'] == 'Cancelled' ? 'selected' : ''; ?>>Cancelled</option>
                                                            </select>
                                                        </div>
                                                        <button type="submit" name="update_status" class="btn btn-primary">Update</button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <!-- Cancel Client Modal -->
                                    <div class="modal fade" id="cancelModal<?php echo $client['id']; ?>" tabindex="-1" aria-labelledby="cancelModalLabel<?php echo $client['id']; ?>" aria-hidden="true">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title" id="cancelModalLabel<?php echo $client['id']; ?>">Cancel Client</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <form method="POST">
                                                        <input type="hidden" name="client_id" value="<?php echo $client['id']; ?>">
                                                        <div class="mb-3">
                                                            <label for="cancel_reason<?php echo $client['id']; ?>" class="form-label">Reason for Cancellation</label>
                                                            <textarea class="form-control" id="cancel_reason<?php echo $client['id']; ?>" name="cancel_reason" rows="4" required></textarea>
                                                        </div>
                                                        <button type="submit" name="cancel_client" class="btn btn-danger">Cancel Client</button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Completed Installations Tab (Tech) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_completed') ? 'show active' : ''; ?>" id="tech_completed" role="tabpanel" aria-labelledby="tech-completed-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Completed Installations</h5>
                        <form method="GET" class="float-end filter-form">
                            <input type="hidden" name="tab" value="tech_completed">
                            <div class="d-flex align-items-end">
                                <div class="me-2">
                                    <label for="from_date_completed" class="form-label">From</label>
                                    <input type="date" class="form-control form-control-sm" id="from_date_completed" name="from_date" value="<?php echo htmlspecialchars($from_date ?? ''); ?>">
                                </div>
                                <div class="me-2">
                                    <label for="to_date_completed" class="form-label">To</label>
                                    <input type="date" class="form-control form-control-sm" id="to_date_completed" name="to_date" value="<?php echo htmlspecialchars($to_date ?? ''); ?>">
                                </div>
                                <button type="submit" class="btn btn-sm btn-primary me-2">Filter</button>
                                <a href="main.php?export_installations=1<?php echo $from_date ? '&from_date=' . urlencode($from_date) : ''; ?><?php echo $to_date ? '&to_date=' . urlencode($to_date) : ''; ?>" class="btn btn-sm btn-outline-secondary">Export to Excel</a>
                            </div>
                        </form>
                    </div>
                    <div class="card-body">
                        <table id="completedTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Phone</th>
                                    <th>Address</th>
                                    <th>Region</th>
                                    <th>Gadget Details</th>
                                    <th>Vehicle Details</th>
                                    <th>Notes</th>
                                    <th>Installation Date</th>
                                    <th>Sales</th>
                                    <th>Technician</th>
                                    <th>Client Details</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $completed_result->fetch_assoc()) { ?>
                                    <tr class="<?php echo isset($_GET['highlight']) && $_GET['highlight'] == $client['client_id'] ? 'table-primary' : ''; ?>">
                                        <td><?php echo $client['client_id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_email']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_phone'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['address'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['region_name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['gadget_details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['vehicle_details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['notes'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['installation_date'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['sales_username'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['tech_username'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['client_details'] ?? 'N/A'); ?></td>
                                    </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Audit Logs Tab (Tech) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'tech_audit') ? 'show active' : ''; ?>" id="tech_audit" role="tabpanel" aria-labelledby="tech-audit-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Audit Logs</h5>
                        <form method="GET" class="float-end filter-form">
                            <input type="hidden" name="tab" value="tech_audit">
                            <div class="d-flex align-items-end">
                                <div class="me-2">
                                    <label for="from_date_audit" class="form-label">From</label>
                                    <input type="date" class="form-control form-control-sm" id="from_date_audit" name="from_date" value="<?php echo htmlspecialchars($from_date ?? ''); ?>">
                                </div>
                                <div class="me-2">
                                    <label for="to_date_audit" class="form-label">To</label>
                                    <input type="date" class="form-control form-control-sm" id="to_date_audit" name="to_date" value="<?php echo htmlspecialchars($to_date ?? ''); ?>">
                                </div>
                                <button type="submit" class="btn btn-sm btn-primary me-2">Filter</button>
                                <a href="main.php?export_audit=1<?php echo $from_date ? '&from_date=' . urlencode($from_date) : ''; ?><?php echo $to_date ? '&to_date=' . urlencode($to_date) : ''; ?>" class="btn btn-sm btn-outline-secondary">Export to Excel</a>
                            </div>
                        </form>
                    </div>
                    <div class="card-body">
                        <table id="auditTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>User</th>
                                    <th>Action</th>
                                    <th>Details</th>
                                    <th>Timestamp</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($log = $audit_result->fetch_assoc()) { ?>
                                    <tr>
                                        <td><?php echo $log['id']; ?></td>
                                        <td><?php echo htmlspecialchars($log['username']); ?></td>
                                        <td><?php echo htmlspecialchars($log['action']); ?></td>
                                        <td><?php echo htmlspecialchars($log['details']); ?></td>
                                        <td><?php echo htmlspecialchars($log['timestamp']); ?></td>
                                    </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Available Installations Tab (Technician) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_available') ? 'show active' : ''; ?>" id="technician_available" role="tabpanel" aria-labelledby="technician-available-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Available Installations</h5>
                    </div>
                    <div class="card-body">
                        <table id="availableTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Details</th>
                                    <th>Region</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $available_result->fetch_assoc()) { ?>
                                    <tr class="<?php echo isset($_GET['highlight']) && $_GET['highlight'] == $client['id'] ? 'table-primary' : ''; ?>">
                                        <td><?php echo $client['id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['region_name']); ?></td>
                                        <td>
                                            <a href="main.php?assign_self=<?php echo $client['id']; ?>" class="btn btn-sm btn-outline-primary">Assign to Self</a>
                                        </td>
                                    </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Assigned Installations Tab (Technician) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_assigned') ? 'show active' : ''; ?>" id="technician_assigned" role="tabpanel" aria-labelledby="technician-assigned-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Assigned Installations</h5>
                    </div>
                    <div class="card-body">
                        <table id="technicianAssignedTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Details</th>
                                    <th>Region</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $technician_assigned_result->fetch_assoc()) { ?>
                                    <tr class="<?php echo isset($_GET['highlight']) && $_GET['highlight'] == $client['id'] ? 'table-primary' : ''; ?>">
                                        <td><?php echo $client['id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['region_name']); ?></td>
                                        <td>
                                            <span class="badge <?php
                                                if ($client['status'] == 'Assigned') echo 'badge-inprogress';
                                                elseif ($client['status'] == 'In Progress') echo 'badge-inprogress';
                                                elseif ($client['status'] == 'Details Submitted') echo 'badge-primary';
                                                else echo 'badge-secondary';
                                            ?>">
                                                <?php echo htmlspecialchars($client['status']); ?>
                                            </span>
                                        </td>
                                        <td>
                                            <?php if ($client['status'] != 'Details Submitted') { ?>
                                                <button class="btn btn-sm btn-outline-primary" data-bs-toggle="modal" data-bs-target="#submitDetailsModal<?php echo $client['id']; ?>">Submit Details</button>
                                            <?php } else { ?>
                                                <button class="btn btn-sm btn-outline-info" data-bs-toggle="modal" data-bs-target="#editDetailsModal<?php echo $client['id']; ?>">Edit Details</button>
                                                <a href="main.php?complete_technician=<?php echo $client['id']; ?>" class="btn btn-sm btn-outline-success">Mark Completed</a>
                                            <?php } ?>
                                        </td>
                                    </tr>
                                    <!-- Submit Details Modal -->
                                    <div class="modal fade" id="submitDetailsModal<?php echo $client['id']; ?>" tabindex="-1" aria-labelledby="submitDetailsModalLabel<?php echo $client['id']; ?>" aria-hidden="true">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title" id="submitDetailsModalLabel<?php echo $client['id']; ?>">Submit Installation Details</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <form method="POST">
                                                        <input type="hidden" name="client_id" value="<?php echo $client['id']; ?>">
                                                        <div class="mb-3">
                                                            <label for="gadget_details<?php echo $client['id']; ?>" class="form-label">Gadget Details</label>
                                                            <textarea class="form-control" id="gadget_details<?php echo $client['id']; ?>" name="gadget_details" rows="4" required></textarea>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="vehicle_details<?php echo $client['id']; ?>" class="form-label">Vehicle Details</label>
                                                            <textarea class="form-control" id="vehicle_details<?php echo $client['id']; ?>" name="vehicle_details" rows="4" required></textarea>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="installation_date<?php echo $client['id']; ?>" class="form-label">Installation Date</label>
                                                            <input type="date" class="form-control" id="installation_date<?php echo $client['id']; ?>" name="installation_date" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="notes<?php echo $client['id']; ?>" class="form-label">Notes</label>
                                                            <textarea class="form-control" id="notes<?php echo $client['id']; ?>" name="notes" rows="4"></textarea>
                                                        </div>
                                                        <button type="submit" name="submit_details" class="btn btn-primary">Submit</button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    <!-- Edit Details Modal -->
                                    <div class="modal fade" id="editDetailsModal<?php echo $client['id']; ?>" tabindex="-1" aria-labelledby="editDetailsModalLabel<?php echo $client['id']; ?>" aria-hidden="true">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h5 class="modal-title" id="editDetailsModalLabel<?php echo $client['id']; ?>">Edit Installation Details</h5>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <form method="POST">
                                                        <input type="hidden" name="client_id" value="<?php echo $client['id']; ?>">
                                                        <div class="mb-3">
                                                            <label for="gadget_details_edit<?php echo $client['id']; ?>" class="form-label">Gadget Details</label>
                                                            <textarea class="form-control" id="gadget_details_edit<?php echo $client['id']; ?>" name="gadget_details" rows="4" required><?php echo htmlspecialchars($client['gadget_details'] ?? ''); ?></textarea>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="vehicle_details_edit<?php echo $client['id']; ?>" class="form-label">Vehicle Details</label>
                                                            <textarea class="form-control" id="vehicle_details_edit<?php echo $client['id']; ?>" name="vehicle_details" rows="4" required><?php echo htmlspecialchars($client['vehicle_details'] ?? ''); ?></textarea>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="installation_date_edit<?php echo $client['id']; ?>" class="form-label">Installation Date</label>
                                                            <input type="date" class="form-control" id="installation_date_edit<?php echo $client['id']; ?>" name="installation_date" value="<?php echo htmlspecialchars($client['installation_date'] ?? ''); ?>" required>
                                                        </div>
                                                        <div class="mb-3">
                                                            <label for="notes_edit<?php echo $client['id']; ?>" class="form-label">Notes</label>
                                                            <textarea class="form-control" id="notes_edit<?php echo $client['id']; ?>" name="notes" rows="4"><?php echo htmlspecialchars($client['notes'] ?? ''); ?></textarea>
                                                        </div>
                                                        <button type="submit" name="update_details" class="btn btn-primary">Update</button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>

            <!-- Completed Installations Tab (Technician) -->
            <div class="tab-pane fade <?php echo (isset($_GET['tab']) && $_GET['tab'] == 'technician_completed') ? 'show active' : ''; ?>" id="technician_completed" role="tabpanel" aria-labelledby="technician-completed-tab">
                <div class="card">
                    <div class="card-header">
                        <h5 class="card-title">Completed Installations</h5>
                    </div>
                    <div class="card-body">
                        <table id="completedTable" class="table table-striped">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Phone</th>
                                    <th>Address</th>
                                    <th>Region</th>
                                    <th>Gadget Details</th>
                                    <th>Vehicle Details</th>
                                    <th>Notes</th>
                                    <th>Installation Date</th>
                                    <th>Sales</th>
                                    <th>Technician</th>
                                    <th>Client Details</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while ($client = $completed_result->fetch_assoc()) { ?>
                                    <tr class="<?php echo isset($_GET['highlight']) && $_GET['highlight'] == $client['client_id'] ? 'table-primary' : ''; ?>">
                                        <td><?php echo $client['client_id']; ?></td>
                                        <td><?php echo htmlspecialchars($client['name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_email']); ?></td>
                                        <td><?php echo htmlspecialchars($client['contact_phone'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['address'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['region_name']); ?></td>
                                        <td><?php echo htmlspecialchars($client['gadget_details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['vehicle_details'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['notes'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['installation_date'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['sales_username'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['tech_username'] ?? 'N/A'); ?></td>
                                        <td><?php echo htmlspecialchars($client['client_details'] ?? 'N/A'); ?></td>
                                    </tr>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            // Initialize DataTables
            $('#clientsTable, #pendingTable, #submittedTable, #allClientsTable, #completedTable, #auditTable, #availableTable, #technicianAssignedTable, #notificationsTable, #recentActivityTable').DataTable({
                "pageLength": 10,
                "lengthMenu": [10, 25, 50, 100],
                "order": [],
                "searching": true,
                "paging": true,
                "info": true,
                "language": {
                    "search": "Filter records:"
                }
            });

            // Initialize Charts
            const regionChart = new Chart(document.getElementById('regionChart'), {
                type: 'bar',
                data: {
                    labels: <?php echo json_encode($region_labels); ?>,
                    datasets: [{
                        label: 'Clients by Region',
                        data: <?php echo json_encode($region_data); ?>,
                        backgroundColor: 'rgba(54, 162, 235, 0.6)',
                        borderColor: 'rgba(54, 162, 235, 1)',
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    scales: {
                        y: { beginAtZero: true, title: { display: true, text: 'Number of Clients' } },
                        x: { title: { display: true, text: 'Region' } }
                    },
                    plugins: { legend: { display: false } }
                }
            });

            const statusChart = new Chart(document.getElementById('statusChart'), {
                type: 'pie',
                data: {
                    labels: <?php echo json_encode($status_labels); ?>,
                    datasets: [{
                        label: 'Client Status Distribution',
                        data: <?php echo json_encode($status_data); ?>,
                        backgroundColor: <?php echo json_encode($status_colors); ?>,
                        borderColor: '#fff',
                        borderWidth: 2
                    }]
                },
                options: {
                    responsive: true,
                    plugins: {
                        legend: { position: 'right' },
                        tooltip: {
                            callbacks: {
                                label: function(context) {
                                    let label = context.label || '';
                                    let value = context.parsed || 0;
                                    let total = context.dataset.data.reduce((a, b) => a + b, 0);
                                    let percentage = ((value / total) * 100).toFixed(1);
                                    return `${label}: ${value} (${percentage}%)`;
                                }
                            }
                        }
                    }
                }
            });

            const technicianChart = new Chart(document.getElementById('technicianChart'), {
                type: 'bar',
                data: {
                    labels: <?php echo json_encode($technician_labels); ?>,
                    datasets: [{
                        label: 'Technician Workload',
                        data: <?php echo json_encode($technician_data); ?>,
                        backgroundColor: 'rgba(255, 99, 132, 0.6)',
                        borderColor: 'rgba(255, 99, 132, 1)',
                        borderWidth: 1
                    }]
                },
                options: {
                    responsive: true,
                    scales: {
                        y: { beginAtZero: true, title: { display: true, text: 'Number of Assignments' } },
                        x: { title: { display: true, text: 'Technician' } }
                    },
                    plugins: { legend: { display: false } }
                }
            });

            // Show toast if present
            $('.toast').toast({ delay: 3000 });
            $('.toast').toast('show');

            // Highlight row if specified in URL
            <?php if (isset($_GET['highlight'])) { ?>
                setTimeout(function() {
                    let element = document.querySelector('tr.table-primary');
                    if (element) {
                        element.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    }
                }, 500);
            <?php } ?>
        });
    </script>
</body>
</html>