<?php
// manage_users.php - Superuser interface for managing user accounts
session_start();
require_once 'db.php';
require_once 'header.php';

if (!isset($_SESSION['user_id']) || $_SESSION['role'] !== 'superuser') {
    header('Location: login.php');
    exit;
}

$page_title = 'Manage Users';
$user_id = $_SESSION['user_id'];
$errors = [];
$success = '';

// Handle user addition
if (isset($_POST['add_user'])) {
    $username = trim($_POST['username']);
    $password = trim($_POST['password']);
    $role = $_POST['role'];
    $email = trim($_POST['email']);
    $phone = trim($_POST['phone']);
    
    if (empty($username) || empty($password) || empty($role)) {
        $errors[] = "Username, password, and role are required.";
    } else {
        $stmt = $conn->prepare("SELECT id FROM users WHERE username = ?");
        $stmt->bind_param("s", $username);
        $stmt->execute();
        if ($stmt->get_result()->num_rows > 0) {
            $errors[] = "Username already exists.";
        }
        $stmt->close();
    }
    
    if (empty($errors)) {
        $stmt = $conn->prepare("INSERT INTO users (username, password, role, email, phone) VALUES (?, ?, ?, ?, ?)");
        $stmt->bind_param("sssss", $username, $password, $role, $email, $phone);
        if ($stmt->execute()) {
            $success = "User added successfully.";
            // Log action
            $action = "Added user";
            $details = "Username: $username, Role: $role";
            $stmt = $conn->prepare("INSERT INTO audit_logs (user_id, action, details, timestamp) VALUES (?, ?, ?, NOW())");
            $stmt->bind_param("iss", $user_id, $action, $details);
            $stmt->execute();
            $stmt->close();
        } else {
            $errors[] = "Failed to add user.";
        }
        $stmt->close();
    }
}

// Handle user deletion
if (isset($_GET['delete_user'])) {
    $delete_id = $_GET['delete_user'];
    $stmt = $conn->prepare("SELECT username FROM users WHERE id = ?");
    $stmt->bind_param("i", $delete_id);
    $stmt->execute();
    $result = $stmt->get_result()->fetch_assoc();
    $username = $result['username'];
    $stmt->close();
    
    $stmt = $conn->prepare("DELETE FROM users WHERE id = ? AND id != ?");
    $stmt->bind_param("ii", $delete_id, $user_id);
    if ($stmt->execute() && $stmt->affected_rows > 0) {
        $success = "User deleted successfully.";
        // Log action
        $action = "Deleted user";
        $details = "Username: $username";
        $stmt = $conn->prepare("INSERT INTO audit_logs (user_id, action, details, timestamp) VALUES (?, ?, ?, NOW())");
        $stmt->bind_param("iss", $user_id, $action, $details);
        $stmt->execute();
        $stmt->close();
    } else {
        $errors[] = "Failed to delete user or user is yourself.";
    }
    $stmt->close();
}

function tableExists($conn, $table_name) {
    $stmt = $conn->prepare("SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ?");
    if ($stmt === false) {
        return false;
    }
    $stmt->bind_param("s", $table_name);
    $stmt->execute();
    $result = $stmt->get_result();
    $exists = $result->num_rows > 0;
    $stmt->close();
    return $exists;
}

// Fetch all users
$users = [];
if (tableExists($conn, 'users')) {
    $stmt = $conn->prepare("SELECT id, username, role FROM users");
    $stmt->execute();
    $users = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($page_title); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
            font-family: 'Inter', sans-serif;
            color: #212529;
        }
        .container {
            max-width: 1200px;
            padding: 2rem 1rem;
        }
        .card {
            border: none;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
        }
        .card-header {
            background-color: #ffffff;
            border-bottom: 1px solid #e9ecef;
            padding: 1.5rem;
            border-radius: 12px 12px 0 0;
            display: flex;
            align-items: center;
        }
        .card-header h5 {
            font-weight: 600;
            color: #1a1a1a;
            margin: 0;
        }
        .card-body {
            padding: 1.5rem;
        }
        .table {
            border-radius: 8px;
            overflow: hidden;
            background-color: #ffffff;
        }
        .table th {
            background-color: #005670;
            color: #ffffff;
            font-weight: 500;
            padding: 1rem;
        }
        .table td {
            vertical-align: middle;
            padding: 0.75rem;
        }
        .table tr {
            transition: background-color 0.2s ease;
        }
        .table tr:hover {
            background-color: #f1f5f9;
        }
        .btn-primary {
            background-color: #005670;
            border-color: #005670;
            border-radius: 8px;
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary:hover {
            background-color: #003d53;
            border-color: #003d53;
            transform: scale(1.03);
        }
        .btn-secondary {
            background-color: #6c757d;
            border-color: #6c757d;
            border-radius: 8px;
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-secondary:hover {
            background-color: #5c636a;
            transform: scale(1.03);
        }
        .btn-outline-primary, .btn-outline-danger {
            border-radius: 8px;
            padding: 0.4rem 1rem;
            font-size: 0.875rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-outline-primary {
            border-color: #005670;
            color: #005670;
        }
        .btn-outline-primary:hover {
            background-color: #005670;
            color: #ffffff;
            transform: scale(1.03);
        }
        .btn-outline-danger {
            border-color: #dc3545;
            color: #dc3545;
        }
        .btn-outline-danger:hover {
            background-color: #dc3545;
            color: #ffffff;
            transform: scale(1.03);
        }
        .form-control, .form-select {
            border-radius: 8px;
            border: 1px solid #ced4da;
            transition: border-color 0.3s ease, box-shadow 0.3s ease;
        }
        .form-control:focus, .form-select:focus {
            border-color: #005670;
            box-shadow: 0 0 5px rgba(0, 86, 112, 0.3);
        }
        .alert {
            border-radius: 8px;
            margin-bottom: 1.5rem;
        }
        .alert-success {
            background-color: #d4edda;
            color: #155724;
        }
        .alert-danger {
            background-color: #f8d7da;
            color: #721c24;
        }
        @media (max-width: 576px) {
            .container {
                padding: 1rem;
            }
            .card {
                margin: 0 0.5rem;
            }
            .card-header, .card-body {
                padding: 1rem;
            }
            .btn {
                width: 100%;
                margin-bottom: 0.5rem;
            }
            .table {
                font-size: 0.875rem;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="mb-4">
            <a href="main.php" class="btn btn-secondary"><i class="fas fa-tachometer-alt me-2"></i>Back to Dashboard</a>
        </div>

        <div class="card mb-4">
            <div class="card-header">
                <h5><i class="fas fa-users-cog me-2"></i>Add New User</h5>
            </div>
            <div class="card-body">
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <?php foreach ($errors as $error): ?>
                            <p class="mb-0"><?php echo htmlspecialchars($error); ?></p>
                        <?php endforeach; ?>
                    </div>
                <?php endif; ?>
                <?php if ($success): ?>
                    <div class="alert alert-success"><?php echo htmlspecialchars($success); ?></div>
                <?php endif; ?>
                <form method="POST" action="manage_users.php">
                    <div class="mb-3">
                        <label for="username" class="form-label">Username</label>
                        <input type="text" class="form-control" id="username" name="username" required>
                    </div>
                    <div class="mb-3">
                        <label for="password" class="form-label">Password</label>
                        <input type="password" class="form-control" id="password" name="password" required>
                    </div>
                    <div class="mb-3">
                        <label for="role" class="form-label">Role</label>
                        <select class="form-select" id="role" name="role" required>
                            <option value="superuser">Superuser</option>
                            <option value="sales">Sales</option>
                            <option value="tech">Tech</option>
                            <option value="technician">Technician</option>
                        </select>
                    </div>
                    <div class="mb-3">
                        <label for="email" class="form-label">Email</label>
                        <input type="email" class="form-control" id="email" name="email">
                    </div>
                    <div class="mb-3">
                        <label for="phone" class="form-label">Phone</label>
                        <input type="text" class="form-control" id="phone" name="phone">
                    </div>
                    <button type="submit" name="add_user" class="btn btn-primary">Add User</button>
                </form>
            </div>
        </div>

        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-users me-2"></i>Existing Users</h5>
            </div>
            <div class="card-body">
                <?php if (empty($users)): ?>
                    <p class="text-muted">No users found.</p>
                <?php else: ?>
                    <table id="usersTable" class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>Username</th>
                                <th>Role</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($users as $user): ?>
                                <tr>
                                    <td><?php echo htmlspecialchars($user['username']); ?></td>
                                    <td><?php echo ucfirst($user['role']); ?></td>
                                    <td>
                                        <a href="edit_user.php?id=<?php echo $user['id']; ?>" class="btn btn-sm btn-outline-primary me-2">Edit</a>
                                        <a href="manage_users.php?delete_user=<?php echo $user['id']; ?>" class="btn btn-sm btn-outline-danger" onclick="return confirm('Are you sure you want to delete this user?');">Delete</a>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#usersTable').DataTable({
                pageLength: 10,
                order: [[0, "asc"]],
                responsive: true,
                language: {
                    search: "_INPUT_",
                    searchPlaceholder: "Search users..."
                }
            });
        });
    </script>
</body>
</html>

<?php include 'footer.php'; ?>