<?php
// notifications.php - View and manage user notifications
session_start();
require_once 'db.php';
require_once 'header.php';

if (!isset($_SESSION['user_id'])) {
    header('Location: login.php');
    exit;
}

$page_title = 'Notifications';
$user_id = $_SESSION['user_id'];

// Mark notification as read
if (isset($_POST['mark_read']) && isset($_POST['notification_id'])) {
    $notification_id = $_POST['notification_id'];
    $stmt = $conn->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
    $stmt->bind_param("ii", $notification_id, $user_id);
    $stmt->execute();
    $stmt->close();
    
    // Log action in audit_logs
    $action = "Marked notification as read";
    $details = "Notification ID: $notification_id";
    $stmt = $conn->prepare("INSERT INTO audit_logs (user_id, action, details, timestamp) VALUES (?, ?, ?, NOW())");
    $stmt->bind_param("iss", $user_id, $action, $details);
    $stmt->execute();
    $stmt->close();
    
    header('Location: notifications.php');
    exit;
}
function tableExists($conn, $table_name) {
    $stmt = $conn->prepare("SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ?");
    if ($stmt === false) {
        return false;
    }
    $stmt->bind_param("s", $table_name);
    $stmt->execute();
    $result = $stmt->get_result();
    $exists = $result->num_rows > 0;
    $stmt->close();
    return $exists;
}

// Fetch all notifications for the user
$notifications = [];
if (tableExists($conn, 'notifications')) {
    $stmt = $conn->prepare("SELECT n.* FROM notifications n ORDER BY n.created_at DESC");
    $stmt->execute();
    $notifications = $stmt->get_result()->fetch_all(MYSQLI_ASSOC);
    $stmt->close();
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo htmlspecialchars($page_title); ?></title>
    <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
    <link href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css" rel="stylesheet">
    <link href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.2/css/all.min.css" rel="stylesheet">
    <style>
        body {
            background-color: #f8f9fa;
            font-family: 'Inter', sans-serif;
            color: #212529;
        }
        .container {
            max-width: 1200px;
            padding: 2rem 1rem;
        }
        .card {
            border: none;
            border-radius: 12px;
            box-shadow: 0 4px 20px rgba(0, 0, 0, 0.08);
            transition: transform 0.3s ease, box-shadow 0.3s ease;
        }
        .card:hover {
            transform: translateY(-4px);
            box-shadow: 0 8px 30px rgba(0, 0, 0, 0.12);
        }
        .card-header {
            background-color: #ffffff;
            border-bottom: 1px solid #e9ecef;
            padding: 1.5rem;
            border-radius: 12px 12px 0 0;
            display: flex;
            align-items: center;
        }
        .card-header h5 {
            font-weight: 600;
            color: #1a1a1a;
            margin: 0;
        }
        .card-body {
            padding: 1.5rem;
        }
        .table {
            border-radius: 8px;
            overflow: hidden;
            background-color: #ffffff;
        }
        .table th {
            background-color: #005670;
            color: #ffffff;
            font-weight: 500;
            padding: 1rem;
        }
        .table td {
            vertical-align: middle;
            padding: 0.75rem;
        }
        .table tr {
            transition: background-color 0.2s ease;
        }
        .table tr:hover {
            background-color: #f1f5f9;
        }
        .btn-primary {
            background-color: #005670;
            border-color: #005670;
            border-radius: 8px;
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-primary:hover {
            background-color: #003d53;
            border-color: #003d53;
            transform: scale(1.03);
        }
        .btn-secondary {
            background-color: #6c757d;
            border-color: #6c757d;
            border-radius: 8px;
            padding: 0.5rem 1.5rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-secondary:hover {
            background-color: #5c636a;
            transform: scale(1.03);
        }
        .btn-outline-primary {
            border-color: #005670;
            color: #005670;
            border-radius: 8px;
            padding: 0.4rem 1rem;
            font-size: 0.875rem;
            font-weight: 500;
            transition: all 0.3s ease;
        }
        .btn-outline-primary:hover {
            background-color: #005670;
            color: #ffffff;
            transform: scale(1.03);
        }
        .text-muted {
            font-size: 0.9rem;
            color: #6c757d;
        }
        @media (max-width: 576px) {
            .container {
                padding: 1rem;
            }
            .card {
                margin: 0 0.5rem;
            }
            .card-header, .card-body {
                padding: 1rem;
            }
            .btn {
                width: 100%;
                margin-bottom: 0.5rem;
            }
            .table {
                font-size: 0.875rem;
            }
        }
    </style>
</head>
<body>
    <div class="container">
        <div class="mb-4">
            <a href="main.php" class="btn btn-secondary"><i class="fas fa-tachometer-alt me-2"></i>Back to Dashboard</a>
        </div>

        <div class="card">
            <div class="card-header">
                <h5><i class="fas fa-bell me-2"></i>My Notifications</h5>
            </div>
            <div class="card-body">
                <?php if (empty($notifications)): ?>
                    <p class="text-muted">No notifications available.</p>
                <?php else: ?>
                    <table id="notificationsTable" class="table table-striped table-hover">
                        <thead>
                            <tr>
                                <th>Date</th>
                                <th>Message</th>
                                <th>Status</th>
                                <th>Action</th>
                            </tr>
                        </thead>
                        <tbody>
                            <?php foreach ($notifications as $notification): ?>
                                <tr>
                                    <td><?php echo date('M d, Y H:i', strtotime($notification['created_at'])); ?></td>
                                    <td><?php echo htmlspecialchars($notification['message']); ?></td>
                                    <td><?php echo $notification['is_read'] ? 'Read' : 'Unread'; ?></td>
                                    <td>
                                        <?php if (!$notification['is_read']): ?>
                                            <form method="POST" action="notifications.php">
                                                <input type="hidden" name="notification_id" value="<?php echo $notification['id']; ?>">
                                                <button type="submit" name="mark_read" class="btn btn-sm btn-outline-primary">Mark as Read</button>
                                            </form>
                                        <?php endif; ?>
                                    </td>
                                </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <script src="https://cdn.jsdelivr.net/npm/jquery@3.6.0/dist/jquery.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#notificationsTable').DataTable({
                pageLength: 10,
                order: [[0, "desc"]],
                responsive: true,
                language: {
                    search: "_INPUT_",
                    searchPlaceholder: "Search notifications..."
                }
            });
        });
    </script>
</body>
</html>

<?php include 'footer.php'; ?>