<?php
// sales_dashboard.php - Sales Role Dashboard with Updates and Exports
include 'auth.php';

// Verify role
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'sales') {
    echo "Access Denied.";
    exit();
}

// Verify session user_id
if (!isset($_SESSION['user_id']) || !is_numeric($_SESSION['user_id'])) {
    die("Error: Invalid or missing user ID in session.");
}

// Handle Mark Notification Read
if (isset($_GET['mark_read'])) {
    $notif_id = $_GET['mark_read'];
    $stmt = $conn->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("ii", $notif_id, $_SESSION['user_id']);
    $stmt->execute();
    $stmt->close();
    header("Location: sales_dashboard.php");
    exit();
}

// Handle Add Client
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['add_client'])) {
    $name = $_POST['name'];
    $contact_email = $_POST['contact_email'];
    $contact_phone = $_POST['contact_phone'];
    $address = $_POST['address'];
    $details = $_POST['details'];
    $region_id = $_POST['region_id'];
    $created_by = $_SESSION['user_id'];

    $stmt = $conn->prepare("INSERT INTO clients (name, contact_email, contact_phone, address, details, region_id, created_by, status, created_at) VALUES (?, ?, ?, ?, ?, ?, ?, 'New', NOW())");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("sssssii", $name, $contact_email, $contact_phone, $address, $details, $region_id, $created_by);
    if ($stmt->execute()) {
        $client_id = $stmt->insert_id;
        log_audit($conn, $_SESSION['user_id'], 'Add Client', "Client ID: " . $client_id);

        // Notify technicians in region with no pending tasks
        $tech_sql = "SELECT u.id FROM users u 
                     LEFT JOIN clients c ON u.id = c.assigned_tech_id AND c.status IN ('Assignment Pending', 'Assigned', 'In Progress', 'Details Submitted') 
                     WHERE u.role = 'technician' AND u.region_id = ? AND c.id IS NULL GROUP BY u.id";
        $tech_stmt = $conn->prepare($tech_sql);
        if ($tech_stmt === false) {
            die("Prepare failed: " . $conn->error);
        }
        $tech_stmt->bind_param("i", $region_id);
        $tech_stmt->execute();
        $tech_result = $tech_stmt->get_result();
        while ($tech_row = $tech_result->fetch_assoc()) {
            $message = "New installation available: Client " . $name . " (ID: " . $client_id . ") in your region. Only available technicians notified.";
            send_notification($conn, $tech_row['id'], $message);
        }
        $tech_stmt->close();

        echo "<div class='alert alert-success'>Client added successfully.</div>";
    } else {
        echo "<div class='alert alert-danger'>Error: " . $conn->error . "</div>";
    }
    $stmt->close();
}

// Handle Update Client
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_client'])) {
    $client_id = $_POST['client_id'];
    $name = $_POST['name'];
    $contact_email = $_POST['contact_email'];
    $contact_phone = $_POST['contact_phone'];
    $address = $_POST['address'];
    $details = $_POST['details'];
    $region_id = $_POST['region_id'];

    $stmt = $conn->prepare("UPDATE clients SET name=?, contact_email=?, contact_phone=?, address=?, details=?, region_id=? WHERE id=? AND status='New' AND created_by=?");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("sssssiii", $name, $contact_email, $contact_phone, $address, $details, $region_id, $client_id, $_SESSION['user_id']);
    if ($stmt->execute()) {
        log_audit($conn, $_SESSION['user_id'], 'Update Client', "Client ID: $client_id");
        echo "<div class='alert alert-success'>Client updated successfully.</div>";
    } else {
        echo "<div class='alert alert-danger'>Error: " . $conn->error . "</div>";
    }
    $stmt->close();
}

// Export Clients to CSV
if (isset($_GET['export_clients'])) {
    header('Content-Type: text/csv');
    header('Content-Disposition: attachment; filename="clients.csv"');
    $output = fopen('php://output', 'w');
    fputcsv($output, ['ID', 'Name', 'Email', 'Phone', 'Address', 'Details', 'Region', 'Status', 'Created At']);

    $sql = "SELECT c.id, c.name, c.contact_email, c.contact_phone, c.address, c.details, r.name AS region, c.status, c.created_at 
            FROM clients c JOIN regions r ON c.region_id = r.id";
    $result = $conn->query($sql);
    if ($result === false) {
        die("Query failed: " . $conn->error);
    }
    while ($row = $result->fetch_assoc()) {
        fputcsv($output, $row);
    }
    log_audit($conn, $_SESSION['user_id'], 'Export Clients', 'Exported all clients to CSV');
    exit();
}

// Fetch regions
$regions_sql = "SELECT * FROM regions";
$regions_result = $conn->query($regions_sql);
if ($regions_result === false) {
    die("Query failed: " . $conn->error);
}

// Fetch all clients for sales
$clients_sql = "SELECT c.*, r.name AS region_name FROM clients c JOIN regions r ON c.region_id = r.id WHERE created_by = ? ORDER BY c.created_at DESC";
$clients_stmt = $conn->prepare($clients_sql);
if ($clients_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$clients_stmt->bind_param("i", $_SESSION['user_id']);
$clients_stmt->execute();
$clients_result = $clients_stmt->get_result();
$clients_stmt->close();
// Delete notifications older than 7 days
$delete_notif_sql = "DELETE FROM notifications WHERE created_at < NOW() - INTERVAL 7 DAY AND user_id = ?";
$delete_notif_stmt = $conn->prepare($delete_notif_sql);
if ($delete_notif_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$delete_notif_stmt->bind_param("i", $_SESSION['user_id']);
$delete_notif_stmt->execute();
$delete_notif_stmt->close();

// Fetch notifications
$notif_sql = "SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC";
$notif_stmt = $conn->prepare($notif_sql);
if ($notif_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$notif_stmt->bind_param("i", $_SESSION['user_id']);
$notif_stmt->execute();
$notif_result = $notif_stmt->get_result();
$notif_stmt->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Sales Dashboard - Vehicle Tracking System</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.4/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        body { background-color: #f8f9fa; }
        .card { box-shadow: 0 4px 8px rgba(0,0,0,0.1); }
        .nav-link { font-weight: 500; }
        .table { font-size: 0.9rem; }
        /* Sticky navbar (already handled by Bootstrap sticky-top) */
        .navbar { height: 56px; } /* Ensure consistent navbar height */
        /* Sticky header for heading and nav tabs */
        .sticky-header {
            position: sticky;
            top: 56px; /* Height of navbar */
            z-index: 1000;
            background-color: #f8f9fa;
            padding-top: 20px;
            padding-bottom: 10px;
        }
        .nav-tabs {
            border-bottom: 1px solid #dee2e6;
        }
        .nav-tabs .nav-link {
            color: #333;
            font-weight: 500;
            border-radius: 0;
            padding: 10px 15px;
        }
        .nav-tabs .nav-link.active {
            background-color: #007bff;
            color: white;
            border-color: #dee2e6 #dee2e6 #f8f9fa;
        }
        /* Adjust container margin to avoid overlap */
        .container { margin-top: 20px; }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark sticky-top">
        <div class="container-fluid">
            <a class="navbar-brand" href="index.php">Vehicle Tracking System</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php"><i class="fas fa-home"></i> Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link btn btn-danger text-white" href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                    </li>
                </ul>
            </div>
        </div>
    </nav>
    <div class="container">
        <div class="sticky-header">
            <h2 class="mb-4 text-center">Sales Dashboard</h2>
            <ul class="nav nav-tabs" id="salesTabs" role="tablist">
                <li class="nav-item" role="presentation">
                    <button class="nav-link active" id="notifications-tab" data-bs-toggle="tab" data-bs-target="#notifications" type="button" role="tab" aria-controls="notifications" aria-selected="true">Notifications</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="addclient-tab" data-bs-toggle="tab" data-bs-target="#addclient" type="button" role="tab" aria-controls="addclient" aria-selected="false">Add New Client</button>
                </li>
                <li class="nav-item" role="presentation">
                    <button class="nav-link" id="clients-tab" data-bs-toggle="tab" data-bs-target="#clients" type="button" role="tab" aria-controls="clients" aria-selected="false">Your Clients</button>
                </li>
            </ul>
        </div>
        <div class="row mb-3">
            <div class="col">
                <a href="?export_clients=1" class="btn btn-info"><i class="fas fa-file-export"></i> Export Clients to CSV</a>
            </div>
        </div>
        <div class="tab-content mt-4">
            <div class="tab-pane fade show active" id="notifications" role="tabpanel" aria-labelledby="notifications-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Notifications</h5>
                        <ul class="list-group">
                            <?php while($notif = $notif_result->fetch_assoc()) { ?>
                                <li class="list-group-item <?php echo $notif['is_read'] ? '' : 'fw-bold'; ?>">
                                    <?php echo htmlspecialchars($notif['message']); ?> (<?php echo htmlspecialchars($notif['created_at']); ?>)
                                    <?php if (!$notif['is_read']) { ?>
                                        - <a href="?mark_read=<?php echo $notif['id']; ?>">Mark as Read</a>
                                    <?php } ?>
                                </li>
                            <?php } ?>
                        </ul>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="addclient" role="tabpanel" aria-labelledby="addclient-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Add New Client</h5>
                        <form method="post" action="" class="form">
                            <input type="hidden" name="add_client" value="1">
                            <div class="form-group mb-3"><label>Name:</label><input type="text" name="name" class="form-control" required></div>
                            <div class="form-group mb-3"><label>Email:</label><input type="email" name="contact_email" class="form-control"></div>
                            <div class="form-group mb-3"><label>Phone:</label><input type="text" name="contact_phone" class="form-control"></div>
                            <div class="form-group mb-3"><label>Address:</label><textarea name="address" class="form-control"></textarea></div>
                            <div class="form-group mb-3"><label>Details:</label><textarea name="details" class="form-control" required></textarea></div>
                            <div class="form-group mb-3">
                                <label>Region:</label>
                                <select name="region_id" class="form-select" required>
                                    <?php while($row = $regions_result->fetch_assoc()) { ?>
                                        <option value="<?php echo $row['id']; ?>"><?php echo htmlspecialchars($row['name']); ?></option>
                                    <?php } ?>
                                </select>
                            </div>
                            <button type="submit" class="btn btn-primary"><i class="fas fa-plus"></i> Add Client</button>
                        </form>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="clients" role="tabpanel" aria-labelledby="clients-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Your Clients List (Editable if New)</h5>
                        <table class="table table-striped" id="clientsTable">
                            <thead>
                                <tr>
                                    <th>ID</th>
                                    <th>Name</th>
                                    <th>Email</th>
                                    <th>Phone</th>
                                    <th>Address</th>
                                    <th>Details</th>
                                    <th>Region</th>
                                    <th>Status</th>
                                    <th>Actions</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php while($row = $clients_result->fetch_assoc()) { ?>
                                    <tr>
                                        <td><?php echo htmlspecialchars($row['id']); ?></td>
                                        <td><?php echo htmlspecialchars($row['name']); ?></td>
                                        <td><?php echo htmlspecialchars($row['contact_email']); ?></td>
                                        <td><?php echo htmlspecialchars($row['contact_phone']); ?></td>
                                        <td><?php echo htmlspecialchars($row['address']); ?></td>
                                        <td><?php echo htmlspecialchars($row['details']); ?></td>
                                        <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                        <td><?php echo htmlspecialchars($row['status']); ?></td>
                                        <td>
                                            <?php if ($row['status'] == 'New') { ?>
                                                <button class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#editModal<?php echo $row['id']; ?>"><i class="fas fa-edit"></i> Edit</button>
                                            <?php } ?>
                                        </td>
                                    </tr>
                                    <!-- Edit Modal -->
                                    <div class="modal fade" id="editModal<?php echo $row['id']; ?>">
                                        <div class="modal-dialog">
                                            <div class="modal-content">
                                                <div class="modal-header">
                                                    <h4 class="modal-title">Edit Client</h4>
                                                    <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                </div>
                                                <div class="modal-body">
                                                    <form method="post" action="">
                                                        <input type="hidden" name="update_client" value="1">
                                                        <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                        <div class="form-group mb-3"><label>Name:</label><input type="text" name="name" class="form-control" value="<?php echo htmlspecialchars($row['name']); ?>" required></div>
                                                        <div class="form-group mb-3"><label>Email:</label><input type="email" name="contact_email" class="form-control" value="<?php echo htmlspecialchars($row['contact_email']); ?>"></div>
                                                        <div class="form-group mb-3"><label>Phone:</label><input type="text" name="contact_phone" class="form-control" value="<?php echo htmlspecialchars($row['contact_phone']); ?>"></div>
                                                        <div class="form-group mb-3"><label>Address:</label><textarea name="address" class="form-control"><?php echo htmlspecialchars($row['address']); ?></textarea></div>
                                                        <div class="form-group mb-3"><label>Details:</label><textarea name="details" class="form-control" required><?php echo htmlspecialchars($row['details']); ?></textarea></div>
                                                        <div class="form-group mb-3">
                                                            <label>Region:</label>
                                                            <select name="region_id" class="form-select" required>
                                                                <?php 
                                                                $regions_result->data_seek(0); // Reset regions
                                                                while($reg = $regions_result->fetch_assoc()) { 
                                                                    $selected = ($reg['id'] == $row['region_id']) ? 'selected' : '';
                                                                ?>
                                                                    <option value="<?php echo $reg['id']; ?>" <?php echo $selected; ?>><?php echo htmlspecialchars($reg['name']); ?></option>
                                                                <?php } ?>
                                                            </select>
                                                        </div>
                                                        <button type="submit" class="btn btn-primary"><i class="fas fa-save"></i> Update</button>
                                                    </form>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                <?php } ?>
                            </tbody>
                        </table>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.4/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#clientsTable').DataTable({
                "paging": true,
                "searching": true,
                "ordering": true,
                "info": true,
                "responsive": true
            });
        });
    </script>
</body>
</html>