<?php
// tech_dashboard.php - Tech Team Role Dashboard with Approvals, Updates, Exports, Audit View
include 'auth.php';
if ($_SESSION['role'] != 'tech') {
    echo "Access Denied.";
    exit();
}

// Define log_audit function if not defined in auth.php
if (!function_exists('log_audit')) {
    function log_audit($conn, $user_id, $action, $details) {
        $stmt = $conn->prepare("INSERT INTO audit_logs (user_id, action, details, timestamp) VALUES (?, ?, ?, NOW())");
        if ($stmt === false) {
            return;
        }
        $stmt->bind_param("iss", $user_id, $action, $details);
        $stmt->execute();
        $stmt->close();
    }
}

// Assume send_notification function exists in auth.php or elsewhere
if (!function_exists('send_notification')) {
    function send_notification($conn, $user_id, $message, $related_id, $related_tab) {
        $stmt = $conn->prepare("INSERT INTO notifications (user_id, message, related_id, related_tab, created_at) VALUES (?, ?, ?, ?, NOW())");
        if ($stmt === false) {
            return;
        }
        $stmt->bind_param("isis", $user_id, $message, $related_id, $related_tab);
        $stmt->execute();
        $stmt->close();
    }
}

// Handle Mark Notification Read
if (isset($_GET['mark_read'])) {
    $notif_id = $_GET['mark_read'];
    $tab = isset($_GET['tab']) ? $_GET['tab'] : 'notifications';
    $client_id = isset($_GET['client_id']) ? $_GET['client_id'] : null;
    $stmt = $conn->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("ii", $notif_id, $_SESSION['user_id']);
    $stmt->execute();
    $stmt->close();
    // Redirect to the specified tab with optional client_id for highlighting
    $redirect_url = "tech_dashboard.php?tab=" . urlencode($tab);
    if ($client_id) {
        $redirect_url .= "&highlight=" . urlencode($client_id);
    }
    header("Location: $redirect_url");
    exit();
}

// Handle approve assignment
if (isset($_GET['approve'])) {
    $client_id = $_GET['approve'];
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET approved = 1, status = 'Assigned' WHERE id = ? AND status = 'Assignment Pending'");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $_SESSION['user_id'], 'Approve Assignment', "Client: $client_name");
        $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) {
            die("Prepare failed: " . $conn->error);
        }
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Assignment approved for client $client_name by technician $tech_name.";
        if ($client_row['created_by']) {
            send_notification($conn, $client_row['created_by'], $message, $client_id, 'pending');
        }
        if ($client_row['assigned_tech_id']) {
            send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'assigned');
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Assignment approved.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: tech_dashboard.php");
    exit();
}

// Handle direct technician assignment
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['assign_technician'])) {
    $client_id = $_POST['client_id'];
    $tech_id = $_POST['technician_id'];
    $client_sql = "SELECT c.name FROM clients c WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $client_stmt->close();
    $tech_sql = "SELECT username FROM users WHERE id = ?";
    $tech_stmt = $conn->prepare($tech_sql);
    if ($tech_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $tech_stmt->bind_param("i", $tech_id);
    $tech_stmt->execute();
    $tech_result = $tech_stmt->get_result();
    $tech_row = $tech_result->fetch_assoc();
    $tech_name = $tech_row['username'] ?? 'Unknown Technician';
    $tech_stmt->close();
    $pending_sql = "SELECT COUNT(*) as count FROM clients WHERE assigned_tech_id = ? AND status IN ('Assignment Pending', 'Assigned', 'In Progress')";
    $pending_stmt = $conn->prepare($pending_sql);
    if ($pending_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $pending_stmt->bind_param("i", $tech_id);
    $pending_stmt->execute();
    $pending_result = $pending_stmt->get_result();
    $pending_row = $pending_result->fetch_assoc();
    if ($pending_row['count'] > 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => "Cannot assign: Technician $tech_name has pending tasks."];
        $pending_stmt->close();
        header("Location: tech_dashboard.php");
        exit();
    }
    $pending_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET assigned_tech_id = ?, status = 'Assigned', approved = 1 WHERE id = ? AND status = 'New' AND assigned_tech_id IS NULL");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("ii", $tech_id, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $_SESSION['user_id'], 'Assign Technician', "Client: $client_name to Technician: $tech_name");
        $client_sql = "SELECT created_by FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) {
            die("Prepare failed: " . $conn->error);
        }
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Technician $tech_name assigned to client $client_name.";
        if ($client_row['created_by']) {
            send_notification($conn, $client_row['created_by'], $message, $client_id, 'allclients');
        }
        send_notification($conn, $tech_id, $message, $client_id, 'assigned');
        $_SESSION['toast'] = ['type' => 'success', 'message' => "Technician $tech_name assigned to client $client_name."];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: tech_dashboard.php");
    exit();
}

// Handle complete installation
if (isset($_GET['complete'])) {
    $client_id = $_GET['complete'];
    $check_sql = "SELECT COUNT(*) AS count FROM installations WHERE client_id = ?";
    $check_stmt = $conn->prepare($check_sql);
    $check_stmt->bind_param("i", $client_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    $check_row = $check_result->fetch_assoc();
    if ($check_row['count'] == 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: No installation record found for this client.'];
        $check_stmt->close();
        header("Location: tech_dashboard.php");
        exit();
    }
    $check_stmt->close();
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET status = 'Completed' WHERE id = ? AND status = 'Details Submitted'");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $_SESSION['user_id'], 'Complete Installation', "Client: $client_name");
        $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) {
            die("Prepare failed: " . $conn->error);
        }
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Installation completed for client $client_name by technician $tech_name. Technician is now free to take new assignments.";
        if ($client_row['created_by']) {
            send_notification($conn, $client_row['created_by'], $message, $client_id, 'completed');
        }
        if ($client_row['assigned_tech_id']) {
            send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'completed');
        }
        $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
        $tech_result = $conn->query($tech_sql);
        if ($tech_result === false) {
            die("Query failed: " . $conn->error);
        }
        while ($tech_row = $tech_result->fetch_assoc()) {
            send_notification($conn, $tech_row['id'], $message, $client_id, 'completed');
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Installation completed, technician freed up, and notifications sent.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: tech_dashboard.php");
    exit();
}

// Handle update status
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_status'])) {
    $client_id = $_POST['client_id'];
    $new_status = $_POST['status'];
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    if ($new_status == 'Completed') {
        $check_sql = "SELECT COUNT(*) AS count FROM installations WHERE client_id = ?";
        $check_stmt = $conn->prepare($check_sql);
        $check_stmt->bind_param("i", $client_id);
        $check_stmt->execute();
        $check_result = $check_stmt->get_result();
        $check_row = $check_result->fetch_assoc();
        if ($check_row['count'] == 0) {
            $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: Cannot mark as Completed. No installation record found for this client.'];
            $check_stmt->close();
            header("Location: tech_dashboard.php");
            exit();
        }
        $check_stmt->close();
    }
    $stmt = $conn->prepare("UPDATE clients SET status = ?, assigned_tech_id = CASE WHEN ? = 'Cancelled' THEN NULL ELSE assigned_tech_id END WHERE id = ?");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("ssi", $new_status, $new_status, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $_SESSION['user_id'], 'Update Status', "Client: $client_name to $new_status");
        if ($new_status == 'Completed') {
            $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
            $client_stmt = $conn->prepare($client_sql);
            if ($client_stmt === false) {
                die("Prepare failed: " . $conn->error);
            }
            $client_stmt->bind_param("i", $client_id);
            $client_stmt->execute();
            $client_result = $client_stmt->get_result();
            $client_row = $client_result->fetch_assoc();
            $client_stmt->close();
            $message = "Installation status updated to Completed for client $client_name by technician $tech_name. Technician freed up.";
            if ($client_row['created_by']) {
                send_notification($conn, $client_row['created_by'], $message, $client_id, 'completed');
            }
            if ($client_row['assigned_tech_id']) {
                send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'completed');
            }
            $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
            $tech_result = $conn->query($tech_sql);
            if ($tech_result === false) {
                die("Query failed: " . $conn->error);
            }
            while ($tech_row = $tech_result->fetch_assoc()) {
                send_notification($conn, $tech_row['id'], $message, $client_id, 'completed');
            }
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Status updated.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: tech_dashboard.php");
    exit();
}

// Handle cancel client with reason
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['cancel_client'])) {
    $client_id = $_POST['client_id'];
    $cancel_reason = trim($_POST['cancel_reason']);
    if (empty($cancel_reason)) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: Cancellation reason is required.'];
        header("Location: tech_dashboard.php");
        exit();
    }
    $cancel_reason = htmlspecialchars($cancel_reason, ENT_QUOTES, 'UTF-8');
    $client_sql = "SELECT c.name, u.username AS tech_name FROM clients c LEFT JOIN users u ON c.assigned_tech_id = u.id WHERE c.id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $tech_name = $client_row['tech_name'] ?? 'Unknown Technician';
    $client_stmt->close();
    $stmt = $conn->prepare("UPDATE clients SET status = 'Cancelled', assigned_tech_id = NULL, approved = 0 WHERE id = ?");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $_SESSION['user_id'], 'Cancel Client', "Client: $client_name, Reason: $cancel_reason");
        $client_sql = "SELECT created_by, assigned_tech_id FROM clients WHERE id = ?";
        $client_stmt = $conn->prepare($client_sql);
        if ($client_stmt === false) {
            die("Prepare failed: " . $conn->error);
        }
        $client_stmt->bind_param("i", $client_id);
        $client_stmt->execute();
        $client_result = $client_stmt->get_result();
        $client_row = $client_result->fetch_assoc();
        $client_stmt->close();
        $message = "Client $client_name has been cancelled. Reason: $cancel_reason";
        if ($client_row['created_by']) {
            send_notification($conn, $client_row['created_by'], $message, $client_id, 'allclients');
        }
        if ($client_row['assigned_tech_id']) {
            send_notification($conn, $client_row['assigned_tech_id'], $message, $client_id, 'allclients');
        }
        $_SESSION['toast'] = ['type' => 'success', 'message' => "Client $client_name cancelled, technician notified with reason: $cancel_reason"];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: tech_dashboard.php");
    exit();
}

// Export Installations to Excel
if (isset($_GET['export_installations'])) {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="installations.xls"');
    $sql = "SELECT c.id AS client_id, c.name, r.name AS region_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes, c.contact_email, c.contact_phone, c.address, c.details AS client_details, u.username AS sales_username, tu.username AS tech_username
            FROM clients c
            JOIN installations i ON c.id = i.client_id
            JOIN regions r ON c.region_id = r.id
            LEFT JOIN users u ON c.created_by = u.id
            LEFT JOIN users tu ON c.assigned_tech_id = tu.id
            WHERE c.status = 'Completed'";
    $params = [];
    $types = "";
    if (isset($_GET['from_date']) && !empty($_GET['from_date'])) {
        $sql .= " AND i.installation_date >= ?";
        $params[] = $_GET['from_date'];
        $types .= "s";
    }
    if (isset($_GET['to_date']) && !empty($_GET['to_date'])) {
        $sql .= " AND i.installation_date <= ?";
        $params[] = $_GET['to_date'];
        $types .= "s";
    }
    $stmt = $conn->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $headers = ['Client ID', 'Name', 'Region', 'Gadget Details', 'Vehicle Details', 'Installation Date', 'Notes', 'Contact Email', 'Contact Phone', 'Address', 'Client Details', 'Sales Person', 'Technician'];
    echo implode("\t", $headers) . "\n";
    while ($row = $result->fetch_assoc()) {
        $line = [
            $row['client_id'],
            $row['name'],
            $row['region_name'],
            $row['gadget_details'] ?? 'N/A',
            $row['vehicle_details'] ?? 'N/A',
            $row['installation_date'] ?? 'N/A',
            $row['notes'] ?? 'N/A',
            $row['contact_email'] ?? 'N/A',
            $row['contact_phone'] ?? 'N/A',
            $row['address'] ?? 'N/A',
            $row['client_details'] ?? 'N/A',
            $row['sales_username'] ?? 'N/A',
            $row['tech_username'] ?? 'N/A'
        ];
        echo implode("\t", array_map(function($v) { return str_replace(["\r", "\n", "\t"], ' ', $v); }, $line)) . "\n";
    }
    $stmt->close();
    $filter_details = "";
    if (isset($_GET['from_date']) || isset($_GET['to_date'])) {
        $filter_details = " with filters: " . (isset($_GET['from_date']) ? "from " . $_GET['from_date'] : "") . (isset($_GET['to_date']) ? " to " . $_GET['to_date'] : "");
    }
    log_audit($conn, $_SESSION['user_id'], 'Export Installations', 'Exported completed installations to Excel' . $filter_details);
    exit();
}

// Export Audit Logs to Excel
if (isset($_GET['export_audit'])) {
    header('Content-Type: application/vnd.ms-excel');
    header('Content-Disposition: attachment; filename="audit_logs.xls"');
    $sql = "SELECT a.id, u.username, a.action, a.details, a.timestamp FROM audit_logs a JOIN users u ON a.user_id = u.id";
    $params = [];
    $types = "";
    if (isset($_GET['from_date']) && !empty($_GET['from_date'])) {
        $sql .= " WHERE a.timestamp >= ?";
        $params[] = $_GET['from_date'] . " 00:00:00";
        $types .= "s";
    }
    if (isset($_GET['to_date']) && !empty($_GET['to_date'])) {
        $sql .= (empty($params) ? " WHERE" : " AND") . " a.timestamp <= ?";
        $params[] = $_GET['to_date'] . " 23:59:59";
        $types .= "s";
    }
    $sql .= " ORDER BY a.timestamp DESC";
    $stmt = $conn->prepare($sql);
    if (!empty($params)) {
        $stmt->bind_param($types, ...$params);
    }
    $stmt->execute();
    $result = $stmt->get_result();
    $headers = ['ID', 'User', 'Action', 'Details', 'Timestamp'];
    echo implode("\t", $headers) . "\n";
    while ($row = $result->fetch_assoc()) {
        $line = [
            $row['id'],
            $row['username'],
            $row['action'],
            $row['details'],
            $row['timestamp']
        ];
        echo implode("\t", array_map(function($v) { return str_replace(["\r", "\n", "\t"], ' ', $v); }, $line)) . "\n";
    }
    $stmt->close();
    $filter_details = "";
    if (isset($_GET['from_date']) || isset($_GET['to_date'])) {
        $filter_details = " with filters: " . (isset($_GET['from_date']) ? "from " . $_GET['from_date'] : "") . (isset($_GET['to_date']) ? " to " . $_GET['to_date'] : "");
    }
    log_audit($conn, $_SESSION['user_id'], 'Export Audit Logs', 'Exported audit logs to Excel' . $filter_details);
    exit();
}

// Fetch available technicians
$tech_sql = "SELECT id, username FROM users WHERE role = 'technician' ORDER BY username";
$tech_result = $conn->query($tech_sql);
if ($tech_result === false) {
    die("Query failed: " . $conn->error);
}
$technicians = [];
while ($tech_row = $tech_result->fetch_assoc()) {
    $technicians[] = $tech_row;
}

// Fetch pending assignments
$pending_sql = "SELECT c.*, u.username AS tech_name, r.name AS region_name FROM clients c
                JOIN users u ON c.assigned_tech_id = u.id
                JOIN regions r ON c.region_id = r.id
                WHERE c.status = 'Assignment Pending'";
$pending_result = $conn->query($pending_sql);
if ($pending_result === false) {
    die("Query failed: " . $conn->error);
}

// Fetch details submitted
$submitted_sql = "SELECT c.*, u.username AS tech_name, r.name AS region_name, i.* FROM clients c
                  JOIN users u ON c.assigned_tech_id = u.id
                  JOIN regions r ON c.region_id = r.id
                  JOIN installations i ON c.id = i.client_id
                  WHERE c.status = 'Details Submitted'";
$submitted_result = $conn->query($submitted_sql);
if ($submitted_result === false) {
    die("Query failed: " . $conn->error);
}

// Fetch active clients
$all_sql = "SELECT c.*, r.name AS region_name, u.username AS tech_name, s.username AS sales_name FROM clients c
            LEFT JOIN regions r ON c.region_id = r.id
            LEFT JOIN users u ON c.assigned_tech_id = u.id
            LEFT JOIN users s ON c.created_by = s.id
            WHERE c.status NOT IN ('Completed', 'Cancelled')
            ORDER BY c.updated_at DESC LIMIT 100";
$all_result = $conn->query($all_sql);
if ($all_result === false) {
    die("Query failed: " . $conn->error);
}

// Fetch completed installations
$completed_sql = "SELECT c.id AS client_id, c.name, c.contact_email, c.contact_phone, c.address, c.details AS client_details, r.name AS region_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes, u.username AS sales_username, tu.username AS tech_username
                 FROM clients c
                 JOIN installations i ON c.id = i.client_id
                 JOIN regions r ON c.region_id = r.id
                 LEFT JOIN users u ON c.created_by = u.id
                 LEFT JOIN users tu ON c.assigned_tech_id = tu.id
                 WHERE c.status = 'Completed'
                 ORDER BY i.updated_at DESC LIMIT 100";
$completed_result = $conn->query($completed_sql);
if ($completed_result === false) {
    die("Query failed: " . $conn->error);
}

// Debug: Check completed clients
$debug_sql = "SELECT COUNT(*) AS count FROM clients WHERE status = 'Completed'";
$debug_result = $conn->query($debug_sql);
$debug_row = $debug_result->fetch_assoc();
$completed_count = $debug_row['count'];

// Fetch audit logs
$audit_sql = "SELECT a.*, u.username FROM audit_logs a JOIN users u ON a.user_id = u.id ORDER BY a.timestamp DESC LIMIT 100";
$audit_result = $conn->query($audit_sql);
if ($audit_result === false) {
    die("Query failed: " . $conn->error);
}
// Delete notifications older than 7 days
$delete_notif_sql = "DELETE FROM notifications WHERE created_at < NOW() - INTERVAL 7 DAY AND user_id = ?";
$delete_notif_stmt = $conn->prepare($delete_notif_sql);
if ($delete_notif_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$delete_notif_stmt->bind_param("i", $_SESSION['user_id']);
$delete_notif_stmt->execute();
$delete_notif_stmt->close();

// Fetch notifications
$notif_sql = "SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC LIMIT 50";
$notif_stmt = $conn->prepare($notif_sql);
if ($notif_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$notif_stmt->bind_param("i", $_SESSION['user_id']);
$notif_stmt->execute();
$notif_result = $notif_stmt->get_result();
$notif_stmt->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Tech Team Dashboard - Vehicle Tracking System</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        body { background-color: #f4f6f9; }
        .navbar-brand { font-weight: bold; }
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .nav-tabs {
            position: sticky;
            top: 56px;
            z-index: 1000;
            background-color: #f4f6f9;
            border-bottom: 1px solid #dee2e6;
        }
        .nav-tabs .nav-link {
            color: #333;
            font-weight: 500;
            border-radius: 0;
            padding: 10px 15px;
        }
        .nav-tabs .nav-link.active {
            background-color: #007bff;
            color: white;
            border-color: #dee2e6 #dee2e6 #f4f6f9;
        }
        .table { font-size: 0.9rem; }
        .btn-sm { font-size: 0.8rem; }
        .badge-pending { background-color: #ffc107; }
        .badge-inprogress { background-color: #17a2b8; }
        .badge-secondary { background-color: #6c757d; }
        .badge-completed { background-color: #28a745; }
        .details-section { background-color: #f8f9fa; padding: 15px; border-radius: 5px; margin-top: 10px; }
        .toast-container { position: fixed; top: 20px; right: 20px; z-index: 1055; }
        .filter-form { background-color: #f8f9fa; padding: 15px; border-radius: 8px; margin-bottom: 20px; }
        .filter-form label { font-weight: 500; }
        .filter-form .form-control { border-radius: 5px; }
        .filter-form .btn { border-radius: 5px; }
        .notification-link { cursor: pointer; color: #007bff; text-decoration: underline; }
        .notification-link:hover { color: #0056b3; }
        /* Active Clients Table */
        #allClientsTable {
            table-layout: fixed;
            width: 100%;
        }
        #allClientsTable th, #allClientsTable td {
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            padding: 8px;
        }
        #allClientsTable th:nth-child(1), #allClientsTable td:nth-child(1) { width: 5%; }
        #allClientsTable th:nth-child(2), #allClientsTable td:nth-child(2) { width: 10%; }
        #allClientsTable th:nth-child(3), #allClientsTable td:nth-child(3) { width: 10%; }
        #allClientsTable th:nth-child(4), #allClientsTable td:nth-child(4) { width: 8%; }
        #allClientsTable th:nth-child(5), #allClientsTable td:nth-child(5) {
            width: 12%;
            white-space: normal;
            word-wrap: break-word;
        }
        #allClientsTable th:nth-child(6), #allClientsTable td:nth-child(6) {
            width: 12%;
            white-space: normal;
            word-wrap: break-word;
        }
        #allClientsTable th:nth-child(7), #allClientsTable td:nth-child(7) { width: 8%; }
        #allClientsTable th:nth-child(8), #allClientsTable td:nth-child(8) { width: 8%; }
        #allClientsTable th:nth-child(9), #allClientsTable td:nth-child(9) { width: 8%; }
        #allClientsTable th:nth-child(10), #allClientsTable td:nth-child(10) { width: 8%; }
        #allClientsTable th:nth-child(11), #allClientsTable td:nth-child(11) { width: 5%; }
        #allClientsTable th:nth-child(12), #allClientsTable td:nth-child(12) { width: 10%; }
        #allClientsTable th:nth-child(13), #allClientsTable td:nth-child(13) { width: 10%; }
        #allClientsTable th:nth-child(14), #allClientsTable td:nth-child(14) { width: 6%; }
        /* Completed Installations Table */
        #completedTable {
            table-layout: fixed;
            width: 100%;
        }
        #completedTable th, #completedTable td {
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            padding: 8px;
        }
        #completedTable th:nth-child(1), #completedTable td:nth-child(1) { width: 5%; }
        #completedTable th:nth-child(2), #completedTable td:nth-child(2) { width: 10%; }
        #completedTable th:nth-child(3), #completedTable td:nth-child(3) { width: 8%; }
        #completedTable th:nth-child(4), #completedTable td:nth-child(4) { width: 8%; }
        #completedTable th:nth-child(5), #completedTable td:nth-child(5) { width: 8%; }
        #completedTable th:nth-child(6), #completedTable td:nth-child(6) { width: 8%; }
        #completedTable th:nth-child(7), #completedTable td:nth-child(7) {
            width: 15%;
            white-space: normal;
            word-wrap: break-word;
        }
        #completedTable th:nth-child(8), #completedTable td:nth-child(8) {
            width: 15%;
            white-space: normal;
            word-wrap: break-word;
        }
        #completedTable th:nth-child(9), #completedTable td:nth-child(9) {
            width: 15%;
            white-space: normal;
            word-wrap: break-word;
        }
        #completedTable th:nth-child(10), #completedTable td:nth-child(10) { width: 10%; }
        #completedTable th:nth-child(11), #completedTable td:nth-child(11) { width: 8%; }
        #completedTable th:nth-child(12), #completedTable td:nth-child(12) { width: 8%; }
        #completedTable th:nth-child(13), #completedTable td:nth-child(13) { width: 10%; }
        /* Audit Logs Table */
        #auditTable {
            table-layout: fixed;
            width: 100%;
        }
        #auditTable th, #auditTable td {
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            padding: 8px;
        }
        #auditTable th:nth-child(1), #auditTable td:nth-child(1) { width: 5%; }
        #auditTable th:nth-child(2), #auditTable td:nth-child(2) { width: 15%; }
        #auditTable th:nth-child(3), #auditTable td:nth-child(3) { width: 20%; }
        #auditTable th:nth-child(4), #auditTable td:nth-child(4) {
            width: 40%;
            white-space: normal;
            word-wrap: break-word;
        }
        #auditTable th:nth-child(5), #auditTable td:nth-child(5) { width: 20%; }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark sticky-top">
        <div class="container-fluid">
            <a class="navbar-brand" href="index.php">Vehicle Tracking System</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php"><i class="fas fa-home"></i> Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link btn btn-danger text-white" href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                    </li>
                </ul>
            </div>
        </div>
    </nav>
    <div class="container mt-5">
        <?php if (isset($_SESSION['toast'])) { ?>
            <div class="toast-container">
                <div class="toast align-items-center text-white bg-<?php echo htmlspecialchars($_SESSION['toast']['type']); ?> border-0" role="alert" aria-live="assertive" aria-atomic="true">
                    <div class="d-flex">
                        <div class="toast-body">
                            <?php echo htmlspecialchars($_SESSION['toast']['message']); ?>
                        </div>
                        <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                    </div>
                </div>
            </div>
            <?php unset($_SESSION['toast']); ?>
        <?php } ?>
        <h2 class="mb-4 text-center">Tech Team Dashboard</h2>
        <ul class="nav nav-tabs" id="techTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="notifications-tab" data-bs-toggle="tab" data-bs-target="#notifications" type="button" role="tab" aria-controls="notifications" aria-selected="true">Notifications</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="pending-tab" data-bs-toggle="tab" data-bs-target="#pending" type="button" role="tab" aria-controls="pending" aria-selected="false">Pending Approvals</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="submitted-tab" data-bs-toggle="tab" data-bs-target="#submitted" type="button" role="tab" aria-controls="submitted" aria-selected="false">Details Submitted</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="allclients-tab" data-bs-toggle="tab" data-bs-target="#allclients" type="button" role="tab" aria-controls="allclients" aria-selected="false">Active Clients</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="completed-tab" data-bs-toggle="tab" data-bs-target="#completed" type="button" role="tab" aria-controls="completed" aria-selected="false">Completed Installations</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="audit-tab" data-bs-toggle="tab" data-bs-target="#audit" type="button" role="tab" aria-controls="audit" aria-selected="false">Audit Logs</button>
            </li>
        </ul>
        <div class="tab-content mt-4">
            <div class="tab-pane fade show active" id="notifications" role="tabpanel" aria-labelledby="notifications-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Notifications</h5>
                        <ul class="list-group">
                            <?php while($notif = $notif_result->fetch_assoc()) { ?>
                                <li class="list-group-item <?php echo $notif['is_read'] ? '' : 'fw-bold bg-light'; ?>">
                                    <i class="fas fa-bell me-2"></i>
                                    <span class="notification-message <?php echo !$notif['is_read'] ? 'notification-link' : ''; ?>"
                                          data-client-id="<?php echo htmlspecialchars($notif['related_id'] ?? ''); ?>"
                                          data-tab="<?php echo htmlspecialchars($notif['related_tab'] ?? 'notifications'); ?>">
                                        <?php echo htmlspecialchars($notif['message']); ?>
                                    </span>
                                    <span class="text-muted">(<?php echo htmlspecialchars($notif['created_at']); ?>)</span>
                                    <?php if (!$notif['is_read']) { ?>
                                        <a href="?mark_read=<?php echo $notif['id']; ?>&tab=<?php echo urlencode($notif['related_tab'] ?? 'notifications'); ?>&client_id=<?php echo urlencode($notif['related_id'] ?? ''); ?>" class="float-end btn btn-link btn-sm">Mark as Read</a>
                                    <?php } ?>
                                </li>
                            <?php } ?>
                        </ul>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="pending" role="tabpanel" aria-labelledby="pending-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Pending Assignment Approvals</h5>
                        <?php if ($pending_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="pendingTable">
                                <thead><tr><th>ID</th><th>Name</th><th>Region</th><th>Tech</th><th>Actions</th></tr></thead>
                                <tbody>
                                    <?php while($row = $pending_result->fetch_assoc()) { ?>
                                        <tr <?php echo (isset($_GET['highlight']) && $_GET['highlight'] == $row['id']) ? 'class="table-warning"' : ''; ?>>
                                            <td><?php echo htmlspecialchars($row['id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['tech_name']); ?></td>
                                            <td>
                                                <a href="?approve=<?php echo $row['id']; ?>" class="btn btn-sm btn-success"><i class="fas fa-check"></i> Approve</a>
                                                <button type="button" class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#cancelModal<?php echo $row['id']; ?>">
                                                    <i class="fas fa-times"></i> Cancel
                                                </button>
                                            </td>
                                        </tr>
                                        <div class="modal fade" id="cancelModal<?php echo $row['id']; ?>" tabindex="-1" aria-labelledby="cancelModalLabel<?php echo $row['id']; ?>" aria-hidden="true">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title" id="cancelModalLabel<?php echo $row['id']; ?>">Cancel Client: <?php echo htmlspecialchars($row['name']); ?></h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <form method="post" action="">
                                                            <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                            <div class="form-group mb-3">
                                                                <label for="cancel_reason">Reason for Cancellation:</label>
                                                                <textarea name="cancel_reason" class="form-control" required placeholder="Enter the reason for cancelling this assignment"></textarea>
                                                            </div>
                                                            <button type="submit" name="cancel_client" class="btn btn-danger"><i class="fas fa-times"></i> Confirm Cancellation</button>
                                                        </form>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>
                                </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No pending approvals.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="submitted" role="tabpanel" aria-labelledby="submitted-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Details Submitted for Review</h5>
                        <?php if ($submitted_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="submittedTable">
                                <thead><tr><th>ID</th><th>Name</th><th>Region</th><th>Tech</th><th>Gadget Details</th><th>Vehicle Details</th><th>Installation Date</th><th>Notes</th><th>Actions</th></tr></thead>
                                <tbody>
                                    <?php while($row = $submitted_result->fetch_assoc()) { ?>
                                        <tr <?php echo (isset($_GET['highlight']) && $_GET['highlight'] == $row['id']) ? 'class="table-warning"' : ''; ?>>
                                            <td><?php echo htmlspecialchars($row['id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['tech_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['gadget_details'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['vehicle_details'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['installation_date'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['notes'] ?? 'N/A'); ?></td>
                                            <td>
                                                <a href="?complete=<?php echo $row['id']; ?>" class="btn btn-sm btn-success" onclick="return confirm('Mark this installation as completed?');"><i class="fas fa-check-circle"></i> Mark Completed</a>
                                                <button type="button" class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#cancelModal<?php echo $row['id']; ?>">
                                                    <i class="fas fa-times"></i> Cancel
                                                </button>
                                            </td>
                                        </tr>
                                        <div class="modal fade" id="cancelModal<?php echo $row['id']; ?>" tabindex="-1" aria-labelledby="cancelModalLabel<?php echo $row['id']; ?>" aria-hidden="true">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title" id="cancelModalLabel<?php echo $row['id']; ?>">Cancel Client: <?php echo htmlspecialchars($row['name']); ?></h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <form method="post" action="">
                                                            <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                            <div class="form-group mb-3">
                                                                <label for="cancel_reason">Reason for Cancellation:</label>
                                                                <textarea name="cancel_reason" class="form-control" required placeholder="Enter the reason for cancelling this assignment"></textarea>
                                                            </div>
                                                            <button type="submit" name="cancel_client" class="btn btn-danger"><i class="fas fa-times"></i> Confirm Cancellation</button>
                                                        </form>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>
                                </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No installations pending review.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="allclients" role="tabpanel" aria-labelledby="allclients-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Active Clients</h5>
                        <?php if ($all_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="allClientsTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Email</th>
                                        <th>Phone</th>
                                        <th>Address</th>
                                        <th>Details</th>
                                        <th>Region</th>
                                        <th>Sales</th>
                                        <th>Status</th>
                                        <th>Assigned Tech</th>
                                        <th>Approved</th>
                                        <th>Update Status</th>
                                        <th>Assign Technician</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($row = $all_result->fetch_assoc()) { ?>
                                        <tr <?php echo (isset($_GET['highlight']) && $_GET['highlight'] == $row['id']) ? 'class="table-warning"' : ''; ?>>
                                            <td><?php echo htmlspecialchars($row['id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['contact_email']); ?></td>
                                            <td><?php echo htmlspecialchars($row['contact_phone']); ?></td>
                                            <td><?php echo htmlspecialchars($row['address']); ?></td>
                                            <td><?php echo htmlspecialchars($row['details']); ?></td>
                                            <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['sales_name']); ?></td>
                                            <td>
                                                <span class="badge <?php
                                                    switch ($row['status']) {
                                                        case 'Assignment Pending':
                                                        case 'Details Submitted':
                                                            echo 'badge-pending';
                                                            break;
                                                        case 'In Progress':
                                                        case 'Assigned':
                                                            echo 'badge-inprogress';
                                                            break;
                                                        default:
                                                            echo 'badge-secondary';
                                                    }
                                                ?>">
                                                    <?php echo htmlspecialchars($row['status']); ?>
                                                </span>
                                            </td>
                                            <td><?php echo htmlspecialchars($row['tech_name'] ?: 'None'); ?></td>
                                            <td><?php echo $row['approved'] ? 'Yes' : 'No'; ?></td>
                                            <td>
                                                <form method="post" action="">
                                                    <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                    <select name="status" class="form-select form-select-sm">
                                                        <option value="New" <?php if($row['status']=='New') echo 'selected'; ?>>New</option>
                                                        <option value="Assignment Pending" <?php if($row['status']=='Assignment Pending') echo 'selected'; ?>>Assignment Pending</option>
                                                        <option value="Assigned" <?php if($row['status']=='Assigned') echo 'selected'; ?>>Assigned</option>
                                                        <option value="In Progress" <?php if($row['status']=='In Progress') echo 'selected'; ?>>In Progress</option>
                                                        <option value="Details Submitted" <?php if($row['status']=='Details Submitted') echo 'selected'; ?>>Details Submitted</option>
                                                        <option value="Completed" <?php if($row['status']=='Completed') echo 'selected'; ?>>Completed</option>
                                                        <option value="Cancelled" <?php if($row['status']=='Cancelled') echo 'selected'; ?>>Cancelled</option>
                                                    </select>
                                                    <button type="submit" name="update_status" class="btn btn-sm btn-primary mt-1"><i class="fas fa-sync"></i> Update</button>
                                                </form>
                                            </td>
                                            <td>
                                                <?php if ($row['status'] == 'New' && empty($row['tech_name'])) { ?>
                                                    <form method="post" action="">
                                                        <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                        <select name="technician_id" class="form-select form-select-sm">
                                                            <option value="">Select Technician</option>
                                                            <?php foreach ($technicians as $tech) { ?>
                                                                <option value="<?php echo $tech['id']; ?>"><?php echo htmlspecialchars($tech['username']); ?></option>
                                                            <?php } ?>
                                                        </select>
                                                        <button type="submit" name="assign_technician" class="btn btn-sm btn-primary mt-1"><i class="fas fa-user-plus"></i> Assign</button>
                                                    </form>
                                                <?php } else { ?>
                                                    <span class="text-muted">N/A</span>
                                                <?php } ?>
                                            </td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-danger" data-bs-toggle="modal" data-bs-target="#cancelModal<?php echo $row['id']; ?>">
                                                    <i class="fas fa-times"></i> Cancel
                                                </button>
                                            </td>
                                        </tr>
                                        <div class="modal fade" id="cancelModal<?php echo $row['id']; ?>" tabindex="-1" aria-labelledby="cancelModalLabel<?php echo $row['id']; ?>" aria-hidden="true">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title" id="cancelModalLabel<?php echo $row['id']; ?>">Cancel Client: <?php echo htmlspecialchars($row['name']); ?></h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <form method="post" action="">
                                                            <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                            <div class="form-group mb-3">
                                                                <label for="cancel_reason">Reason for Cancellation:</label>
                                                                <textarea name="cancel_reason" class="form-control" required placeholder="Enter the reason for cancelling this assignment"></textarea>
                                                            </div>
                                                            <button type="submit" name="cancel_client" class="btn btn-danger"><i class="fas fa-times"></i> Confirm Cancellation</button>
                                                        </form>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>
                                </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No active clients.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="completed" role="tabpanel" aria-labelledby="completed-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Completed Installations</h5>
                        <div class="filter-form">
                            <h6>Filter and Export Installations</h6>
                            <form method="get" class="row g-3">
                                <input type="hidden" name="export_installations" value="1">
                                <div class="col-md-4">
                                    <label for="from_date" class="form-label">From Date</label>
                                    <input type="date" name="from_date" id="from_date" class="form-control">
                                </div>
                                <div class="col-md-4">
                                    <label for="to_date" class="form-label">To Date</label>
                                    <input type="date" name="to_date" id="to_date" class="form-control">
                                </div>
                                <div class="col-md-4 d-flex align-items-end">
                                    <button type="submit" class="btn btn-info w-100"><i class="fas fa-file-export"></i> Export Filtered Installations (Excel)</button>
                                </div>
                            </form>
                            <a href="?export_installations=1" class="btn btn-info btn-sm mt-3"><i class="fas fa-file-export"></i> Export All Installations (Excel)</a>
                        </div>
                        <?php if ($completed_count == 0) { ?>
                            <div class="alert alert-warning">No clients with 'Completed' status found in the database.</div>
                        <?php } elseif ($completed_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="completedTable">
                                <thead>
                                    <tr>
                                        <th>Client ID</th>
                                        <th>Name</th>
                                        <th>Email</th>
                                        <th>PhoneNo</th>
                                        <th>Address</th>
                                        <th>Region</th>
                                        <th>Client</th>
                                        <th>Gadgets</th>
                                        <th>Vehicle Details</th>
                                        <th>Date</th>
                                        <th>SalesPerson</th>
                                        <th>Technician</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($row = $completed_result->fetch_assoc()) { ?>
                                        <tr <?php echo (isset($_GET['highlight']) && $_GET['highlight'] == $row['client_id']) ? 'class="table-warning"' : ''; ?>>
                                            <td><?php echo htmlspecialchars($row['client_id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['contact_email']); ?></td>
                                            <td><?php echo htmlspecialchars($row['contact_phone']); ?></td>
                                            <td><?php echo htmlspecialchars($row['address']); ?></td>
                                            <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['client_details'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['gadget_details'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['vehicle_details'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['installation_date'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['sales_username'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['tech_username'] ?? 'N/A'); ?></td>
                                            <td>
                                                <a href="view_installation.php?client_id=<?php echo $row['client_id']; ?>" class="btn btn-sm btn-primary"><i class="fas fa-eye"></i> View</a>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No completed installations found. Possible issue: Clients with 'Completed' status lack installation records.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="audit" role="tabpanel" aria-labelledby="audit-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Audit Logs (Last 100)</h5>
                        <div class="filter-form">
                            <h6>Filter and Export Audit Logs</h6>
                            <form method="get" class="row g-3">
                                <input type="hidden" name="export_audit" value="1">
                                <div class="col-md-4">
                                    <label for="audit_from_date" class="form-label">From Date</label>
                                    <input type="date" name="from_date" id="audit_from_date" class="form-control">
                                </div>
                                <div class="col-md-4">
                                    <label for="audit_to_date" class="form-label">To Date</label>
                                    <input type="date" name="to_date" id="audit_to_date" class="form-control">
                                </div>
                                <div class="col-md-4 d-flex align-items-end">
                                    <button type="submit" class="btn btn-info w-100"><i class="fas fa-file-export"></i> Export Filtered Audit Logs (Excel)</button>
                                </div>
                            </form>
                            <a href="?export_audit=1" class="btn btn-info btn-sm mt-3"><i class="fas fa-file-export"></i> Export All Audit Logs (Excel)</a>
                        </div>
                        <?php if ($audit_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="auditTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>User</th>
                                        <th>Action</th>
                                        <th>Details</th>
                                        <th>Timestamp</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($row = $audit_result->fetch_assoc()) { ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($row['id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['username']); ?></td>
                                            <td><?php echo htmlspecialchars($row['action']); ?></td>
                                            <td><?php echo htmlspecialchars($row['details']); ?></td>
                                            <td><?php echo htmlspecialchars($row['timestamp']); ?></td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No audit logs available.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#pendingTable, #submittedTable, #allClientsTable, #completedTable, #auditTable').DataTable({
                "paging": true,
                "searching": true,
                "ordering": true,
                "info": true,
                "responsive": true,
                "pageLength": 10
            });
            var toastElList = [].slice.call(document.querySelectorAll('.toast'));
            var toastList = toastElList.map(function(toastEl) {
                return new bootstrap.Toast(toastEl, {
                    autohide: true,
                    delay: 5000
                });
            });
            toastList.forEach(function(toast) {
                toast.show();
            });
            // Handle notification clicks
            $('.notification-link').on('click', function() {
                var tab = $(this).data('tab');
                var clientId = $(this).data('client-id');
                // Redirect to mark as read with tab and client_id
                if ($(this).closest('.list-group-item').hasClass('fw-bold')) {
                    var markReadUrl = $(this).closest('li').find('a').attr('href');
                    window.location.href = markReadUrl;
                    return;
                }
                // Switch to tab
                $('#techTabs .nav-link[data-bs-target="#' + tab + '"]').tab('show');
                // Highlight row if clientId is provided
                if (clientId && tab !== 'notifications') {
                    setTimeout(function() {
                        var tableId = '#' + tab + 'Table';
                        $(tableId + ' tbody tr').removeClass('table-warning');
                        $(tableId + ' tbody tr td:first').filter(function() {
                            return $(this).text() == clientId;
                        }).closest('tr').addClass('table-warning');
                        // Scroll to the highlighted row
                        var $row = $(tableId + ' tbody tr td:first').filter(function() {
                            return $(this).text() == clientId;
                        }).closest('tr');
                        if ($row.length) {
                            var table = $(tableId).DataTable();
                            var rowIndex = table.row($row).index();
                            table.page(Math.floor(rowIndex / table.page.len())).draw('page');
                            $('html, body').animate({
                                scrollTop: $row.offset().top - 100
                            }, 500);
                        }
                    }, 100);
                }
            });
            // Activate the correct tab on page load
            <?php if (isset($_GET['tab'])) { ?>
                $('#techTabs .nav-link[data-bs-target="#<?php echo htmlspecialchars($_GET['tab']); ?>"]').tab('show');
            <?php } ?>
        });
    </script>
</body>
</html>