<?php
// technician_dashboard.php - Technician Role Dashboard with Updates
include 'auth.php';

// Verify role
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'technician') {
    echo "Access Denied.";
    exit();
}

// Verify session variables
if (!isset($_SESSION['user_id']) || !is_numeric($_SESSION['user_id']) || !isset($_SESSION['region_id']) || !is_numeric($_SESSION['region_id'])) {
    die("Error: Invalid or missing user ID or region ID in session.");
}

$user_id = $_SESSION['user_id'];
$region_id = $_SESSION['region_id'];

// Fetch technician's username
$tech_sql = "SELECT username FROM users WHERE id = ?";
$tech_stmt = $conn->prepare($tech_sql);
if ($tech_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$tech_stmt->bind_param("i", $user_id);
$tech_stmt->execute();
$tech_result = $tech_stmt->get_result();
$tech_row = $tech_result->fetch_assoc();
$tech_name = $tech_row['username'] ?? 'Unknown Technician';
$tech_stmt->close();

// Handle Mark Notification Read
if (isset($_GET['mark_read'])) {
    $notif_id = $_GET['mark_read'];
    $stmt = $conn->prepare("UPDATE notifications SET is_read = 1 WHERE id = ? AND user_id = ?");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("ii", $notif_id, $user_id);
    $stmt->execute();
    $stmt->close();
    header("Location: technician_dashboard.php");
    exit();
}

// Check if technician has pending tasks (exclude Details Submitted)
$has_pending_sql = "SELECT COUNT(*) as count FROM clients WHERE assigned_tech_id = ? AND status IN ('Assignment Pending', 'Assigned', 'In Progress')";
$has_pending_stmt = $conn->prepare($has_pending_sql);
if ($has_pending_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$has_pending_stmt->bind_param("i", $user_id);
$has_pending_stmt->execute();
$has_pending_result = $has_pending_stmt->get_result();
$has_pending_row = $has_pending_result->fetch_assoc();
$has_pending = $has_pending_row['count'] > 0;
$has_pending_stmt->close();

// Handle self-assign (only if no pending tasks)
if (isset($_GET['assign']) && !$has_pending) {
    $client_id = $_GET['assign'];

    // Fetch client name
    $client_sql = "SELECT name FROM clients WHERE id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $client_stmt->close();

    $stmt = $conn->prepare("UPDATE clients SET assigned_tech_id = ?, status = 'Assignment Pending', approved = 0 WHERE id = ? AND region_id = ? AND status = 'New'");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("iii", $user_id, $client_id, $region_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Request Assignment', "Client: $client_name");
        $message = "Assignment requested for client $client_name by technician $tech_name.";

        // Notify technician
        send_notification($conn, $user_id, $message);

        // Notify tech team for approval
        $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
        $tech_result = $conn->query($tech_sql);
        if ($tech_result === false) {
            die("Query failed: " . $conn->error);
        }
        while ($tech_row = $tech_result->fetch_assoc()) {
            send_notification($conn, $tech_row['id'], $message);
        }

        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Assignment requested successfully. Waiting for tech team approval.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: technician_dashboard.php");
    exit();
} elseif (isset($_GET['assign']) && $has_pending) {
    $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Cannot assign new client: You have pending tasks.'];
    header("Location: technician_dashboard.php");
    exit();
}

// Handle submit details (insert new details and update status to Details Submitted)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['submit_details'])) {
    $client_id = $_POST['client_id'];
    $gadget_details = $_POST['gadget_details'];
    $vehicle_details = $_POST['vehicle_details'];
    $installation_date = $_POST['installation_date'];
    $notes = $_POST['notes'];

    // Fetch client name
    $client_sql = "SELECT name FROM clients WHERE id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $client_stmt->close();

    // Check for existing installation record
    $check_sql = "SELECT COUNT(*) AS count FROM installations WHERE client_id = ?";
    $check_stmt = $conn->prepare($check_sql);
    $check_stmt->bind_param("i", $client_id);
    $check_stmt->execute();
    $check_result = $check_stmt->get_result();
    $check_row = $check_result->fetch_assoc();
    if ($check_row['count'] > 0) {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: Installation details already submitted for this client.'];
        $check_stmt->close();
        header("Location: technician_dashboard.php");
        exit();
    }
    $check_stmt->close();

    $stmt = $conn->prepare("INSERT INTO installations (client_id, gadget_details, vehicle_details, installation_date, notes, updated_at) VALUES (?, ?, ?, ?, ?, NOW())");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("issss", $client_id, $gadget_details, $vehicle_details, $installation_date, $notes);
    if ($stmt->execute()) {
        $update_stmt = $conn->prepare("UPDATE clients SET status = 'Details Submitted' WHERE id = ? AND assigned_tech_id = ? AND status IN ('Assigned', 'In Progress')");
        if ($update_stmt === false) {
            die("Prepare failed: " . $conn->error);
        }
        $update_stmt->bind_param("ii", $client_id, $user_id);
        if ($update_stmt->execute()) {
            log_audit($conn, $user_id, 'Submit Details', "Client: $client_name");
            $message = "Installation details submitted for client $client_name by technician $tech_name.";
            send_notification($conn, $user_id, $message);

            // Notify tech team
            $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
            $tech_result = $conn->query($tech_sql);
            if ($tech_result === false) {
                die("Query failed: " . $conn->error);
            }
            while ($tech_row = $tech_result->fetch_assoc()) {
                send_notification($conn, $tech_row['id'], $message);
            }

            $_SESSION['toast'] = ['type' => 'success', 'message' => 'Details submitted, waiting for tech review.'];
        } else {
            $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
        }
        $update_stmt->close();
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: technician_dashboard.php");
    exit();
}

// Handle update details (update existing details, no status change)
if ($_SERVER["REQUEST_METHOD"] == "POST" && isset($_POST['update_details'])) {
    $client_id = $_POST['client_id'];
    $gadget_details = $_POST['gadget_details'];
    $vehicle_details = $_POST['vehicle_details'];
    $installation_date = $_POST['installation_date'];
    $notes = $_POST['notes'];

    // Fetch client name
    $client_sql = "SELECT name FROM clients WHERE id = ?";
    $client_stmt = $conn->prepare($client_sql);
    if ($client_stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $client_stmt->bind_param("i", $client_id);
    $client_stmt->execute();
    $client_result = $client_stmt->get_result();
    $client_row = $client_result->fetch_assoc();
    $client_name = $client_row['name'] ?? 'Unknown Client';
    $client_stmt->close();

    $stmt = $conn->prepare("UPDATE installations SET gadget_details = ?, vehicle_details = ?, installation_date = ?, notes = ?, updated_at = NOW() WHERE client_id = ?");
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("ssssi", $gadget_details, $vehicle_details, $installation_date, $notes, $client_id);
    if ($stmt->execute()) {
        log_audit($conn, $user_id, 'Update Details', "Client: $client_name");
        $message = "Installation details updated for client $client_name by technician $tech_name.";
        send_notification($conn, $user_id, $message);

        // Notify tech team
        $tech_sql = "SELECT id FROM users WHERE role = 'tech'";
        $tech_result = $conn->query($tech_sql);
        if ($tech_result === false) {
            die("Query failed: " . $conn->error);
        }
        while ($tech_row = $tech_result->fetch_assoc()) {
            send_notification($conn, $tech_row['id'], $message);
        }

        $_SESSION['toast'] = ['type' => 'success', 'message' => 'Details updated successfully.'];
    } else {
        $_SESSION['toast'] = ['type' => 'danger', 'message' => 'Error: ' . $conn->error];
    }
    $stmt->close();
    header("Location: technician_dashboard.php");
    exit();
}

// Fetch available clients (New in region) only if no pending tasks
$new_result = null;
if (!$has_pending) {
    $new_sql = "SELECT c.*, r.name AS region_name FROM clients c JOIN regions r ON c.region_id = r.id WHERE c.region_id = ? AND c.status = 'New' LIMIT 50";
    $stmt = $conn->prepare($new_sql);
    if ($stmt === false) {
        die("Prepare failed: " . $conn->error);
    }
    $stmt->bind_param("i", $region_id);
    $stmt->execute();
    $new_result = $stmt->get_result();
    $stmt->close();
}

// Fetch pending assignments by this technician
$pending_sql = "SELECT c.*, r.name AS region_name FROM clients c JOIN regions r ON c.region_id = r.id WHERE c.assigned_tech_id = ? AND c.status = 'Assignment Pending'";
$pending_stmt = $conn->prepare($pending_sql);
if ($pending_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$pending_stmt->bind_param("i", $user_id);
$pending_stmt->execute();
$pending_result = $pending_stmt->get_result();
$pending_stmt->close();

// Fetch assigned (approved) clients with installation details
$assigned_sql = "SELECT c.*, r.name AS region_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes 
                 FROM clients c 
                 JOIN regions r ON c.region_id = r.id 
                 LEFT JOIN installations i ON c.id = i.client_id 
                 WHERE c.assigned_tech_id = ? AND c.approved = 1 AND c.status IN ('Assigned', 'In Progress', 'Details Submitted')";
$assigned_stmt = $conn->prepare($assigned_sql);
if ($assigned_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$assigned_stmt->bind_param("i", $user_id);
$assigned_stmt->execute();
$assigned_result = $assigned_stmt->get_result();
$assigned_stmt->close();

// Fetch completed installations by this technician
$completed_sql = "SELECT c.id, c.name, c.contact_email, c.contact_phone, c.address, c.details, r.name AS region_name, i.gadget_details, i.vehicle_details, i.installation_date, i.notes, u.username AS sales_username, tu.username AS tech_username 
                 FROM clients c 
                 JOIN installations i ON c.id = i.client_id 
                 JOIN regions r ON c.region_id = r.id
                 LEFT JOIN users u ON c.created_by = u.id
                 LEFT JOIN users tu ON c.assigned_tech_id = tu.id
                 WHERE c.status = 'Completed' AND c.assigned_tech_id = ? 
                 ORDER BY i.updated_at DESC LIMIT 100";
$completed_stmt = $conn->prepare($completed_sql);
if ($completed_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$completed_stmt->bind_param("i", $user_id);
$completed_stmt->execute();
$completed_result = $completed_stmt->get_result();
$completed_stmt->close();
// Delete notifications older than 7 days
$delete_notif_sql = "DELETE FROM notifications WHERE created_at < NOW() - INTERVAL 7 DAY AND user_id = ?";
$delete_notif_stmt = $conn->prepare($delete_notif_sql);
if ($delete_notif_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$delete_notif_stmt->bind_param("i", $_SESSION['user_id']);
$delete_notif_stmt->execute();
$delete_notif_stmt->close();

// Fetch notifications
$notif_sql = "SELECT * FROM notifications WHERE user_id = ? ORDER BY created_at DESC LIMIT 50";
$notif_stmt = $conn->prepare($notif_sql);
if ($notif_stmt === false) {
    die("Prepare failed: " . $conn->error);
}
$notif_stmt->bind_param("i", $user_id);
$notif_stmt->execute();
$notif_result = $notif_stmt->get_result();
$notif_stmt->close();
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Technician Dashboard - Vehicle Tracking System</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        body { background-color: #f4f6f9; }
        .navbar-brand { font-weight: bold; }
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .nav-tabs .nav-link { color: #333; font-weight: 500; }
        .nav-tabs .nav-link.active { background-color: #007bff; color: white; }
        .table { font-size: 0.9rem; }
        .btn-sm { font-size: 0.8rem; }
        .badge-completed { background-color: #28a745; }
        .badge-pending { background-color: #ffc107; }
        .badge-inprogress { background-color: #17a2b8; }
        .badge-secondary { background-color: #6c757d; }
        .form-control, .form-select { border-radius: 5px; }
        .details-section { background-color: #f8f9fa; padding: 15px; border-radius: 5px; margin-top: 10px; }
        .toast-container { position: fixed; top: 20px; right: 20px; z-index: 1055; }
        .notification-link { cursor: pointer; color: #007bff; text-decoration: underline; }
        .notification-link:hover { color: #0056b3; }
    </style>
</head>
<body>
    <nav class="navbar navbar-expand-lg navbar-dark bg-dark sticky-top">
        <div class="container-fluid">
            <a class="navbar-brand" href="index.php">Vehicle Tracking System</a>
            <button class="navbar-toggler" type="button" data-bs-toggle="collapse" data-bs-target="#navbarNav" aria-controls="navbarNav" aria-expanded="false" aria-label="Toggle navigation">
                <span class="navbar-toggler-icon"></span>
            </button>
            <div class="collapse navbar-collapse" id="navbarNav">
                <ul class="navbar-nav ms-auto">
                    <li class="nav-item">
                        <a class="nav-link" href="index.php"><i class="fas fa-home"></i> Home</a>
                    </li>
                    <li class="nav-item">
                        <a class="nav-link btn btn-danger text-white" href="logout.php"><i class="fas fa-sign-out-alt"></i> Logout</a>
                    </li>
                </ul>
            </div>
        </div>
    </nav>
    <div class="container mt-5">
        <!-- Toast Notification Container -->
        <?php if (isset($_SESSION['toast'])) { ?>
            <div class="toast-container">
                <div class="toast align-items-center text-white bg-<?php echo htmlspecialchars($_SESSION['toast']['type']); ?> border-0" role="alert" aria-live="assertive" aria-atomic="true">
                    <div class="d-flex">
                        <div class="toast-body">
                            <?php echo htmlspecialchars($_SESSION['toast']['message']); ?>
                        </div>
                        <button type="button" class="btn-close btn-close-white me-2 m-auto" data-bs-dismiss="toast" aria-label="Close"></button>
                    </div>
                </div>
            </div>
            <?php unset($_SESSION['toast']); ?>
        <?php } ?>

        <h2 class="mb-4 text-center">Technician Dashboard</h2>

        <ul class="nav nav-tabs" id="technicianTabs" role="tablist">
            <li class="nav-item" role="presentation">
                <button class="nav-link active" id="notifications-tab" data-bs-toggle="tab" data-bs-target="#notifications" type="button" role="tab" aria-controls="notifications" aria-selected="true">Notifications</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="available-tab" data-bs-toggle="tab" data-bs-target="#available" type="button" role="tab" aria-controls="available" aria-selected="false">Available Clients</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="pending-tab" data-bs-toggle="tab" data-bs-target="#pending" type="button" role="tab" aria-controls="pending" aria-selected="false">Pending Assignments</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="assigned-tab" data-bs-toggle="tab" data-bs-target="#assigned" type="button" role="tab" aria-controls="assigned" aria-selected="false">Assigned Clients</button>
            </li>
            <li class="nav-item" role="presentation">
                <button class="nav-link" id="completed-tab" data-bs-toggle="tab" data-bs-target="#completed" type="button" role="tab" aria-controls="completed" aria-selected="false">Completed Installations</button>
            </li>
        </ul>

        <div class="tab-content mt-4">
            <div class="tab-pane fade show active" id="notifications" role="tabpanel" aria-labelledby="notifications-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Notifications</h5>
                        <ul class="list-group">
                            <?php while($notif = $notif_result->fetch_assoc()) { ?>
                                <li class="list-group-item <?php echo $notif['is_read'] ? '' : 'fw-bold bg-light'; ?>">
                                    <i class="fas fa-bell me-2"></i> 
                                    <span class="notification-message <?php echo !$notif['is_read'] ? 'notification-link' : ''; ?>" 
                                          data-client-id="<?php echo $notif['related_id'] ?? ''; ?>" 
                                          data-tab="<?php echo $notif['related_tab'] ?? 'notifications'; ?>">
                                        <?php echo htmlspecialchars($notif['message']); ?>
                                    </span> 
                                    <span class="text-muted">(<?php echo htmlspecialchars($notif['created_at']); ?>)</span>
                                    <?php if (!$notif['is_read']) { ?>
                                        <a href="?mark_read=<?php echo $notif['id']; ?>" class="float-end btn btn-link btn-sm">Mark as Read</a>
                                    <?php } ?>
                                </li>
                            <?php } ?>
                        </ul>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="available" role="tabpanel" aria-labelledby="available-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Available Clients (New)</h5>
                        <?php if ($has_pending) { ?>
                            <div class="alert alert-warning"><i class="fas fa-exclamation-circle"></i> You must complete pending tasks (Assignment Pending, Assigned, or In Progress) before assigning new clients.</div>
                        <?php } else { ?>
                            <?php if ($new_result && $new_result->num_rows > 0) { ?>
                                <table class="table table-striped table-hover" id="availableTable">
                                    <thead>
                                        <tr>
                                            <th>ID</th>
                                            <th>Name</th>
                                            <th>Details</th>
                                            <th>Region</th>
                                            <th>Actions</th>
                                        </tr>
                                    </thead>
                                    <tbody>
                                        <?php while($row = $new_result->fetch_assoc()) { ?>
                                            <tr>
                                                <td><?php echo htmlspecialchars($row['id']); ?></td>
                                                <td><?php echo htmlspecialchars($row['name']); ?></td>
                                                <td><?php echo htmlspecialchars($row['details']); ?></td>
                                                <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                                <td>
                                                    <a href="?assign=<?php echo $row['id']; ?>" class="btn btn-sm btn-primary"><i class="fas fa-plus"></i> Request Assignment</a>
                                                </td>
                                            </tr>
                                        <?php } ?>
                                    </tbody>
                                </table>
                            <?php } else { ?>
                                <div class="alert alert-info">No new clients available in your region.</div>
                            <?php } ?>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="pending" role="tabpanel" aria-labelledby="pending-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Pending Assignments</h5>
                        <?php if ($pending_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="pendingTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Details</th>
                                        <th>Region</th>
                                        <th>Status</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($row = $pending_result->fetch_assoc()) { ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($row['id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['details']); ?></td>
                                            <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                            <td><span class="badge badge-secondary">Waiting for Approval</span></td>
                                        </tr>
                                    <?php } ?>
                                    </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No pending assignments.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="assigned" role="tabpanel" aria-labelledby="assigned-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Assigned Clients</h5>
                        <?php if ($assigned_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="assignedTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Details</th>
                                        <th>Region</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($row = $assigned_result->fetch_assoc()) { ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($row['id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['details']); ?></td>
                                            <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                            <td><span class="badge <?php 
                                                switch ($row['status']) {
                                                    case 'Details Submitted':
                                                        echo 'badge-pending';
                                                        break;
                                                    case 'In Progress':
                                                    case 'Assigned':
                                                        echo 'badge-inprogress';
                                                        break;
                                                    default:
                                                        echo 'badge-secondary';
                                                }
                                            ?>">
                                                <?php echo htmlspecialchars($row['status']); ?>
                                            </span></td>
                                            <td>
                                                <?php if ($row['status'] == 'Assigned' || $row['status'] == 'In Progress') { ?>
                                                    <form method="post" action="" class="mt-2">
                                                        <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                        <div class="form-group mb-3">
                                                            <label>Gadget Details:</label>
                                                            <textarea name="gadget_details" class="form-control" required></textarea>
                                                        </div>
                                                        <div class="form-group mb-3">
                                                            <label>Vehicle Details:</label>
                                                            <textarea name="vehicle_details" class="form-control" required></textarea>
                                                        </div>
                                                        <div class="form-group mb-3">
                                                            <label>Installation Date:</label>
                                                            <input type="date" name="installation_date" class="form-control" required>
                                                        </div>
                                                        <div class="form-group mb-3">
                                                            <label>Notes:</label>
                                                            <textarea name="notes" class="form-control"></textarea>
                                                        </div>
                                                        <button type="submit" name="submit_details" class="btn btn-sm btn-primary"><i class="fas fa-paper-plane"></i> Submit Details</button>
                                                    </form>
                                                <?php } elseif ($row['status'] == 'Details Submitted') { ?>
                                                    <form method="post" action="" class="mt-2">
                                                        <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                        <div class="form-group mb-3">
                                                            <label>Gadget Details:</label>
                                                            <textarea name="gadget_details" class="form-control" required><?php echo htmlspecialchars($row['gadget_details'] ?? ''); ?></textarea>
                                                        </div>
                                                        <div class="form-group mb-3">
                                                            <label>Vehicle Details:</label>
                                                            <textarea name="vehicle_details" class="form-control" required><?php echo htmlspecialchars($row['vehicle_details'] ?? ''); ?></textarea>
                                                        </div>
                                                        <div class="form-group mb-3">
                                                            <label>Installation Date:</label>
                                                            <input type="date" name="installation_date" class="form-control" required value="<?php echo htmlspecialchars($row['installation_date'] ?? ''); ?>">
                                                        </div>
                                                        <div class="form-group mb-3">
                                                            <label>Notes:</label>
                                                            <textarea name="notes" class="form-control"><?php echo htmlspecialchars($row['notes'] ?? ''); ?></textarea>
                                                        </div>
                                                        <button type="submit" name="update_details" class="btn btn-sm btn-warning"><i class="fas fa-edit"></i> Update Details</button>
                                                    </form>
                                                <?php } ?>
                                            </td>
                                        </tr>
                                    <?php } ?>
                                </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No assigned clients.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
            <div class="tab-pane fade" id="completed" role="tabpanel" aria-labelledby="completed-tab">
                <div class="card">
                    <div class="card-body">
                        <h5 class="card-title">Completed Installations</h5>
                        <?php if ($completed_result->num_rows > 0) { ?>
                            <table class="table table-striped table-hover" id="completedTable">
                                <thead>
                                    <tr>
                                        <th>ID</th>
                                        <th>Name</th>
                                        <th>Contact Email</th>
                                        <th>Contact Phone</th>
                                        <th>Address</th>
                                        <th>Details</th>
                                        <th>Region</th>
                                        <th>Sales Person</th>
                                        <th>Technician</th>
                                        <th>Status</th>
                                        <th>Actions</th>
                                    </tr>
                                </thead>
                                <tbody>
                                    <?php while($row = $completed_result->fetch_assoc()) { ?>
                                        <tr>
                                            <td><?php echo htmlspecialchars($row['id']); ?></td>
                                            <td><?php echo htmlspecialchars($row['name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['contact_email']); ?></td>
                                            <td><?php echo htmlspecialchars($row['contact_phone']); ?></td>
                                            <td><?php echo htmlspecialchars($row['address']); ?></td>
                                            <td><?php echo htmlspecialchars($row['details']); ?></td>
                                            <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                            <td><?php echo htmlspecialchars($row['sales_username'] ?? 'N/A'); ?></td>
                                            <td><?php echo htmlspecialchars($row['tech_username'] ?? 'N/A'); ?></td>
                                            <td><span class="badge badge-completed">Completed</span></td>
                                            <td>
                                                <button type="button" class="btn btn-sm btn-info" data-bs-toggle="modal" data-bs-target="#viewModal<?php echo $row['id']; ?>">
                                                    <i class="fas fa-eye"></i> View
                                                </button>
                                                <button type="button" class="btn btn-sm btn-warning" data-bs-toggle="modal" data-bs-target="#editModal<?php echo $row['id']; ?>">
                                                    <i class="fas fa-edit"></i> Edit
                                                </button>
                                            </td>
                                        </tr>
                                        <!-- View Modal -->
                                        <div class="modal fade" id="viewModal<?php echo $row['id']; ?>" tabindex="-1" aria-labelledby="viewModalLabel<?php echo $row['id']; ?>" aria-hidden="true">
                                            <div class="modal-dialog modal-lg">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title" id="viewModalLabel<?php echo $row['id']; ?>">View Installation Details for <?php echo htmlspecialchars($row['name']); ?></h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <div class="row">
                                                            <div class="col-md-6">
                                                                <strong>Sales Person:</strong> <?php echo htmlspecialchars($row['sales_username'] ?? 'N/A'); ?><br>
                                                                <strong>Technician:</strong> <?php echo htmlspecialchars($row['tech_username'] ?? 'N/A'); ?><br>
                                                                <strong>Email:</strong> <?php echo htmlspecialchars($row['contact_email']); ?><br>
                                                                <strong>Phone:</strong> <?php echo htmlspecialchars($row['contact_phone']); ?><br>
                                                                <strong>Address:</strong> <?php echo htmlspecialchars($row['address']); ?><br>
                                                                <strong>Region:</strong> <?php echo htmlspecialchars($row['region_name']); ?>
                                                            </div>
                                                            <div class="col-md-6">
                                                                <strong>Client Details:</strong><br><?php echo nl2br(htmlspecialchars($row['details'])); ?>
                                                            </div>
                                                        </div>
                                                        <hr>
                                                        <div class="row">
                                                            <div class="col-md-6">
                                                                <strong>Gadget Details:</strong><br><?php echo nl2br(htmlspecialchars($row['gadget_details'] ?? '')); ?>
                                                            </div>
                                                            <div class="col-md-6">
                                                                <strong>Vehicle Details:</strong><br><?php echo nl2br(htmlspecialchars($row['vehicle_details'] ?? '')); ?>
                                                            </div>
                                                        </div>
                                                        <hr>
                                                        <div class="row">
                                                            <div class="col-md-6">
                                                                <strong>Installation Date:</strong> <?php echo htmlspecialchars($row['installation_date'] ?? 'N/A'); ?>
                                                            </div>
                                                            <div class="col-md-6">
                                                                <strong>Notes:</strong><br><?php echo nl2br(htmlspecialchars($row['notes'] ?? '')); ?>
                                                            </div>
                                                        </div>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                        <!-- Modal for Editing Installation Details -->
                                        <div class="modal fade" id="editModal<?php echo $row['id']; ?>" tabindex="-1" aria-labelledby="editModalLabel<?php echo $row['id']; ?>" aria-hidden="true">
                                            <div class="modal-dialog">
                                                <div class="modal-content">
                                                    <div class="modal-header">
                                                        <h5 class="modal-title" id="editModalLabel<?php echo $row['id']; ?>">Edit Installation Details for <?php echo htmlspecialchars($row['name']); ?></h5>
                                                        <button type="button" class="btn-close" data-bs-dismiss="modal" aria-label="Close"></button>
                                                    </div>
                                                    <div class="modal-body">
                                                        <form method="post" action="">
                                                            <input type="hidden" name="client_id" value="<?php echo $row['id']; ?>">
                                                            <div class="form-group mb-3">
                                                                <label>Gadget Details:</label>
                                                                <textarea name="gadget_details" class="form-control" required><?php echo htmlspecialchars($row['gadget_details'] ?? ''); ?></textarea>
                                                            </div>
                                                            <div class="form-group mb-3">
                                                                <label>Vehicle Details:</label>
                                                                <textarea name="vehicle_details" class="form-control" required><?php echo htmlspecialchars($row['vehicle_details'] ?? ''); ?></textarea>
                                                            </div>
                                                            <div class="form-group mb-3">
                                                                <label>Installation Date:</label>
                                                                <input type="date" name="installation_date" class="form-control" required value="<?php echo htmlspecialchars($row['installation_date'] ?? ''); ?>">
                                                            </div>
                                                            <div class="form-group mb-3">
                                                                <label>Notes:</label>
                                                                <textarea name="notes" class="form-control"><?php echo htmlspecialchars($row['notes'] ?? ''); ?></textarea>
                                                            </div>
                                                            <button type="submit" name="update_details" class="btn btn-primary"><i class="fas fa-save"></i> Save Changes</button>
                                                        </form>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    <?php } ?>
                                </tbody>
                            </table>
                        <?php } else { ?>
                            <div class="alert alert-info">No completed installations found. Possible issue: No installation records for completed clients.</div>
                        <?php } ?>
                    </div>
                </div>
            </div>
        </div>
    </div>
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#availableTable, #pendingTable, #assignedTable, #completedTable').DataTable({
                "paging": true,
                "searching": true,
                "ordering": true,
                "info": true,
                "responsive": true,
                "pageLength": 10
            });

            // Initialize toasts
            var toastElList = [].slice.call(document.querySelectorAll('.toast'));
            var toastList = toastElList.map(function(toastEl) {
                return new bootstrap.Toast(toastEl, {
                    autohide: true,
                    delay: 5000 // Toast disappears after 5 seconds
                });
            });
            toastList.forEach(function(toast) {
                toast.show();
            });

            // Make notifications clickable
            $('.notification-link').on('click', function() {
                var tab = $(this).data('tab');
                var clientId = $(this).data('client-id');
                // Mark as read if not already
                if ($(this).closest('.list-group-item').hasClass('fw-bold')) {
                    window.location.href = '?mark_read=' + $(this).closest('li').find('a').attr('href').split('=')[1];
                    return;
                }
                // Switch to tab
                $('#technicianTabs .nav-link[data-bs-target="#' + tab + '"]').tab('show');
                // If client ID provided, highlight the row in the table
                if (clientId && tab !== 'notifications') {
                    setTimeout(function() {
                        var tableId = '#' + tab + 'Table';
                        $(tableId + ' tbody tr').removeClass('table-warning');
                        $(tableId + ' tbody tr td:first').filter(function() {
                            return $(this).text() == clientId;
                        }).closest('tr').addClass('table-warning');
                        $(tableId).DataTable().row($(tableId + ' tbody tr td:first').filter(function() {
                            return $(this).text() == clientId;
                        }).closest('tr')).scrollTo();
                    }, 100);
                }
            });
        });
    </script>
</body>
</html>