<?php
// view_all_clients.php - Display all active clients for superusers
include 'auth.php';

// Verify superuser role
if (!isset($_SESSION['role']) || $_SESSION['role'] != 'superuser') {
    echo "Access Denied.";
    exit();
}

// Verify session user_id
if (!isset($_SESSION['user_id']) || !is_numeric($_SESSION['user_id'])) {
    die("Error: Invalid or missing user ID in session.");
}

$user_id = $_SESSION['user_id'];

// Define tableExists function to check if a table exists
function tableExists($conn, $table_name) {
    $stmt = $conn->prepare("SELECT 1 FROM information_schema.tables WHERE table_schema = DATABASE() AND table_name = ?");
    if ($stmt === false) {
        return false;
    }
    $stmt->bind_param("s", $table_name);
    $stmt->execute();
    $result = $stmt->get_result();
    $exists = $result->num_rows > 0;
    $stmt->close();
    return $exists;
}

// Check if clients table exists
if (!tableExists($conn, 'clients')) {
    die("Error: Clients table does not exist in the database.");
}

// Fetch all active clients
$allclients_sql = "SELECT c.*, r.name AS region_name, u.username AS tech_name, s.username AS sales_name 
                   FROM clients c
                   LEFT JOIN regions r ON c.region_id = r.id
                   LEFT JOIN users u ON c.assigned_tech_id = u.id
                   LEFT JOIN users s ON c.created_by = s.id
                   WHERE c.status NOT IN ('Completed', 'Cancelled')
                   ORDER BY c.updated_at DESC LIMIT 100";
$allclients_result = $conn->query($allclients_sql);
if ($allclients_result === false) {
    die("Query failed: " . $conn->error);
}
?>

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>All Clients - Vehicle Tracking System</title>
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.6/css/dataTables.bootstrap5.min.css">
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.5.1/css/all.min.css">
    <style>
        body { background-color: #f4f6f9; }
        .card { border-radius: 10px; box-shadow: 0 4px 12px rgba(0,0,0,0.1); }
        .table { font-size: 0.9rem; }
        .btn-sm { font-size: 0.8rem; }
        .badge-pending { background-color: #ffc107; }
        .badge-inprogress { background-color: #17a2b8; }
        .badge-completed { background-color: #28a745; }
        .badge-secondary { background-color: #6c757d; }
        #allClientsTable {
            table-layout: fixed;
            width: 100%;
        }
        #allClientsTable th, #allClientsTable td {
            white-space: nowrap;
            overflow: hidden;
            text-overflow: ellipsis;
            padding: 8px;
        }
        #allClientsTable th:nth-child(1), #allClientsTable td:nth-child(1) { width: 5%; }
        #allClientsTable th:nth-child(2), #allClientsTable td:nth-child(2) { width: 10%; }
        #allClientsTable th:nth-child(3), #allClientsTable td:nth-child(3) { width: 10%; }
        #allClientsTable th:nth-child(4), #allClientsTable td:nth-child(4) { width: 8%; }
        #allClientsTable th:nth-child(5), #allClientsTable td:nth-child(5) { width: 12%; white-space: normal; }
        #allClientsTable th:nth-child(6), #allClientsTable td:nth-child(6) { width: 12%; white-space: normal; }
        #allClientsTable th:nth-child(7), #allClientsTable td:nth-child(7) { width: 8%; }
        #allClientsTable th:nth-child(8), #allClientsTable td:nth-child(8) { width: 8%; }
        #allClientsTable th:nth-child(9), #allClientsTable td:nth-child(9) { width: 8%; }
        #allClientsTable th:nth-child(10), #allClientsTable td:nth-child(10) { width: 8%; }
        #allClientsTable th:nth-child(11), #allClientsTable td:nth-child(11) { width: 5%; }
    </style>
</head>
<body>
    <?php include 'header.php'; ?>
    <div class="container mt-5">
        <h2 class="mb-4 text-center">All Active Clients</h2>
        <div class="card">
            <div class="card-body">
                <h5 class="card-title">Active Clients List</h5>
                <table class="table table-striped table-hover" id="allClientsTable">
                    <thead>
                        <tr>
                            <th>ID</th>
                            <th>Name</th>
                            <th>Email</th>
                            <th>Phone</th>
                            <th>Address</th>
                            <th>Details</th>
                            <th>Region</th>
                            <th>Status</th>
                            <th>Sales</th>
                            <th>Technician</th>
                            <th>Updated</th>
                        </tr>
                    </thead>
                    <tbody>
                        <?php while ($row = $allclients_result->fetch_assoc()) { ?>
                            <tr>
                                <td><?php echo htmlspecialchars($row['id']); ?></td>
                                <td><?php echo htmlspecialchars($row['name']); ?></td>
                                <td><?php echo htmlspecialchars($row['contact_email']); ?></td>
                                <td><?php echo htmlspecialchars($row['contact_phone']); ?></td>
                                <td><?php echo htmlspecialchars($row['address']); ?></td>
                                <td><?php echo htmlspecialchars($row['details']); ?></td>
                                <td><?php echo htmlspecialchars($row['region_name']); ?></td>
                                <td>
                                    <span class="badge <?php
                                        switch ($row['status']) {
                                            case 'Assignment Pending':
                                                echo 'badge-pending';
                                                break;
                                            case 'In Progress':
                                                echo 'badge-inprogress';
                                                break;
                                            case 'Completed':
                                                echo 'badge-completed';
                                                break;
                                            default:
                                                echo 'badge-secondary';
                                        }
                                    ?>">
                                        <?php echo htmlspecialchars($row['status']); ?>
                                    </span>
                                </td>
                                <td><?php echo htmlspecialchars($row['sales_name'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($row['tech_name'] ?? 'N/A'); ?></td>
                                <td><?php echo htmlspecialchars($row['updated_at']); ?></td>
                            </tr>
                        <?php } ?>
                    </tbody>
                </table>
            </div>
        </div>
    </div>
    <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.6/js/dataTables.bootstrap5.min.js"></script>
    <script>
        $(document).ready(function() {
            $('#allClientsTable').DataTable({
                "pageLength": 10,
                "order": [[10, "desc"]]
            });
        });
    </script>
</body>
</html>
<?php $conn->close(); ?>